#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>

// variables to hold statistics
int cache_hits, cache_misses, cache_reads, cache_writes = 0;

// variables to hold user input
int block_size, cache_size, cache_sets, assoc = 0;

// used to find least access block in LRU
long time = 0;

//replace policy
char replacement;

typedef struct  {
	int tag;
	int valid;

	// used in LRU algorithm implementation
	long time;	
} Line;

typedef struct {
	Line **base_line;
} Set;

Set **cache;

Set ** cache_create();

void cache_free();
void cache_fetch(Line **, int, int);
//void cache_write(struct Line **, int, int);
void cache_search(Line **, int, int);
void cache_stats();
void cache_error();


int main(int argc, char **argv) {
	// cache size argument
	cache_size = atoi(argv[1]);
	if (ceil(log2(cache_size)) != floor(log2(cache_size))) {
		cache_error();
	}	

	// block size argument
	// we have to parse it first as it's used in cache_sets calculations
	block_size = atoi(argv[4]);
	if (ceil(log2(block_size)) != floor(log2(block_size))) {
		cache_error();
	}

	// associativity argument
	char *assoc_arg = argv[2];
	if (strcmp(assoc_arg, "direct") == 0) {
		cache_sets = cache_size / block_size;
		assoc = 1;
	}
	else if (strcmp(assoc_arg, "assoc") == 0) {
		cache_sets = 1;
		assoc = cache_size / block_size;
	}
	else if (assoc_arg[5] == ':') {
		assoc = atoi(&assoc_arg[6]);
		// check assoc (n) is power of 2
		if (ceil(log2(assoc)) != floor(log2(assoc))) {
			cache_error();
		}
		cache_sets = cache_size / block_size / assoc;
	}

	// replace policy argument
	if (strcmp(argv[3], "lru") == 0) {
		replacement = 'l';
	}
	else if (strcmp(argv[3], "fifo") == 0) {
		replacement = 'f';
	}
	else {
		cache_error();
	}

	// trace file agrument
	FILE *cache_file = fopen(argv[5], "r");
	if (cache_file == NULL) {
		cache_error();
	}

	// block offset and set index size and mask
	int b, s, mask;
	b = log(block_size) / log(2);
	s = log(cache_sets) / log(2);
	
	mask = (1 << s) - 1;
	
	// variables to hold data read from tracefile
	char cache_mode; //R or W
	int address, tag, set_index;
	// only store PC jus to check #eof
	char pc[20];

	cache = cache_create();
	
	// read trace file line by line
	while (fscanf(cache_file, "%s %c %x\n", pc, &cache_mode, &address) != EOF) {
		//break the loop when file ends
		if (strcmp(pc, "#eof") == 0) {
			break;
		}

		// calculate the index bits and tag bits
		set_index = (((address >> b) & mask));
		tag = address >> (b + s);

		// extract the cache line to search
		Line **current_line = cache[set_index]->base_line;

		if (cache_mode == 'R') {
			cache_search(current_line, tag, address);
		}
		else if (cache_mode == 'W') {
			cache_writes++;
			//cache_write(current_line, tag, address);
			cache_search(current_line, tag, address);
		}
	}
	// free up the resources
	cache_free();
	fclose(cache_file);
	
	// print the cache statistics
	cache_stats();

	return 0;
}

// this function creates the initial empty cache
// and returns the cache struct
Set ** cache_create() {
	// use calloc to allocate memory
	Set **cache = calloc(cache_sets, sizeof(Set*));

	for (int i = 0; i < cache_sets; i++) {
		// allocate memory to each set in cahce
		//cache[i] = calloc(1, sizeof(Set));
		cache[i] = (Set *)malloc(1 * sizeof(Set));
		//cache[i]->base_line = calloc(assoc, sizeof(Line*));
		cache[i]->base_line = (Line **)malloc(assoc * sizeof(Line*));
		for (int j = 0; j < assoc; j++) {
			//cache[i]->base_line[j] = calloc(1, sizeof(Line));
			cache[i]->base_line[j] = (Line *)malloc(1 * sizeof(Line));
		}
	}
	return cache;
}

// this function search the cache
// default is FIFO algorithm implementation
void cache_search(Line **current_line, int tag, int address) {
	for (int i = 0; i < assoc; i++) {
		if (current_line[i]->valid == 0 || current_line[i]->tag != tag) {
			continue;
		}
		else if (current_line[i]->tag == tag) {
			// LRU algorithm implementation
			if (replacement == 'l') {
				current_line[i]->time = time++;
			}
			cache_hits++;
			return;
		}
	}

	// it's  a miss because we did not find in cache above
	cache_misses++;
	
	cache_fetch(current_line, tag, address);
}


// this function fetch the prefetch size amount of values
void cache_fetch(Line **current_line, int tag, int address) {

	int least_time = time;
	int x, least_index = 0;
	least_time = time;

	//create line to add to cache
	// add the line to cache
	Line *temp_line = calloc(1, sizeof(Line));
	temp_line->valid = 1;
	temp_line->tag = tag;
	temp_line->time = time++;

	// if prefetch already exists in cache 
	for (x = 0; x < assoc; x++) {
		if (current_line[x]->valid == 0) {
			// no data in cache
			// read from memory
			cache_reads++;

			current_line[x] = temp_line;
			break;
		}
		else {
			// LRU algorithm implementation
			if (current_line[x]->time < least_time) {
				least_time = current_line[x]->time;
				least_index = x;
			}
		}
	}
	
	// no free space available
	// so we evict least recently accessed block
	if (x == assoc) {
		// read from memory
		cache_reads++;

		// evict least recently accessed block
		free(current_line[least_index]);
		current_line[least_index] = temp_line;
	}
}


// This function free memory and clears the cache
void cache_free() {
	for (int i = 0; i < cache_sets; i++) {
		for (int j = 0; j < assoc; j++) {
			free(cache[i]->base_line[j]);
		}
		free(cache[i]->base_line);
		free(cache[i]);
	}	
	free(cache);
	
}

// This function prints the final output
void cache_stats() {
	printf("Memory reads: %d\n", cache_reads);
	printf("Memory writes: %d\n", cache_writes);
	printf("Cache hits: %d\n", cache_hits);
	printf("Cache misses: %d\n", cache_misses);
	return;
}

void cache_error(void) {
	printf("Error \n");
	printf("Standard Usage: ./cache <cache size> <associativity> <replace policy> <block size> <trace file> \n");
	exit(0);
}