/*JS*********************************************************************
*
*    Program : VAPRINTF
*    Language: ANSI-C
*    Author  : Joerg Schoen
*    Purpose : Similar to the 'sprintf' family functions, but allocates
*              string automatically with 'malloc' and returns it.
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id: vaprintf.c,v 1.2 1997/06/16 18:25:42 joerg Stab joerg $";
#endif

/*********     INCLUDES                                         *********/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>

#include <math.h>

#include <limits.h>

/*********     DEFINES                                          *********/
/*  START-DEFINITIONS */
#include <stdarg.h>  /*  For "va_list" definition */
/*  END-DEFINITIONS */

/*#define DEBUG*/

/*  Default upper length for integers and pointers. Should be
 *   enough even for 64 bit machines and if pointers are
 *   printed as bits.
 */
#define MAX_INTPOINT   (sizeof(long) * CHAR_BIT)

/*  Default upper length for floating point values (very crude)  */
#define MAX_FLOATVAL   (sizeof(double) * CHAR_BIT + 4)

#define Prototype extern
/*********     PROTOTYPES                                       *********/
Prototype char          *aprintf(const char *format, ...);
Prototype char          *vaprintf(const char *format,va_list ap);

Prototype long           vlprintf(const char *format,va_list ap);

/*********     GLOBAL VARIABLES                                 *********/

/*JS*********************************************************************
*   Similar to 'sprintf', but no buffer argument is supplied. Space will
*    be allocated automatically using 'malloc' and returned.
*************************************************************************/

char *aprintf(const char *format, ...)

/************************************************************************/
{
  va_list ap;
  char *s;

  va_start(ap,format);
  s = vaprintf(format,ap);
  va_end(ap);

  return(s);
}

/*JS*********************************************************************
*   Similar to 'vsprintf', but no buffer argument is supplied. Space will
*    be allocated automatically using 'malloc' and returned.
*************************************************************************/

char *vaprintf(const char *format,va_list ap)

/************************************************************************/
{
  char *string;
  long len,len2;

  /*  Estimate length  */
  len = vlprintf(format,ap);

  /*  Now allocate buffer and call vsprintf  */
  if((string = (char *)malloc(len + 1)) == NULL)
    return(NULL);

  if((len2 = vsprintf(string,format,ap)) < 0 || len2 > len) {
#ifdef DEBUG
    if(len2 >= 0) {
      fprintf(stderr,"\
FATAL: Wrong estimation in vaprintf (%d > %d) -- Break.\n",len2,len);
      abort();
    }
#endif
    free(string);
    return(NULL);
  }

  /*  Minimize memory usage (might be dropped to speed up)  */
  string = (char *)realloc(string,len2 + 1);

  return(string);
}

/*JS*********************************************************************
*   Estimates the needed length for the printf format with arguments.
*************************************************************************/

long vlprintf(const char *format,va_list ap)

/************************************************************************/
{
  char *string;
  long len,i;

  /*  Scan format and estimate needed length  */
  for(len = 0 ; *format ; ) {
    long minField,precField,flag;
    union {
      long          IVal;
      unsigned char UVal;
      char         *SVal;
      double        DVal;
      void         *PVal;
      short        *NSVal;
      int          *NIVal;
      long         *NLVal;
    } value;

    /*  Find next command  */
    string = strchr(format,'%');

    if(string == NULL) {
      /* ***  Nothing more to come  *** */
      len += strlen(format);
      break;
    } else if(string != format) {
      len += string - format;
      format = string;
      continue;
    }

    /* ***  Handle '%' formats  *** */
    format++;
    while(strchr("-+ 0#",*format)) format++; /*  Skip flags  */

    /*  Read 'min[.prec]' */
    if(isdigit(*(unsigned char *)format)) {
      /*  Note: strtoul cannot fail since we have already seen a digit  */
      minField = strtoul(format,(char **)&format,10);
    } else if(*format == '*') {
      minField = va_arg(ap,int);
      format++;
    } else
      minField = -1;

    if(*format != '.') {
      precField = -1;
    } else if(format++, isdigit(*(unsigned char *)format)) {
      /* **  Read prec  ** */
      precField = strtoul(format,(char **)&format,10);
    } else if(*format == '*') {
      precField = va_arg(ap,int);
      format++;
    } else
      precField = -1;

    /*  Skip size specifier  */
    if(*format == 'h') {
      flag = 0;
      format++;
    } else if(*format == 'l' || *format == 'L') {
      flag = 2;
      format++;
    } else
      flag = 1;

    /*  Read conversion specifier and estimate length  */
    switch(*format) {
    case 'd': case 'i': case 'o': /* **  integers  ** */
    case 'x': case 'X': case 'u':
      value.IVal = (flag == 0) ? va_arg(ap,short) :
	((flag == 1) ? va_arg(ap,int) : va_arg(ap,long));
      i = MAX_INTPOINT;
      break;
    case 'c': /* **  character  ** */
      value.UVal = (unsigned char)va_arg(ap,int);
      i = 1;
      break;
    case 's': /* **  string  ** */
      value.SVal = va_arg(ap,char *);

      /*  We do *not* use strlen in case precision is given, since
       *   string might not be properly terminated.
       */
      i = (precField >= 0) ? precField : strlen(value.SVal);
      break;
    case 'f':
      value.DVal = va_arg(ap,double);

      /*  That's tricky and might result in *very* long printouts  */
      if(value.DVal != 0.0)
        i = (int)log10(fabs(value.DVal)) +
	  (precField < 0 ? 10: (precField + 4));
      else
        i = MAX_FLOATVAL;
      break;
    case 'e': case 'E': case 'g': case 'G': /* **  double  ** */
      value.DVal = va_arg(ap,double);
      i = MAX_FLOATVAL;
      if(precField >= 0) precField += (MAX_FLOATVAL - 6);

      break;
    case 'p': /* **  pointer  ** */
      value.PVal = va_arg(ap,void *);
      i = MAX_INTPOINT;
      break;
    case 'n': /* **  store number of chars  ** */
      if(flag == 0)
	value.NSVal = va_arg(ap,short *);
      else if(flag == 1)
	value.NIVal = va_arg(ap,int *);
      else
	value.NLVal = va_arg(ap,long *);
      i = 0; /*  ignore  */
      break;
    case '%':
      i = 1;
      break;
    default:
      /*  Undefined, copy symbol  */
#ifdef DEBUG
      fprintf(stderr,"WARNING: Unknown format specifier '%c'!\n",*format);
#endif
      i = 1;
      break;
    }
    if(*format) format++;

    if(i < minField) i = minField;
    if(i < precField) i = precField;

    len += i;
  }

  return(len);
}

#ifdef TEST
int main(int argc,char *argv[])
{
  char *string;
  int i1,i2;
  double d1,d2;

  i1 = 1342423424;
  i2 = 17;
  d1 = 1.9876532424242e-50;
  d2 = 1.9876532424242E60;

  string = aprintf("hello %d %020i--\n\
>>%50o\n\
--%#x%*.*d--\n\
%s %f\n\
!%10.10g%%\n",
		    i1,i2,i1,i2,30,5,i2,"WATISDAT",
		    d2,d1);

  if(string == NULL) {
    perror("testvaprintf");
    exit(30);
  }

  fputs(string,stdout);

  return(0);
}
#endif
