/*JS*********************************************************************
*
*    Program : SPLINE
*    Language: ANSI-C
*    Author  : Joerg Schoen
*    Purpose : Cubic spline interpolation.
*
*      This material is largely taken from "Numerical Recipes in C, second
*       edition" and altered to fit in my programming techniques.
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id: spline.c,v 1.2 1996/07/31 18:57:23 joerg Stab joerg $";
#endif

/*********     INCLUDES                                         *********/
#include <jsalloca.h>

#include <stdio.h>
#include <stdlib.h>

#include <math.h>

#include <jsconfig.h>

#include <jssubs.h>

/*********     DEFINES                                          *********/
/* ERROR-DEFINITIONS from splint label _ERR_SPLINE ord 11
   X values are identical
*/

#define Prototype extern
/*********     PROTOTYPES                                       *********/
Prototype double        *spline(const double xa[],const double ya[],int n,
				double yp1,double ypn,double *y2);
Prototype double         splint(double x,const double xa[],const double ya[],
				const double y2[],int n,int lower);

Prototype double        *spline2d(const double x1a[],const double x2a[],
				  const double ya[],int n1,int n2,double *y2);
Prototype double         splint2d(double x1,double x2,const double x1a[],
				  const double x2a[],const double ya[],
				  int n1,int n2,double *y2);

Prototype double        *splineData(const double *x,const double *y,
				    int n,int nNew,double *yNew);

/*JS*********************************************************************
*   Given arrays xa[0 .. n-1] and ya[0 .. n-1] containing a tabulated
*    function, i.e., ya[i] = f(xa[i]), with xa[0] < xa[1] < .. < xa[n-1],
*    and given values yp1 and ypn for the first derivative of the
*    interpolating function at points 0 and n-1, respectively, this
*    routine returns an array that contains the second derivatives of the
*    interpolating function at the tabulated points x[i]. If yp1 and/or
*    ypn are equal to HUGE_VAL, the routine is signaled to set the
*    corresponding boundary condition for a natural spline, with zero
*    second derivative on that boundary. If y2 is set to NULL, memory is
*    automatically allocated.
*    If n is negative, the x values are equidistant in the interval
*    [xa[0],xa[1]].
*************************************************************************/

double *spline(const double xa[],const double ya[],int n,
	       double yp1,double ypn,double *y2)

/************************************************************************/
{
  double *u;
  double qn,un,dx;
  int i,isign;

  if(n > 0) {
    isign = 1;
  } else {
    isign = -1;
    n = -n;
    dx = (xa[1] - xa[0]) / (n - 1);
  }

  if(y2 == NULL && (y2 = (double *)malloc(n * sizeof(*y2))) == NULL)
    return(NULL);

#ifndef CONFIG_USE_ALLOCA
  if((u = (double *)malloc((n - 1) * sizeof(*u))) == NULL) return(NULL);
#else
  if((u = (double *)alloca((n - 1) * sizeof(*u))) == NULL) return(NULL);
#endif

  if(yp1 == HUGE_VAL) {
    /*  Lower boundary condition set to be "natural"  */
    y2[0] = u[0] = 0.0;
  } else {
    /*  Lower boundary condition is set to specified first derivative  */
    y2[0] = -0.5;
    if(isign > 0)
      u[0] = (3.0 / (xa[1] - xa[0])) * ((ya[1] - ya[0]) / (xa[1] - xa[0]) -
					yp1);
    else
      u[0] = (3.0 / dx) * ((ya[1] - ya[0]) / dx - yp1);
  }

  /*  This is the decomposition loop of the tridiagonal algorithm.
   *   y2 and u are used for temporary storage of the decomposed factors.
   */
  for(i = 1 ; i < (n-1) ; i++) {
    double sig,p;

    sig = (isign > 0) ? ((xa[i] - xa[i-1]) / (xa[i+1] - xa[i-1])) : 0.5;
    p = sig * y2[i-1] + 2.0;
    y2[i] = (sig - 1.0) / p;

    if(isign > 0) {
      u[i] = (ya[i+1] - ya[i]) / (xa[i+1] - xa[i]) -
	(ya[i] - ya[i-1]) / (xa[i] - xa[i-1]);
      u[i] = (6.0 * u[i] / (xa[i+1] - xa[i-1]) - sig * u[i-1]) / p;
    } else {
      u[i] = (ya[i+1] - 2.0 * ya[i] + ya[i - 1]) / dx;
      u[i] = (3.0 * u[i] / dx - sig * u[i-1]) / p;
    }
  }

  if(ypn == HUGE_VAL) {
    /*  Upper boundary condition set to be "natural"  */
    qn = un = 0.0;
  } else {
    /*  Upper boundary condition is set to specified first derivative  */
    qn = 0.5;
    if(isign > 0)
      un = (3.0 / (xa[n-1] - xa[n-2])) * (ypn - (ya[n-1] - ya[n-2]) /
					  (xa[n-1] - xa[n-2]));
    else
      un = (3.0 / dx) * (ypn - (ya[n-1] - ya[n-2]) / dx);
  }

  y2[n-1] = (un - qn * u[n-2]) / (qn * y2[n-2] + 1.0);

  /*  This is the backsubstitution loop of the tridiagonal algorithm  */
  for(i = n-2 ; i >= 0 ; i--)
    y2[i] = y2[i] * y2[i+1] + u[i];

#ifndef CONFIG_USE_ALLOCA
  free(u);
#endif

  return(y2);
}

/*JS*********************************************************************
*   Given the arrays xa[0 .. n-1] and ya[0 .. n-1], which tabulate a
*    function (with the xa's in order), and given the array y2[0 .. n-1]
*    which is the output from spline above, and given a value of x, this
*    routine returns a cubic-spline interpolated value y.
*    The parameter lower specifies the interval where x is contained,
*    i.e. x in [xa[lower],xa[lower+1]]. If lower is negative, it is
*    obtained via bisection. If x lies not within the interval, the value
*    is extrapolated using a quadratic polynomial, so that the second
*    derivative is constant (taken from R. Schork).
*    If n is negative, the x values are equidistant in the interval
*    [xa[0],xa[1]].
*************************************************************************/

double splint(double x,const double xa[],const double ya[],const double y2[],
	      int n,int lower)

/************************************************************************/
{
  int upper,isign,outside;
  double a,b,h;

  a = xa[0];  b = xa[n < 0 ? 1 : (n - 1)];

  if(n > 0) {
    isign = 1;
  } else {
    isign = -1;
    n = -n;
  }

  if(x < a && x < b) {
    if(a < b) {
      lower   = 0;
      outside = -1;
    } else {
      lower   = n - 2;
      outside = 1;
    }
  } else if(x > a && x > b) {
    if(a < b) {
      lower   = n - 2;
      outside = 1;
    } else {
      lower   = 0;
      outside = -1;
    }
  } else
    outside = 0;

  if(isign > 0) {
    if(lower < 0) {
      /*  Find position via bisection  */
      lower = 0;
      upper = n - 1;
      while((upper - lower) > 1) {
	int m = (lower + upper) >> 1;

	if(xa[m] > x) upper = m;
	else lower = m;
      }
    } else
      upper = lower + 1;

    if((h = xa[upper] - xa[lower]) == 0.0) goto error;

    a = (xa[upper] - x) / h;
    b = (x - xa[lower]) / h;
  } else {
    if((h = (xa[1] - xa[0]) / (n - 1)) == 0.0) goto error;

    if(lower < 0) {
      lower = (int) ((x - xa[0]) / h);

      if(lower < 0)
	lower = 0;
      else if(lower > (n - 2))
	lower = n - 2;

      upper = lower + 1;
    } else
      upper = lower + 1;

    a = (xa[0] + upper * h - x) / h;
    b = (x - xa[0] - lower * h) / h;
  }

  /*  Cubic spline polynomial is now evaluated  */
  if(outside == 0)
    return(a * ya[lower] + b * ya[upper] +
	   (1.0 / 6.0) * h * h *
	   ((a * a * a - a) * y2[lower] + (b * b * b - b) * y2[upper]));

  return(outside == -1 ?
	 (a * ya[lower] + b * ya[upper] -
	  (1.0 / 6.0) * h * h * b * (y2[upper] + 2 * y2[lower]) +
	  0.5 * h * h * b * b * y2[lower]) :
	 (a * ya[lower] + b * ya[upper] -
	  (1.0 / 6.0) * h * h * a * (y2[lower] + 2 * y2[upper]) +
	  0.5 * h * h * a * a * y2[upper]));
error:
  JSErrNo = _ERR_SPLINE + 0;
  return(HUGE_VAL);
}

/*JS*********************************************************************
*   Given an m by n tabulated function ya[0 .. n1*n2-1], where the first
*    index corresponding to x1a has stride 1 and tabulated independent
*    variables x2a[0 .. n2], this routine constructs one-dimensional
*    natural cubic-splines of the rows of ya and returns the second
*    derivatives in the array y2a[0 .. n1*n2-1].
*************************************************************************/

double *spline2d(const double x1a[],const double x2a[],const double ya[],
		 int n1,int n2,double *y2)

/************************************************************************/
{
  int j,save;

  /*  Accept negative n's as well  */
  save = n1;
  if(n1 < 0) n1 = -n1;
  if(n2 < 0) n2 = -n2;

  if(y2 == NULL && (y2 = (double *)malloc(n1 * n2 * sizeof(*y2))) == NULL)
    return(NULL);

  for(j = 0 ; j < n2 ; j++)
    if(spline(x1a,&ya[n1 * j],save,HUGE_VAL,HUGE_VAL,&y2[n1 * j]) == NULL) {
      free(y2);
      return(NULL);
    }

  return(y2);
}

/*JS*********************************************************************
*   Given x1a, x2a, ya, n1, n2 as described in spline2d and y2a as
*    produced by that routine; and given a desired interpolating point
*    x1,x2; this routine returns an interpolated function value y by
*    bicubic spline interpolation.
*************************************************************************/

double splint2d(double x1,double x2,const double x1a[],const double x2a[],
		const double ya[],int n1,int n2,double *y2)

/************************************************************************/
{
  int j,save1,save2;
  double *yTmp,*y2Tmp,y;

  /*  Accept negative n's as well  */
  save1 = n1;
  if(n1 < 0) n1 = -n1;
  save2 = n2;
  if(n2 < 0) n2 = -n2;

  if((yTmp = (double *)malloc(n2 * sizeof(*yTmp))) == NULL) goto error;

  /*  Perform n2 evaluations of the row splines constructed by spline2d,
   *   using the one-dimensional spline evaluator 'splint'.
   */
  for(j = 0 ; j < n2 ; j++)
    yTmp[j] = splint(x1,x1a,&ya[n1 * j],&y2[n1 * j],save1,-1);

  /*  Construct the one dimensional column spline and evaluate it  */
  if((y2Tmp = spline(x2a,yTmp,save2,HUGE_VAL,HUGE_VAL,NULL)) == NULL)
    goto error;

  y = splint(x2,x2a,yTmp,y2Tmp,save2,-1);

  free(yTmp);
  free(y2Tmp);

  return(y);
error:
  return(HUGE_VAL);
}

/*JS*********************************************************************
*   Simple application: Interpolate data
*      y[i] = f(x[i]), i=0, .. ,n-1
*    to yield nNew values in yNew that cover same area on equidistant grid.
*    x may be NULL meaning original data are equidistant as well. If yNew
*    is NULL, memory is automatically allocated.
*************************************************************************/

double *splineData(const double *x,const double *y,int n,int nNew,
		   double *yNew)

/************************************************************************/
{
  double *y2,h,x0,xa[2];
  int i;

  if(x) {
    x0 = x[0];
    h = (x[n-1] - x0) / (nNew - 1);

    if((y2 = spline(x,y,n,HUGE_VAL,HUGE_VAL,NULL)) == NULL) goto error;
  } else {
    x0 = xa[0] = 0.0;
    xa[1] = (double)(nNew - 1);
    h = 1.0;

    if((y2 = spline(xa,y,-n,HUGE_VAL,HUGE_VAL,NULL)) == NULL) goto error;
  }

  if(yNew == NULL && (yNew = (double *)malloc(nNew * sizeof(*yNew))) == NULL)
    return(NULL);

  for(i = 0 ; i < nNew ; i++)
    yNew[i] = splint(x0 + i * h,x ? x : xa,y,y2,x ? n : -n,-1);

  free(y2);

  return(yNew);
error:
  if(y2) free(y2);
  return(NULL);
}

#ifdef TEST
/* **  Plot a function with a few values and then spline it  ** */
#include <jssubs.h>

/*#define GEN_XDATA*/

#define X0   0.0
#define X1  10.0

#define N   10
#define N2  1000
#define F(x)   sin(x)

#ifdef GEN_XDATA
double X[N];
#else
double X[2];
#endif

double Y[N];
double X2[N2],Y2[N2],Yo[N2];

int main(int argc,char *argv[])
{
  int i;
  double h,*yder;
  FPFile *fp;

#ifdef GEN_XDATA
  h = (X1 - X0) / (N - 1);
  for(i = 0 ; i < N ; i++) {
    X[i] = X0 + h * i;
    Y[i] = F(X[i]);
  }

  if((yder = spline(X,Y,N,HUGE_VAL,HUGE_VAL,NULL)) == NULL) goto error;
#else
  X[0] = X0;
  X[1] = X1;
  h = (X1 - X0) / (N - 1);
  for(i = 0 ; i < N ; i++) Y[i] = F(X[0] + i * h);

  if((yder = spline(X,Y,-N,HUGE_VAL,HUGE_VAL,NULL)) == NULL) goto error;
#endif

  /*for(i = 0 ; i < N ; i++) printf("y2[%d] = %g\n",i,yder[i]);*/

  h = (X1 - X0) / (N2 - 1);
  for(i = 0 ; i < N2 ; i++) {
    X2[i] = X0 + h * i;
    Y2[i] = splint(X2[i],X,Y,yder,
#ifdef GEN_XDATA
		   N
#else
		   -N
#endif
		   ,-1);
    Yo[i] = F(X2[i]);
  }

  free(yder);

  if((fp = fplotOpen("spli.out",FPLOT_WRITE)) == NULL ||
     fplotStart(fp) ||
     fplotText(fp,"Original data (%d values)",N) ||
     fplotVector(fp,0,
#ifdef GEN_XDATA
		 X
#else
		 NULL
#endif
		 ,Y,N,1,X0,X1,1.0,1.0) ||
     fplotEnd(fp) ||
     fplotStart(fp) ||
     fplotText(fp,"Splined data versus original ones (%d values)",N2) ||
     fplotVector(fp,0,X2,Y2,N2,1,X0,X1,1.0,1.0) ||
     fplotVector(fp,0,X2,Yo,N2,1,X0,X1,1.0,1.0) ||
     fplotEnd(fp) ||
     fplotClose(fp)) goto error;

  return(0);
error:
  fprintf(stderr,"ERROR\n");
  return(30);
}
#endif
