/*JS*********************************************************************
*
*    Program : RIDDERIV
*    Language: ANSI-C
*    Author  : Joerg Schoen
*    Purpose : Calculate the derivative of a function using Ridder's
*              method.
*
*      This material is largely taken from "Numerical Recipes in C, second
*       edition" and altered to fit in my programming techniques.
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id: ridderiv.c,v 1.1 1996/07/17 12:01:07 joerg Stab joerg $";
#endif

/*********     INCLUDES                                         *********/
#include <math.h>

/*********     DEFINES                                          *********/
#define CON    1.4  /*  stepsize is decreased by CON at each iteration  */
#define CON2   (CON * CON)
#define BIG    1.0e200
#define NTAB   10   /*  sets maximum size of tableau  */
#define SAFE   2.0  /*  return when error is SAFE worse than the best so far */

#define Prototype extern
/*********     PROTOTYPES                                       *********/
Prototype double         ridDeriv(double x,double h,double (*func)(double),
				  double *pErr);

/*********     GLOBAL VARIABLES                                 *********/

/*JS*********************************************************************
*   Returns the derivative of a function func at at point x by Ridders'
*    method of polynomial extrapolation. The value h is input as an
*    estimated initial stepsize; it need not be small, but rather should
*    be an increment in x over which func changes substantially. An
*    estimate of the error in the derivative is returned as *pErr.
*************************************************************************/

double ridDeriv(double x,double h,double (*func)(double),double *pErr)

/************************************************************************/
{
  int i;
  double hh,ans,err;
  double a[NTAB][NTAB];

  ans = HUGE_VAL;

  if((hh = h) == 0.0) {
    /*  Wrong input value  */
    return(ans);
  }

  a[0][0] = ((*func)(x + hh) - (*func)(x - hh)) / (2.0 * hh);
  err = BIG;
  for(i = 1 ; i < NTAB ; i++) {
    double fac;
    int j;

    /*  Successive columns in the Neville tableau will go to
     *   smaller stepsizes and higher orders of extrapolation.
     */
    hh /= CON;

    /*  Try new, smaller stepsize  */
    a[0][i] = ((*func)(x + hh) - (*func)(x - hh)) / (2.0 * hh);
    fac = CON2;
    for(j = 1 ; j <= i ; j++) {
      double errt,temp;

      /*  Compute extrapolations of various orders,
       *   requiring no new function evaluations.
       */
      a[j][i] = (a[j-1][i] * fac - a[j-1][i-1]) / (fac - 1.0);
      fac *= CON2;
      temp = fabs(a[j][i] - a[j-1][i]);
      errt = fabs(a[j][i] - a[j-1][i-1]);
      if(temp > errt) errt = temp;

      /*  The error stategy is to compare each new extrapolation
       *   to one order lower, both at the present stepsize and
       *    the previuos one.
       */
      if(errt <= err) {
	/*  If error is decreased, save the improved answer.  */
	err = errt;
	ans = a[j][i];
      }
    }

    /*  If higher order is worse by a significant
     *   factor SAFE, then quit early.
     */
    if(fabs(a[i][i] - a[i-1][i-1]) >= SAFE * err) break;
  }

  if(pErr) *pErr = err;

  return(ans);
}
