/*JS*********************************************************************
*
*    Program : RFFT
*    Language: ANSI-C
*    Author  : Joerg Schoen
*    Purpose : Real Fast Fourier Tranformation.
*
*      This material is taken from "Numerical Recipes in C" and slightly
*       altered (The original code was a little bit cryptic). The former
*       name of the routine was realft.
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id: rfft.c,v 1.1 1994/12/19 09:14:14 joerg Stab joerg $";
#endif

/*********     INCLUDES 					*********/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <jssubs.h>

/*********     DEFINES						*********/

#define Prototype extern
/*********     PROTOTYPES					*********/
Prototype int            rfft(double data[],unsigned long n,int isign);

/*JS*********************************************************************
*   Double precision real Fast Fourier transform for n a power of 2.
*    Calculates the Fourier transform of a set of n real- valued data
*    points. Replaces this data (which is stored in array data[0..n-1])
*    by the positive frequency half of its complex Fourier transform.
*    The real- valued first and last components of the complex transform
*    are returned as elements data[0] and data[1] respectively. n must be
*    a power of 2. This routine also calculates the inverse transform of
*    a complex data array if it is the transform of real data. (Result in
*    this case must be multiplied by 2/n).
*************************************************************************/

int rfft(double data[],unsigned long n,int isign)

/************************************************************************/
{
  double c2,dtemp;
  dcmplx w,wpm1;
  int i,nhalf;

  /*  We need n >> 1 in the following  */
  nhalf = n >> 1;

  /*  Initialize the forward recurrence  */
  dtemp = PI / (double) nhalf;

  if(isign == 1) {
    c2 = -0.5;

    /*  The forward transform is here  */
    if(cfft((dcmplx *)data,nhalf,1))
      return(1);
  } else {
    c2 =  0.5;
    dtemp = -dtemp;
  }

  /*  Set up for recurrence and set initial value  */
  {
    double temp;

    temp = sin(0.5 * dtemp);
#ifndef __cplusplus
    /*  include minus 1  */
    SETC(wpm1,-2.0 * temp * temp,sin(dtemp));
    SETC(w,1.0 + RE(wpm1),IM(wpm1));
#else
    wpm1 = dcmplx(-2.0 * temp * temp,sin(dtemp));
    w = 1.0 + wpm1;
#endif
  }

  /*  Case i1 = 0 done separately below  */
  for(i = 2 ; i < nhalf ; i += 2) {
    int iinv;
    dcmplx h1,h2;

    iinv = n - i;

    /*  The two separate transforms are separated out of data.  */
    SETC(h1,0.5 * (data[i] + data[iinv]),0.5 * (data[i + 1] - data[iinv + 1]));

    SETC(h2,-c2 * (data[i + 1] + data[iinv + 1]),c2 * (data[i] - data[iinv]));

    /*  Here they are recombined to form the true  */
    /*   transform of the original real data.      */
#ifndef __cplusplus
    data[i]        =  RE(h1) + RE(w) * RE(h2) - IM(w) * IM(h2);
    data[i + 1]    =  IM(h1) + RE(w) * IM(h2) + IM(w) * RE(h2);
    data[iinv]     =  RE(h1) - RE(w) * RE(h2) + IM(w) * IM(h2);
    data[iinv + 1] = -IM(h1) + RE(w) * IM(h2) + IM(w) * RE(h2);

    /*  The recurrence  */
    dtemp = RE(w);
    RE(w) += RE(wpm1) * dtemp - IM(wpm1) * IM(w);
    IM(w) += RE(wpm1) * IM(w) + IM(wpm1) * dtemp;
#else
    {
      dcmplx temp = h1 + w * h2;
      data[i]     = temp.real();
      data[i + 1] = temp.imag();

      temp = conj(h1 - w * h2);
      data[iinv]     = temp.real();
      data[iinv + 1] = temp.imag();
    }

    w += wpm1 * w;
#endif
  }

  if(isign == 1) {
    /*  Squeeze the first and last data together to  */
    /*   get them all within the original array.     */
    dtemp = data[0];
    data[0] += data[1];
    data[1] = dtemp - data[1];
  } else {
    dtemp = data[0];
    data[0] = 0.5 * (data[0] + data[1]);
    data[1] = 0.5 * (dtemp - data[1]);

    /*  This is the inverse transform for the case isign = -1  */
    if(cfft((dcmplx *)data,nhalf,-1))
      return(1);
  }

  return(0);
}
