/*JS*********************************************************************
*
*    Program : PROGOPTS
*    Language: ANSI-C
*    Author  : Joerg Schoen
*    Purpose : Parse options for a program from strings.
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id: progopts.c,v 1.2 1997/08/02 21:38:13 joerg Stab joerg $";
#endif

/*********     INCLUDES                                         *********/
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <limits.h>

#include <jssubs.h>

/*********     DEFINES                                          *********/
#ifndef PROPT_TYPE_MASK
/*  START-DEFINITIONS  */
#include <stdio.h>   /*	 For "FILE" definition    */

/*  For PO_Flags field  */
#define PROPT_TYPE_MASK  (15<<24)

/*  Flags set by the routine  */
#define PROPT_FLAG_ISSET  (1<<28)
#define PROPT_FLAG_HIDE   (1<<29)

/*  Currently implemented types  */
#define PROPT_IVALUE      (1<<24)
#define PROPT_DVALUE      (2<<24)
#define PROPT_STRING      (3<<24)
#define PROPT_BOOL        (4<<24)
#define PROPT_ISET        (5<<24)
#define PROPT_SUBOPT     (10<<24)
#define PROPT_ABBREV     (11<<24)

#define PROPT_LENGTH     ((1<<24)-1)

typedef struct {
  const char   *PO_Name;
  unsigned long PO_Flags;
  void         *PO_Ptr;
} ProgOpt;
/*  END-DEFINITIONS  */
#endif

/* ERROR-DEFINITIONS from parseProgOpts label _ERR_PPROGOPTS ord 14
   Syntax error
   Unknown (sub-)option name
   Garbage in abbreviation
*/

#define Prototype extern
/*********     PROTOTYPES                                       *********/
Prototype int            parseProgOpts(const char *input,ProgOpt opts[]);
Prototype void           showProgOpts(FILE *fp,const ProgOpt opts[],
				      const char *help[],const char *delimit);

/*********     GLOBAL VARIABLES                                 *********/
static int Shift;

/*JS*********************************************************************
*   Parse options from input. Allowed options are found in the table opts.
*    Supported types are integers, doubles, strings, boolean options,
*    options with suboptions and abbreviations of often used combinations.
*    Multiple options may be separated with ','; to end suboption parsing,
*    use ';'.
*    In the case of integers and doubles, the lower 24 bits tell the
*    number of values to read at most, 0 means one value. If more than
*    one value is to be read, the first entry is used to store the number
*    of read values.
*    For strings, these bits tell the maximal length the character pointer
*    holds space. For boolean options, the bits hold the bit to set/clear.
*    > 32 means access following ulongs (for sizeof(unsigned long)==4).
*    For suboptions, the bits are a shift value to add for boolean
*    suboptions. Abbreviations just contains strings to parse.
*************************************************************************/

int parseProgOpts(const char *input,ProgOpt opts[])

/************************************************************************/
{
  const char *string;

  string = input;

  for(;;) {
    long type,len;
    int i,invFlag;

    while(isspace(*(unsigned char *)string)) string++;

    if((len = strtoid(string)) == 0) break;

    invFlag = FALSE; /*  for boolean options  */

    for(i = 0 ; opts[i].PO_Name ; i++)
      if(strncmp(opts[i].PO_Name,string,len) == 0 &&
	 opts[i].PO_Name[len] == '\0')
	break;

    if(opts[i].PO_Name == NULL && string[0] == 'n' && string[1] == 'o') {
      /*  Remove "no"  */
      string += 2;
      len -= 2;
      invFlag = TRUE;
      for(i = 0 ; opts[i].PO_Name ; i++)
	if(strncmp(opts[i].PO_Name,string,len) == 0 &&
	   opts[i].PO_Name[len] == '\0' &&
	   (opts[i].PO_Flags & PROPT_TYPE_MASK) == PROPT_BOOL)
	  break;
    }

    if(opts[i].PO_Name == NULL) {
      JSErrNo = _ERR_PPROGOPTS + 1;
      goto error;
    }

    string += len;

    /*  Mark option was seen  */
    opts[i].PO_Flags |= PROPT_FLAG_ISSET;

    type = (opts[i].PO_Flags & PROPT_TYPE_MASK);
    len = opts[i].PO_Flags & PROPT_LENGTH;

    if(!(type == PROPT_BOOL || type == PROPT_ABBREV)) {
      if(*string == '=')
	string++;
      else if(*string != ',' && *string != '\0') {
	JSErrNo = _ERR_PPROGOPTS + 0;
	goto error;
      }
    }

    switch(type) {
    case PROPT_IVALUE: case PROPT_DVALUE:
      {
	long j;

	j = (len == 0) ? 0 : 1;
	for( ; j <= len && *input ; j++) {
	  const char *string2;
	  long ival;
	  double dval;

	  if(type == PROPT_IVALUE) {
	    ival = strtol2(string,(char **)&string2,0);
	  } else {
	    dval = strtod(string,(char **)&string2);
	  }

	  if(string == string2) break; /*  stop processing  */

	  if(type == PROPT_IVALUE) {
	    ((int *)(opts[i].PO_Ptr))[j] = ival;
	  } else {
	    ((double *)(opts[i].PO_Ptr))[j] = dval;
	  }

	  string = string2;

	  while(isspace(*(unsigned char *)string)) string++;

	  /*  In arguments to options allow only ','  */
	  if(*string == ',') string++;
	}

	if(len > 0) {
	  /*  Save number of read values  */
	  if(type == PROPT_IVALUE) {
	    ((int *)(opts[i].PO_Ptr))[0] = j - 1;
	  } else {
	    ((double *)(opts[i].PO_Ptr))[0] = (double) (j - 1);
	  }
	}
      }
      break;
    case PROPT_ISET:
      *(int *)opts[i].PO_Ptr = len;
      break;
    case PROPT_STRING:
      {
	const char *string2;

	/*  Look for first non-escaped ',' or ';'  */
	for(string2 = string ; *string2 ; string2++) {
	  if(*string2 == '\\' && string2[1])
	    string2++;
	  else if(*string2 == ',' || *string2 == ';')
	    break;
	}

	len = MIN(len,string2 - string);

	memcpy((char *)(opts[i].PO_Ptr),string,len);
	((char *)(opts[i].PO_Ptr))[len] = '\0';

	string = string2;
      }
      break;
    case PROPT_BOOL:
      {
	int offset,bitNr;

	offset = (Shift + len) / (sizeof(unsigned long) * CHAR_BIT);
	bitNr  = (Shift + len) % (sizeof(unsigned long) * CHAR_BIT);

	if(!invFlag) /*  set bit  */
	  ((unsigned long *)(opts[i].PO_Ptr))[offset] |=  (1UL << bitNr);
	else /*  clear bit  */
	  ((unsigned long *)(opts[i].PO_Ptr))[offset] &=  ~(1UL << bitNr);
      }
      break;
    case PROPT_SUBOPT:
      {
	int ret;

	Shift += len;
	ret = parseProgOpts(string,(ProgOpt *)opts[i].PO_Ptr);
	Shift -= len;

	if(ret < 0) goto error;
	string += ret;

	/*  Allow ',' or ';' as delimiter  */
	if(*string == ',' || *string == ';')
	  string++;
      }
      break;
    case PROPT_ABBREV:
      {
	const char *string2;
	int ret;

	string2 = (const char *)opts[i].PO_Ptr;
	if((ret = parseProgOpts(string2,opts)) < 0) goto error;
	if(string2[ret]) {
	  JSErrNo = _ERR_PPROGOPTS + 2;
	  goto error;
	}
      }
      break;
    }

    if(*string == ',') string++;
  }

  /*  Return number of successfully read characters  */
  return(string - input);
error:
  return(-1);
}

/*JS*********************************************************************
*   To let a program show all options that are possible with optional help
*    texts.
*************************************************************************/

void showProgOpts(FILE *fp,const ProgOpt opts[],const char *help[],
		  const char *delimit)

/************************************************************************/
{
  int i;

  for(i = 0 ; opts[i].PO_Name ; i++) {
    long type,len;

    if(opts[i].PO_Flags & PROPT_FLAG_HIDE) continue;

    type = (opts[i].PO_Flags & PROPT_TYPE_MASK);
    len = opts[i].PO_Flags & PROPT_LENGTH;

    if(help && help[i]) fprintf(fp,"# %s%s",help[i],delimit);

    switch(type) {
    case PROPT_IVALUE: case PROPT_DVALUE:
      fprintf(fp,"%s=",opts[i].PO_Name);
      {
	int j;

	if(len == 0)
	  j = 0;
	else {
	  /*  First entry contains default number to set  */
	  j = 1;
	  if(type == PROPT_IVALUE)
	    len = ((int *)(opts[i].PO_Ptr))[0];
	  else
	    len = (int) ((double *)(opts[i].PO_Ptr))[0];
	}

	for( ; j <= len ; j++) {
	  if(type == PROPT_IVALUE)
	    fprintf(fp,"%d",((int *)(opts[i].PO_Ptr))[j]);
	  else
	    fprintf(fp,"%g",((double *)(opts[i].PO_Ptr))[j]);
	  if(j != len) putc(',',fp);
	}
      }
      break;
    case PROPT_STRING:
      fprintf(fp,"%s=%s",opts[i].PO_Name,(char *)opts[i].PO_Ptr);
      break;
    case PROPT_BOOL:
      if(!(((unsigned long *)(opts[i].PO_Ptr))
	   [(Shift + len) / (sizeof(unsigned long) * CHAR_BIT)] &
	   (1UL << ((Shift + len) % (sizeof(unsigned long) * CHAR_BIT)))))
	fputs("no",fp);
      fputs(opts[i].PO_Name,fp);
      break;
    case PROPT_SUBOPT:
      fprintf(fp,"%s=",opts[i].PO_Name);

      Shift += len;
      showProgOpts(fp,(ProgOpt *)opts[i].PO_Ptr,NULL,",");
      Shift -= len;
      break;
    case PROPT_ABBREV:
      fprintf(fp,"(%s=\"%s\")",opts[i].PO_Name,(char *)opts[i].PO_Ptr);
      break;
    }

    if(opts[i+1].PO_Name) fputs(delimit,fp);
  }
}

#ifdef TEST
/* ***  Test all kind of options  *** */
static int Flag;
static int IntValue,IValue,IValues[11];
static double DoubleValue,DValue,DValues[11];
static char Name[101];

static ProgOpt SubOpts[] = {
  { "int",      PROPT_IVALUE,       &IntValue    },
  { "double",   PROPT_DVALUE,       &DoubleValue },
  { "verbose",  PROPT_BOOL | 5,     &Flag        },
  { NULL, 0, NULL }
};

static ProgOpt Options[] = {
  { "ival",     PROPT_IVALUE,       &IValue   },
  { "ivals",    PROPT_IVALUE | 10,   IValues  },
  { "dval",     PROPT_DVALUE,       &DValue   },
  { "dvals",    PROPT_DVALUE | 10,   DValues  },
  { "name",     PROPT_STRING | 100,  Name     },
  { "verbose",  PROPT_BOOL | 5,     &Flag     },
  { "subopt",   PROPT_SUBOPT | 10,   SubOpts  },

  { "shortcut", PROPT_ABBREV, "ival=4711,ivals=18,20,23,dval=3.145,noverbose"},
  { NULL, 0, NULL }
};

static const char *Help[] = {
  "Integer",
  "Up to 10 integers",
  "Double",
  "Up to 10 doubles",
  "String",
  "Flag",
  "Suboption",
  "Shortcut"
};

int main(int argc,char *argv[])
{
  if(argc == 1) {
    printf("Usage: testprogopts <opts>\n");
  } else {
    while(--argc > 0) {
      int len;
      len = parseProgOpts(*++argv,Options);
      if(len < 0) goto error;
      else if((*argv)[len]) {
	fprintf(stderr,"ERROR: Trailing garbage ignored: \"%s\"\n",
		(*argv) + len);
      }
    }
  }

  showProgOpts(stdout,Options,Help,"\n");
  printf("\n");

  return(0);
error:
  jsperror("testprogopts");
  return(30);
}
#endif
