/*JS*********************************************************************
*
*    Program : POLINT
*    Language: ANSI-C
*    Author  : Joerg Schoen
*    Purpose : Polynomial interpolation.
*
*      This material is largely taken from "Numerical Recipes in C, second
*       edition" and altered to fit in my programming techniques.
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id$";
#endif

/*********     INCLUDES                                         *********/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <jsalloca.h>

#include <jssubs.h>

/*********     DEFINES                                          *********/
/* ERROR-DEFINITIONS from polint label _ERR_POLINT ord 9
   X values are identical
*/

#define Prototype extern
/*********     PROTOTYPES                                       *********/
Prototype double         polInterpol(double x,const double xa[],
				     const double ya[],int n,double *pDy);
Prototype double        *polIntData(const double *x,const double *y,
				    int n,int nNew,double *yNew);

/*********     GLOBAL VARIABLES                                 *********/

/*JS*********************************************************************
*   Given arrays xa[0 .. n-1] and ya[0 .. n-1], and given a value x, this
*    routine returns a value y, and an error estimate *pDy. If P(x) is the
*    polynomial of degree n-1 such that P(xa[i]) = ya[i], i=0,...,n-1,
*    then the returned value y = P(x). If n < 0, then the values are
*    assumed to be equidistant in [xa[0],xa[1]].
*************************************************************************/

double polInterpol(double x,const double xa[],const double ya[],int n,
		   double *pDy)

/************************************************************************/
{
  int m,ns,isign;
  double dif,*c,*d,y,dy;

  if(n > 0) {
    isign = 1;
  } else {
    isign = -1;
    n = -n;
  }

  if((c = (double *)
#ifdef CONFIG_USE_ALLOCA
      alloca(2 * n * sizeof(*c))
#else
      malloc(2 * n * sizeof(*c))
#endif
      ) == NULL) return(HUGE_VAL);
  d = &c[n];

  /*  Here we find the index ns of the closest table entry  */
  if(isign > 0) {
    ns = 0;
    dif = fabs(x - xa[0]);
    c[0] = d[0] = ya[0];
    for(m = 1 ; m < n ; m++) {
      double dift;

      if((dift = fabs(x - xa[m])) < dif) {
	ns = m;
	dif = dift;
      }
      /*  Initialize the tableaus of c's and d's  */
      c[m] = d[m] = ya[m];
    }
  } else {
    dif = (xa[1] - xa[0]) / (n - 1);
    ns = (int)((x - xa[0]) / dif + 0.5);
    if(ns < 0) ns = 0;
    else if(ns >= n) ns = n - 1;

    for(m = 0 ; m < n ; m++) c[m] = d[m] = ya[m];
  }

  y = ya[ns--];
  for(m = 1 ; m < n ; m++) { /*  for each column of the tableau  */
    int i;

    /*  We loop over the current c's and d's and update them  */
    for(i = 0 ; i < (n - m) ; i++) {
      double ho,hp,w,den;

      if(isign > 0) {
	ho = xa[i    ] - x;
	hp = xa[i + m] - x;
      } else {
	ho = xa[0] + dif * i - x;
	hp = ho + dif * m;
      }
      w = c[i + 1] - d[i];

      /*  This error can occur only if two input xa's are
       *   (to within roundoff) identical.
       */
      if((den = ho - hp) == 0.0) {
	JSErrNo = _ERR_POLINT + 0;
	return(HUGE_VAL);
      }
      den = w / den;

      /*  Here c's and d's are updated  */
      d[i] = hp * den;
      c[i] = ho * den;
    }

    /*  After each column in the tableau is completed, we decide
     *   which correction, c or d, we want to add to our accumulating
     *   value of y, i.e., which path to take through the tableau --
     *   forking up or down. We do this in such a way as to take the
     *   most "straight line" route through the tableau to its apex,
     *   updating ns accordingly to keep track of where we are. This
     *   route keeps the partial approximations centered (insofar as
     *   possible) on the target x. The last dy added is thus the error
     *   indication.
     */
    dy = (2 * (ns + 1) < (n - m) ? c[ns + 1] : d[ns--]);
    y += dy;
  }

  if(pDy) *pDy = dy;

#ifndef CONFIG_USE_ALLOCA
  free(c);
#endif

  return(y);
}

/*JS*********************************************************************
*   Simple application: Interpolate data y[i] = f(x[i]), i=0, .. ,n-1
*    to yield nNew values in yNew that cover same area on equidistant grid.
*    x may be NULL meaning original data are equidistant as well. If yNew
*    is NULL, memory is automatically allocated.
*************************************************************************/

double *polIntData(const double *x,const double *y,int n,int nNew,double *yNew)

/************************************************************************/
{
  double h,x0,xa[2];
  int i;

  if(x) {
    x0 = x[0];
    h = (x[n-1] - x0) / (nNew - 1);
  } else {
    x0 = xa[0] = 0.0;
    xa[1] = (double)(nNew - 1);
    h = 1.0;
  }

  if(yNew == NULL && (yNew = (double *)malloc(nNew * sizeof(*yNew))) == NULL)
    return(NULL);

  for(i = 0 ; i < nNew ; i++)
    yNew[i] = polInterpol(x0 + i * h,x ? x : xa,y,x ? n : -n,NULL);

  return(yNew);
}
