/*JS*********************************************************************
*
*    Program : PLOTPARSE
*    Language: ANSI-C to be used with the 'yacc' processor.
*    Author  : Joerg Schoen
*    Purpose : Input parser for program plot.
*
*************************************************************************/
%{

#ifndef lint
static char rcsid[] = "$Id: plotparse.y,v 4.1 1995/10/30 09:34:47 joerg Stab joerg $";
#endif

/*********     INCLUDES 					*********/
#include <errno.h>

#include "plot.h"

#define YYDEBUG 0

/*********     DEFINES						*********/
/*  What message to display when waiting for a command	 */
#define COMMANDMESSAGE "PLOT: "

#define MAXLINE 500

typedef struct PlotCommand {
  int PC_Command;
  union {
    unsigned long PCC_Part;
    char	 *PCC_Name;
  } PC_Content;

  int PC_Length; /*  used for '(' command    */
  BoundPair *PC_Bounds;
  int        PC_NBounds;

  BoundPair *PC_Scales;
  int        PC_NScales;

  char **PC_DeviceOptions;
  int    PC_NOptions;
} PlotCommand;

/*  Some useful abbreviations	*/
#define PC_Part PC_Content.PCC_Part
#define PC_Name PC_Content.PCC_Name

/*  Memory for the commands    */
static PlotCommand Commands[MAXCOMMAND];
static int CurrentCommand;
static int CurrentBracket;

static int DoNext = 0;

static FPPicture *Picture;
static int  ReadMode;
static char *Cursor;

static char LineBuffer[MAXLINE];

/*********     PROTOTYPES					*********/
Prototype int		 parse(char *commandString,int mode);
static int		 yyparse(void);
static int		 yylex(void);
static void		 yyerror(char *string);
static int		 interpretCommand(int start,int len,int indent);
static void              freeCommand(PlotCommand *command);

%}

/*   Start symbol for parsing  */
%start lines

/*   Tokens can be integers of double values   */
%union
{
  int	        IValue;
  unsigned long UValue;
  double        DValue;
  BoundValue    BValue;
  char	       *SValue;
}

/* ****  Tokens  **** */
%token <IValue> INUMBER
%token <DValue> DNUMBER
%token <SValue> STRING

/* ****  Types	 **** */
%type <UValue> parts part norminum
%type <DValue> dnumber
%type <BValue> dnumbound
%type <SValue> opt_string

%%  /* **********   SECTION RULES    *********** */

lines: /*  empty   */
  /*
   *  Commands may not be continued through line end
   *   or we might run into trouble, because strings
   *   etc are only buffered linewise.
   */
  | lines commands '\n'
    {
      int i;

      /*   Interpret commands	*/
      if(interpretCommand(0,CurrentCommand,0) == 1)
	YYACCEPT;

      for(i = 0 ; i < CurrentCommand ; i++)
	freeCommand(&Commands[i]);

      CurrentCommand = 0; /*  Reset to start  */
    }
  /*  In case of error, discard everything till end of line  */
  | lines error '\n'
    {
      yyerrok;
      CurrentCommand = 0; /*  Reset to start  */
    }
  ;

commands: /*  empty   */
  | commands command
    {
      /*   Check number of argumenst against overflow	*/
      if(CurrentCommand > MAXCOMMAND) {
	fprintf(stderr,"ERROR: To many commands -- Line ignored.\n");
	CurrentCommand = 0;
	YYERROR;
      }
    }
  ;

command:
  command_0
    {
      Commands[CurrentCommand++].PC_Command = $<IValue>1;
    }
  | command_args commandargs
    {
      Commands[CurrentCommand++].PC_Command = $<IValue>1;
    }
  | command_s_opt opt_string
    {
      Commands[CurrentCommand].PC_Command = $<IValue>1;
      Commands[CurrentCommand].PC_Name	  = $2;
      CurrentCommand++;
    }
  /* *****   Repeat command  ****** */
  | INUMBER
    {
      /*   Make repeat command	  */
      Commands[CurrentCommand].PC_Command = '(';
      Commands[CurrentCommand].PC_Part	  = $1; /*  repeat count   */
      Commands[CurrentCommand].PC_Length  = 1;
      CurrentCommand++;
    }
    command
  | INUMBER '('
    {
      /*  Save start of bracket  */
      Commands[CurrentCommand].PC_Command = '(';
      Commands[CurrentCommand].PC_Part	  = $1; /*  repeat count   */
      Commands[CurrentCommand].PC_Length  = CurrentBracket;
      CurrentBracket = CurrentCommand;
      CurrentCommand++;
    }
    /*	 Don't want to allow no commands in brackets like "()",   */
    /*	  so I have to give "commands command" instead of simply  */
    /*	  "commands"                                              */
    commands command ')'
    {
      int saveBracket;

      /*  Set end of this bracket    */
      saveBracket = Commands[CurrentBracket].PC_Length;
      Commands[CurrentBracket].PC_Length = CurrentCommand - CurrentBracket - 1;

      CurrentBracket = saveBracket;
    }
  | '=' '('
    {
      /*  Save start of bracket  */
      Commands[CurrentCommand].PC_Command = '=';
      Commands[CurrentCommand].PC_Length  = CurrentBracket;
      CurrentBracket = CurrentCommand;
      CurrentCommand++;
    }
    /*	 Don't want to allow no commands in brackets like "()",   */
    /*	  so I have to give "commands command" instead of simply  */
    /*	  "commands"                                              */
    commands command ')'
    {
      int saveBracket;

      /*  Set end of this bracket    */
      saveBracket = Commands[CurrentBracket].PC_Length;
      Commands[CurrentBracket].PC_Length = CurrentCommand - CurrentBracket - 1;

      CurrentBracket = saveBracket;
    }
  ;

command_0:
  'h' | '+' | 'n' | 'q' | 'e'
  ;

command_args:
  'p' | 'P' | 's' | 'S' | 'c' | 'C' | 'd' | 'D' | 'w' | 'W'
  ;

command_s_opt:
  'o' | 'a' | 'l' | 'L' | 't' | 'x'
  ;

opt_string: /*  empty  */
  {
    $$ = NULL;
  }
  | STRING
  ;

/*  Optional arguments for commands   */
commandargs: /*  empty  */
  | partlist commandargs
  | arealist commandargs
  | scalelist commandargs
  | devoptarg commandargs
  ;

partlist:
  '[' parts ']'
    {
      if(Commands[CurrentCommand].PC_Part) {
	fprintf(stderr,"WARNING: Part given multiple times -- Ignored.\n");
      } else
	Commands[CurrentCommand].PC_Part = ~($2);
    }
  ;

/* ***	To set range of subpictures to plot   **** */
parts: part
    {
      $$ = $1;
    }
  | parts ',' part
    {
      $$ = $1 | $3;
    }
  ;

part:
  norminum
    {
      /*  Get mask from bit   */
      $$ = (1UL << ($1 - 1));
    }
  | norminum '-' norminum
    {
      /*  Get mask from bitrange   */
      if($1 <= $3) {
	$$ = ((2UL << ($3 - $1)) - 1) << ($1 - 1);
      } else {
	$$ = ((2UL << ($1 - $3)) - 1) << ($3 - 1);
      }
    }
  ;

/*  Normalize number to range 1, .., 31   */
norminum: INUMBER
    {
      if($1 < 1)
	$$ = 1;
      else if($1 > 31)
	$$ = 31;
      else
	$$ = $1;
    }
  ;

/* **  Arealist  ** */
arealist:
  '{' areas '}'
  ;

areas: area
  | areas ';' area
  ;

area:
  inc_area dnumbound
    {
      int n = Commands[CurrentCommand].PC_NBounds - 1;

      /*  Treat empty input as interval too  */
      Commands[CurrentCommand].PC_Bounds[n].BP_Type =
	$2.BV_Attr == BV_NSET ? BP_INTERVALL : BP_POINT;
      Commands[CurrentCommand].PC_Bounds[n].BP_Min =
	Commands[CurrentCommand].PC_Bounds[n].BP_Max = $2;
    }
  | inc_area dnumbound ',' dnumbound
    {
      int n = Commands[CurrentCommand].PC_NBounds - 1;
      Commands[CurrentCommand].PC_Bounds[n].BP_Type = BP_INTERVALL;
      Commands[CurrentCommand].PC_Bounds[n].BP_Min = $2;
      Commands[CurrentCommand].PC_Bounds[n].BP_Max = $4;
    }
  ;

inc_area:
    {
      /*  Increase memory  */
      if((Commands[CurrentCommand].PC_Bounds = (BoundPair *)
	  realloc(Commands[CurrentCommand].PC_Bounds,
		  (Commands[CurrentCommand].PC_NBounds + 1) *
		  sizeof(*(Commands[CurrentCommand].PC_Bounds)))) == NULL) {
	fprintf(stderr,"No memory!\n");
	YYERROR;
      }
      Commands[CurrentCommand].PC_NBounds++;
    }
  ;

scalelist:
  '(' scales ')'
  ;

scales: scale
  | scales ';' scale
  ;

scale:
  inc_scale dnumbound
    {
      int n = Commands[CurrentCommand].PC_NScales - 1;

      /*  Treat empty input as a point too  */
      Commands[CurrentCommand].PC_Scales[n].BP_Type = BP_POINT;
      Commands[CurrentCommand].PC_Scales[n].BP_Min =
	Commands[CurrentCommand].PC_Scales[n].BP_Max = $2;
    }
  | inc_scale dnumbound ',' dnumbound
    {
      int n = Commands[CurrentCommand].PC_NScales - 1;
      Commands[CurrentCommand].PC_Scales[n].BP_Type = BP_INTERVALL;
      Commands[CurrentCommand].PC_Scales[n].BP_Min = $2;
      Commands[CurrentCommand].PC_Scales[n].BP_Max = $4;
    }
  ;

inc_scale:
    {
      /*  Increase memory  */
      if((Commands[CurrentCommand].PC_Scales = (BoundPair *)
	  realloc(Commands[CurrentCommand].PC_Scales,
		  (Commands[CurrentCommand].PC_NScales + 1) *
		  sizeof(*(Commands[CurrentCommand].PC_Scales)))) == NULL) {
	fprintf(stderr,"No memory!\n");
	YYERROR;
      }
      Commands[CurrentCommand].PC_NScales++;
    }
  ;

dnumbound: /*  empty  */
    {
      $$.BV_Attr = BV_NSET;
      $$.BV_Value = 0.0;
    }
  | dnumber
    {
      $$.BV_Attr = BV_EQUAL;
      $$.BV_Value = $1;
    }
  | '>' dnumber
    {
      $$.BV_Attr = BV_LLIM;
      $$.BV_Value = $2;
    }
  | '<' dnumber
    {
      $$.BV_Attr = BV_ULIM;
      $$.BV_Value = $2;
    }
  | '*' dnumber
    {
      $$.BV_Attr = BV_FACTOR;
      $$.BV_Value = $2;
    }
  ;

/*  Change integer to double if needed	  */
dnumber: DNUMBER
  | '-' DNUMBER  {  $$ = -$2; }
  | INUMBER      {  $$ =  $1; }
  | '-' INUMBER  {  $$ = -$2; }
  ;

devoptarg:
  '<' devopts_list '>'
  ;

devopts_list:
  devopt
  | devopts_list ',' devopt
  ;

devopt: STRING
    {
      if((Commands[CurrentCommand].PC_DeviceOptions = (char **)
	  realloc(Commands[CurrentCommand].PC_DeviceOptions,
		  (Commands[CurrentCommand].PC_NOptions + 1) *
		  sizeof(*(Commands[CurrentCommand].PC_DeviceOptions))))
	 == NULL) {
	fprintf(stderr,"No memory!\n");
	YYERROR;
      }

      if((Commands[CurrentCommand].
	  PC_DeviceOptions[Commands[CurrentCommand].PC_NOptions] =
	  strdup($1)) == NULL) {
	fprintf(stderr,"No memory!\n");
	YYERROR;
      }
      Commands[CurrentCommand].PC_NOptions++;
    }
  ;

%%  /* **********   SECTION SUBROUTINES    *********** */

/*JS*********************************************************************
*   YYLEX The lexical analyzer for the plot program.
*************************************************************************/

static int yylex(void)

/************************************************************************/
{
  char *string;

  if(Cursor == NULL) {
    if(Verbose)
      printf(COMMANDMESSAGE);
#ifndef CONFIG_NO_POSIX
    else if(isatty(fileno(stdin)))
      /*  If in quiet mode, do not expect anything from stdin  */
      return(EOF);
#endif

    /*	  Read line    */
    if(fgets(LineBuffer,MAXLINE,stdin) == NULL) return(EOF);
    Cursor = LineBuffer;
  }

  /*  Skip blanks   */
  while(isspace(*(unsigned char *)Cursor)) Cursor++;

  if(isdigit(*(unsigned char *)Cursor) || *Cursor == '.') {
    if(*Cursor != '.') {
      /*  Read in integer    */
      yylval.IValue = strtol(Cursor,&string,10);

      /*   Is number really integer?   */
      if(*string != '.' && *string != 'e') {
	Cursor = string;
	return(INUMBER);
      }
    }

    /*	Read double value   */
    yylval.DValue = strtod(Cursor,&string);

    /*	Check if input was really a number starting with '.'   */
    if(string != Cursor) {
      Cursor = string;
      return(DNUMBER);
    }
  } else if(*Cursor == '\'' || *Cursor == '\"') { /* " */
    /*	Is a string, seek for end (must be in current line)    */
    yylval.SValue = string = Cursor + 1;

    while(*string && *string != *Cursor) {
      if(*string == '\\' && string[1])
	/*  Skip escaped quotes  */
	string += 2;
      else
	string++;
    }

    if(*string) {
      *string = '\0';

      if(*Cursor == '\"') {
        /*  Strings enclosed in '"' may contain C escape sequences.  */
        yylval.SValue[cStrCompile(yylval.SValue,yylval.SValue)] = '\0';
      }
      Cursor = string + 1;
    } else {
      Cursor = string;
      fprintf(stderr,"WARNING: Unfinished String in line!\n");
    }
    return(STRING);
  }

  /*  Check end of line   */
  if(*Cursor == '\0') {
    Cursor = NULL;
    return('\n');
  }

  /*  Return command char as token value too  */
  yylval.IValue = *Cursor++;

  return(yylval.IValue);
}

/*JS*********************************************************************
*   YYERROR Error routine for the parser.
*************************************************************************/

static void yyerror(char *string)

/************************************************************************/
{
  fprintf(stderr,"SYNTAX ERROR: Line ignored.\n");

  return;
}

/*JS*********************************************************************
*   INTERPRETCOMMAND
*************************************************************************/

static int interpretCommand(int start,int len,int indent)

/************************************************************************/
{
  int i,j,ret,c,plFlag;
  PlotMode plotMode;
static int flag; /*  Is current picture already shown  */
  char *text;

  if(indent == 0) flag = 0;

  /*   Set end mark correctly	*/
  if( (len += start) > CurrentCommand) {
    fprintf(stderr,"  SOFTWARE ERROR: Repeat area too big -- Break.\n");
    goto error;
  }

  /*  Here we will interpret the commands   */
  for(i = start ; i < len ; i++) {
    plFlag = 0;
    plotMode = PLAIN;
    text = NULL;

    switch(c = tolower(Commands[i].PC_Command)) {
    case 'h':
      printf("Available commands are:\n\
  h : Print this help information.\n\
  p : Plot current picture with default (surface or contour).\n\
  s : Plot current picture as surface plot. May have an optional argument\n\
      in the form '<angle>' to set the viewpoint.\n\
  c : Plot current picture as contour plot.\n\
  d : Plot current picture as combined surface and contour plot.\n\
  + : Skip current picture.\n\
  n : New page.\n\n\
  l [\"file\"] : Discard current file and load \"file\" for plotting.\n\
               If <file> is omitted, reload the current file.\n\
               Uppercase 'L' prevents starting with a new page.\n\
  o [\"file\"] : Open file for output with command \"w\". If <file> is\n\
               omitted, use stdout.\n\
  a [\"file\"] : Same as \"o\", but appends.\n\
  w : Write current picture to output file.\n\
  t [\"text\"] : Write text hunk to output file. If no text is given, use\n\
               the title of the current picture. A '%%s' sequence in <text>\n\
               accesses the current pictures title.\n\
  x [\"text\"] : Write axe description to output file. If no text is given,\n\
               use axes description of the current picture. Text for\n\
               different axes must be separated with ';'.\n\
  e : Write end hunk to output file.\n\n\
  q : Quit program.\n\n\
 String arguments may appear as \'string\' or \"string\",the second form\n\
 understands C-like escape sequences (like \"\\n\").\n\
  Single commands can be repeated by preceding them with a number and\n\
 command sequences by enclosing them in brackets (Like '4(sc)').\n\
  Plot commands enclosed in brackets preceded by a '=' are plotted into the\n\
 same area (Like '=(ppp)').\n\
  The 's', 'c', 'd' and 'p' command can have optional arguments:\n\n\
   <\"opts\",...>\n\
     Suboptions for device. Similar to command line option '-D<opts>'. The\n\
     options are only valid for the local command.\n\n\
   (xcale) or (xscale;yscale) or (xscale;yscale;zscale)\n\
     Set the scalings for x, y and z values. If the scaling is a single\n\
     number, the data is simply multiplied by it. The form \"start,end\" for\n\
     a scaling makes a linear transformation (y=mx+b) so that the values\n\
     cover the interval [start,end] afterwards. Either start or end may be\n\
     dropped, to set only the upper or lower bound to the given value. In\n\
     that case b=0 in the linear transformation is used.\n\
     To make more general transformations, use command \"analyze\", mode\n\
     \"TRANSFORM\".\n\
   [partlist] where partlist is a comma separated list of subparts of\n\
	      the picture to be plotted (Example: '[1-2,3,5,9-10]').\n\n\
   {xstart,xend} or\n\
   {xstart,xend;ystart,yend} or\n\
   {xstart,xend;ystart,yend;zstart,zend}\n\
     The area of the picture to plot. Boundary values can be given in the\n\
     following forms:\n\
       \"\" (empty): Use the pregiven value.\n\
       \"value\"   : Set to value.\n\
       \"<value\"  : value is upper bound and may decrease.\n\
       \">value\"  : value is lower bound and may increase.\n\
       \"*value\"  : Rescale pregiven one with value.\n\
     Giving only one value for boundaries (like \"{xpoint;...}\" without a\n\
     comma) means taking a cut in this dimension at the value and therefore\n\
     reducing the dimension of the data by one.\n\
 If the command letter is given in upper case, then the current picture is\n\
 kept and can be plotted several times.\n");
      break;
    case '+': /*  Don't need to see that picture is skipped   */
      if(indent == 0) text = "skipped";
      break;
    case 'p': /*  Normal plot  */
      text = "plotted";
      plotMode = PLAIN;
      plFlag = TRUE;
      break;
    case 's': /*  Surface    */
      text = "plotted as surface";
      plotMode = SURFACE;
      plFlag = TRUE;
      break;
    case 'c':  /*   Contour   */
      text = "plotted as contour plot";
      plotMode = CONTOUR;
      plFlag = TRUE;
      break;
    case 'd':  /*   Surface and contour   */
      text = "plotted as combined surface & contour plot";
      plotMode = SURFCONTOUR;
      plFlag = TRUE;
      break;
    case 'n': /*   Newpage   */
      if(Verbose) printf("%*s New page.\n",indent,"");

      if(DoNext == 0) DEVICE->Next(OUTPUTDEV_NEXTNEWPAGE);
      break;
    case 'q': /*   Quit   */
      if(Verbose) printf("%*s Quit.",indent,"");
      goto quit;
    case 'l':
      {
	char *string;

	if((string = Commands[i].PC_Name) == NULL &&
	   (string = DEVICE->fileName) == NULL) {
	  fprintf(stderr,"ERROR command 'l': No default file name!");
	  goto error;
	}

	if(Verbose) printf("%*s Loading file \"%s\".\n",indent,"",string);

	/*  Close old file  */
	if(PlotFile) fplotClose(PlotFile);

	/*  Open new one	  */
	if((PlotFile = fplotOpenExt(string,FPLOT_READ)) == NULL) {
	  fprintf(stderr,"ERROR: Could not open \"%s\"!\n",string);
	  goto error;
	}
	if(DEVICE->fileName) free(DEVICE->fileName);
	DEVICE->fileName = strdup(FPlotOpenName ? FPlotOpenName : string);

	/*  Start new page when lower case  */
	if(islower(Commands[i].PC_Command) && DoNext == 0)
	  DEVICE->Next(OUTPUTDEV_NEXTNEWPAGE|OUTPUTDEV_NEXTNOINC);
      }

      /*  The next picture will automatically loaded in  */
      break;
    case 'o': case 'a':
      if(Verbose)
	printf("%*s Opening file \"%s\".\n",indent,"",
	       Commands[i].PC_Name ? Commands[i].PC_Name : "-");

      /*  Close old file  */
      if(PlotOutFile) {
	/*  Write an end hunk  */
	fplotClose(PlotOutFile);
      }

      /*  Open new one	  */
      if((PlotOutFile = fplotOpenExt(Commands[i].PC_Name,c == 'a' ?
				     FPLOT_APPEND : FPLOT_WRITE)) == NULL) {
	fprintf(stderr,"ERROR: Could not open \"%s\"!\n",
		Commands[i].PC_Name ? Commands[i].PC_Name : "-");
	goto error;
      }

      PlotOutState = 0;
      break;
    case 'w':
      text = "written to output file";
      break;
    case 't':
      text = "title written";
      break;
    case 'x':
      text = "axe description written";
      break;
    case 'e': /*  Write end hunk  */
      text = "end hunk written";
      break;
    case '(': /*   Repeat Command    */
      for(j = 0 ; j < Commands[i].PC_Part ; j++)
	if((ret = interpretCommand(i + 1,Commands[i].PC_Length,indent + 1))) {
	  /*  Check if quit command   */
	  if(ret == 1) goto quit;
	  goto error;
	}

      /*  Skip next commands   */
      i += Commands[i].PC_Length;
      continue;
    case '=': /*   Draw into same plot  */
      DoNext++;
      ret = interpretCommand(i + 1,Commands[i].PC_Length,indent + 1);
      DoNext--;
      if(ret) {
	/*  Check if quit command   */
	if(ret == 1) goto quit;
	goto error;
      }

      /*  Skip next commands   */
      i += Commands[i].PC_Length;
      continue;
    default:
      fprintf(stderr,"  SOFTWARE ERROR: Unknown command '%d' -- Ignored.\n",
	     Commands[i].PC_Command);
      goto error;
    }

    /*	 Show message text  */
    if(text) {
      if(!flag && Picture) {
	showpicture(Picture);
	flag = 1;
      }

      if(Verbose) printf("%*s Picture %d %s.\n",indent,"",CurrentPicture,text);
    }

    /*	 Now plot the picture if wanted    */
    if(plFlag && Picture) {
      /*  Pass options to device  */
      if(Commands[i].PC_NOptions > 0) {
	char *opt;
	int k;

	for(j = 0 ; j < Commands[i].PC_NOptions ; j++) {
	  opt = Commands[i].PC_DeviceOptions[j];
	  if((k = parseProgOpts(opt,DEVICE->devOpts)) < 0)
	    goto error;
	  else if(opt[k]) {
	    fprintf(stderr,"WARNING: Trailing garbage in option ignored.\n\
>>%s\n",opt + k);
	  }
	}
      }

      if(plotPicture(Picture,plotMode,Commands[i].PC_Part,
		     Commands[i].PC_Bounds,Commands[i].PC_NBounds,
		     Commands[i].PC_Scales,Commands[i].PC_NScales)) {
	fprintf(stderr,"ERROR in plotting picture -- Break.\n");
	goto error;
      }

      /*   Next position    */
      if(DoNext == 0) DEVICE->Next((Verbose & 32) ? OUTPUTDEV_NEXTVERBOSE : 0);
    } else if(strchr("twxe",c)) {
      if(c != 't') {
	if(PlotOutFile == NULL) {
	  fprintf(stderr,"ERROR: No output file opened!\n");
	  goto error;
	}

	/*  Have to call fplotStart ?  */
	if(PlotOutState == 0) {
	  PlotOutState = 1;

	  if(fplotStart(PlotOutFile)) {
	    fprintf(stderr,"ERROR: Cannot start fplot routines!\n");
	    goto error;
	  }
	}
      }

      switch(c) {
      case 't':
	{
	  char *string;

	  if((string = Commands[i].PC_Name) == NULL &&
	     (Picture == NULL || (string = Picture->FP_Title) == NULL)) {
	    fprintf(stderr,"WARNING: No title to write!\n");
	  } else if(PlotOutFile == NULL) {
	    /*  Write title into current picture  */
	    if(Picture == NULL)
	      fprintf(stderr,"WARNING: No data loaded!\n");
	    else {
	      if(Picture->FP_Title) free(Picture->FP_Title);
	      if((Picture->FP_Title = strdup(string)) == NULL) goto error;
	    }

	    /*  Allow %s in text to access current pictures title  */
	  } else if(fplotText(PlotOutFile,string,Picture ? Picture->FP_Title :
			      "")) {
	    fprintf(stderr,"ERROR in writing text -- Break.\n");
	    goto error;
	  }
	}
	break;
      case 'w':
	if(Picture == NULL) {
	  fprintf(stderr,"WARNING: No data to write!\n");
	} else if(writePicture(Picture,0,Commands[i].PC_Part,
			       Commands[i].PC_Bounds,Commands[i].PC_NBounds,
			       Commands[i].PC_Scales,Commands[i].PC_NScales,
			       PlotOutFile)) {
	  fprintf(stderr,"ERROR in writing picture -- Break.\n");
	  goto error;
	}
	break;
      case 'x':
	if(Commands[i].PC_Name) {
	  if(fplotAxesS(PlotOutFile,Commands[i].PC_Name)) {
	    fprintf(stderr,"ERROR in writing axe description -- Break.\n");
	    goto error;
	  }
	} else if(Picture && Picture->FP_Axes) {
	  if(fplotAxes(PlotOutFile,Picture->FP_NAxes,
		       (const char **)Picture->FP_Axes)) {
	    fprintf(stderr,"ERROR in writing axe description -- Break.\n");
	    goto error;
	  }
	} else {
	  fprintf(stderr,"WARNING: No axe description to write!\n");
	}
	break;
      case 'e':
	if(fplotEnd(PlotOutFile)) goto error;
	PlotOutState = 0;
	break;
      }
    }

    /*  Go to next picture when command is lowercase or + or 'L'  */
    /*   Exclude n,a,o,t,e,x commands  */
    if((islower(Commands[i].PC_Command) ||
	strchr("+L",Commands[i].PC_Command)) &&
       strchr("naotehx",Commands[i].PC_Command) == NULL) {
      if(Picture) {
	fplotFreepic(Picture);
	Picture = NULL;
      }

      /*  Check if currently a file is open   */
      if(PlotFile) {
	/*    Read next picture    */
	if((Picture = fplotRead(PlotFile,ReadMode|FPLOT_MODE_READ))) {
	  CurrentPicture++;

	  /*  Only in lowest iteration show pictures   */
	  if(indent == 0) {
	    showpicture(Picture);
	    flag = 1;
	  } else {
	    flag = 0;
	  }
	} else {
	  /*  If files remain, quit this run of parsing   */
	  if(RemainingFiles) goto quit;

	  /*  Close current file   */
	  fplotClose(PlotFile);
	  if(DEVICE->fileName) free(DEVICE->fileName);
	  PlotFile = NULL;
	  DEVICE->fileName = NULL;

	  CurrentPicture = 0;

	  if(Verbose) printf("No more pictures in file -- Closed.\n");
	}
      } else {
	fprintf(stderr,"ERROR: No plot file opened -- Load one or quit!\n");
	goto error;
      }
    }
  }

  return(0);
error:
  if(errno > 0) {
    fprintf(stderr,"SYSTEM ERROR: %s\n",strerror(errno));
    errno = 0;
  } else if(JSErrNo) {
    fprintf(stderr,"LIBRARY ERROR: %s\n",jsStrError(JSErrNo));
    JSErrNo = 0;
  }

  return(-1);
quit:
  return(1);
}

/*JS*********************************************************************
*   FREECOMMAND
*************************************************************************/

static void freeCommand(PlotCommand *command)

/************************************************************************/
{
  /*  No boundaries given  */
  if(command->PC_NBounds > 0) {
    free(command->PC_Bounds);
    command->PC_Bounds  = NULL;
    command->PC_NBounds = 0;
  }

  command->PC_Part = 0; /*  All parts to be plotted  */

  /*  All scalings set to default  */
  if(command->PC_NScales > 0) {
    free(command->PC_Scales);
    command->PC_Scales  = NULL;
    command->PC_NScales = 0;
  }

  if(command->PC_NOptions > 0) {
    int i;

    for(i = 0 ; i < command->PC_NOptions ; i++)
      free(command->PC_DeviceOptions[i]);

    free(command->PC_DeviceOptions);

    command->PC_DeviceOptions = NULL;
    command->PC_NOptions = 0;
  }


}

/*JS*********************************************************************
*   PARSE Call the parser and plot the picture.
*************************************************************************/

int parse(char *commandString,int mode)

/************************************************************************/
{
  /*  Copy to global variable	*/
  CurrentCommand = 0;
  CurrentBracket = -1;

  ReadMode = mode;

  Cursor = commandString;

  if(PlotFile) {
    /*	Read first picture	  */
    if((Picture = fplotRead(PlotFile,ReadMode|FPLOT_MODE_READ)) == NULL)
      return(-1);
  }

  /*   Call parser and return	 */
  return( yyparse() );
}
