/*JS*********************************************************************
*
*    Program : PLOT
*    Language: ANSI-C, using TW-Graph library for producing graphics.
*    Author  : Joerg Schoen
*    Purpose : Plots data
*
*    Part    : Device dependent routines.
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id: plotdev.c,v 4.1 1995/10/30 09:34:47 joerg Stab joerg $";
#endif

/*********     INCLUDES 					*********/
#include "plot.h"
#include <time.h>

/*********     DEFINES						*********/
/*  Demo device  */
#define CONFIG_USE_DEMODEV

/*  For now just TWGraph....  */
#ifndef CONFIG_NO_TWGRAPH
# define CONFIG_USE_TWGRAPH
#endif

/*********     PROTOTYPES					*********/
Prototype OutputDevice  *OutDev[];
Prototype int            NOutputDevices,CurrentDevice;

static OutputDevice      DeviceTWG,DeviceDemo;

/*********     GLOBAL VARIABLES 				*********/
OutputDevice *OutDev[] = {
#ifdef CONFIG_USE_TWGRAPH
  &DeviceTWG,
#endif
#ifdef CONFIG_USE_DEMODEV
  &DeviceDemo,
#endif
};

int NOutputDevices = sizeof(OutDev) / sizeof(OutputDevice*);
int CurrentDevice = 0;

#ifdef CONFIG_USE_DEMODEV
/* ***  Local prototypes  *** */
static int   demoInit(char *name);
static char *demoDeInit(void);

static int demoArea(PlotType type,double xMin,double xMax,
		   double yMin,double yMax,double zMin,double zMax);
static int demoNext(int what);

static int demoDraw2D(PlotMode mode,void *xValues,void *yValues,int nx);
static int demoDraw3D(PlotMode mode,void *xValues,void *yValues,
		     void *data,int nx,int ldx,int ny);

static int demoDrawLegend(PlotType type,PlotMode mode,char *title,
			 char *xAxe,char *yAxe,char *zAxe);

static OutputDevice DeviceDemo = {
  "demo",
  0,
  NULL, NULL,
  NULL,

  demoInit,demoDeInit,
  demoArea,demoNext,
  demoDraw2D,demoDraw3D,
  demoDrawLegend,

  0.0,0.0
};

/*JS*********************************************************************
*   DEMOINIT
*************************************************************************/

static int demoInit(char *name)

/************************************************************************/
{
  OutputDevice *d = &DeviceDemo;

  printf("**** DEMO DEVICE: Initializing ****\n");

  return(0);
}

/*JS*********************************************************************
*   DEMODEINIT
*************************************************************************/

static char *demoDeInit(void)

/************************************************************************/
{
  printf("**** DEMO DEVICE: Deinitializing ****\n");

  return(NULL);
}

/*JS*********************************************************************
*   DEMOAREA
*************************************************************************/

static int demoArea(PlotType type,double xMin,double xMax,
		   double yMin,double yMax,double zMin,double zMax)

/************************************************************************/
{
  OutputDevice *d = &DeviceDemo;

  printf("**** DEMO DEVICE: Set area [%g,%g]x[%g,%g]  [%g,%g] ****\n",
	 xMin,xMax,yMin,yMax,zMin,zMax);

  return(0);
}

/*JS*********************************************************************
*   DEMONEXT
*************************************************************************/

static int demoNext(int what)

/************************************************************************/
{
  OutputDevice *d = &DeviceDemo;

  printf("**** DEMO DEVICE: Next picture %d ****\n",what);

  return(0);
}

/*JS*********************************************************************
*   DEMODRAW2D
*************************************************************************/

static int demoDraw2D(PlotMode mode,void *xValues,void *yValues,int nx)

/************************************************************************/
{
  OutputDevice *d = &DeviceDemo;

  printf("**** DEMO DEVICE: Draw 2D (%d) %s values ****\n",nx,
	 xValues ? "with" : "without");

  return(0);
}

/*JS*********************************************************************
*   DEMODRAW3D
*************************************************************************/

static int demoDraw3D(PlotMode mode,void *xValues,void *yValues,
		     void *data,int nx,int ldx,int ny)

/************************************************************************/
{
  OutputDevice *d = &DeviceDemo;
  int i,j;

  printf("**** DEMO DEVICE: Draw 3D (%dx%d) 3D %s values ****\n",nx,ny,
	 xValues ? "with" : "without");

#ifdef XDEBUG
  for(i = 0 ; i < nx ; i++) {
    for(j = 0 ; j < ny ; j++) {
      printf("A[");
      if(xValues) printf("x=%g",((float *)xValues)[i]);
      else        printf("%d",i);
      if(yValues) printf(" y=%g",((float *)yValues)[i]);
      else        printf(" %d",j);
      printf("] = %g\n",((float *)data)[i + j * ldx]);
    }
  }
#endif

  return(0);
}

/*JS*********************************************************************
*   DEMODRAWLEGEND
*************************************************************************/

static int demoDrawLegend(PlotType type,PlotMode mode,char *title,
			 char *xAxis,char *yAxis,char *zAxis)

/************************************************************************/
{
  OutputDevice *d = &DeviceDemo;

  printf("**** DEMO DEVICE: Draw legend: \"%s\" ****\n\
****              Axes: \"%s\" \"%s\" \"%s\" ****\n",
	 title ? title : "<no title>",xAxis ? xAxis : "<no text>",
	 yAxis ? yAxis : "<no text>",zAxis ? zAxis : "<no text>");

  return(0);
}
#endif

#ifdef CONFIG_USE_TWGRAPH
/* ************************************************************ */
/* ***                 TW Graph device                      *** */
/* ************************************************************ */
#include <twgraph.h>
#include <fortname.h>

/* ***  Precision and workspace to grant TW Graph  *** */
#define TWG_PRECISION  float
#define LWORK 100000

/* ***  Internal limits  *** */
#define MAXXPICS         3 /*  maximal number of pictures in x direction  */
#define MAXYPICS         4 /*  maximal number of pictures in y direction  */
#define MAXPICS         12 /*  maximal number of pictures on page  */
#define PICSPERPAGE      1 /*  Default number of pictures per page  */
#define MAXLEVELS      100 /*  Maximal number of levels for contour plot  */
#define NOOFLEVELS      10 /*  Default number of levels for contour plot  */
#define MAXLINEVALUES   50 /*  Maximum number of different line types  */
/*  Position of contour plot in case of combined surface + contour plot  */
#define CONTOURHEIGHT 0.20

/*  Rotation angle for 3D plots  */
#define ROTATEANGLE   30.0

#define PICSTRETCH  0.8  /*  Scale picture size so that text won't overlap  */
/*  Margin settings around pictures  */
#define SPACEX      1.0
#define SPACEY      1.0
/*  Usable page size  */
#define PAGELEN     (29.7 - 2.0 * SPACEX)
#define PAGEWIDTH   (21.0 - 2.0 * SPACEY)
/*  Start values for first picture  */
#define STARTX      SPACEX
#define STARTY      (PAGEWIDTH + SPACEY)

#define MAXLINETYPES  10
#define LINEWIDTH    1.8 /*  Thickness for lines  */

#define OUTPUTFILE   "PSC1" /*  default output file of TW Graph  */

/* ***  Increase work space for TWGraph  *** */
TWG_PRECISION FORTRANNAME(twgwrk)[LWORK];

/* ***  Global settings (only used during initialization)  *** */
static int TWPicsPerPage = PICSPERPAGE;
static double TWPicWidth,TWPicLen;

/* ***  Definitions for TW{Global,Local}.Mode variable  *** */
#define TWMODE_CONTOURPLOT_N   0
#define TWMODE_HIDDENLINES_N   1
#define TWMODE_PLOTLINES_N     2
#define TWMODE_PAGELABEL_N     3
#define TWMODE_TITLE_N         4
#define TWMODE_CONTDESCR_N     5
#define TWMODE_LEGEND_N        6
#define TWMODE_VERBOSE_N       7
#define TWMODE_PSFONTS_N       8
#define TWMODE_USECOLOR_N      9
#define TWMODE_USEBBEXT_N     10

/* ***  Definitions for TW{Global,Local}.AxeMode  *** */
/*  Offset for x / y / z axe  */
#define TWAFL_XOFFSET_N    0
#define TWAFL_YOFFSET_N   11
#define TWAFL_ZOFFSET_N   22

/*   Suboffsets for axes   */
#define TWAFL_LABEL_N      0 /*  Offset for axe label  */
#define TWAFL_LOG_N        1 /*  Logarithmic scale  */

/*  Suboffsets for one axe description  */
/*  x/y "bottom" and z "left" */
#define TWAFL_LINE_BN      2
#define TWAFL_TICKS_BN     3
#define TWAFL_NUMBERS_BN   4
#define TWAFL_OUTWARDS_BN  5

/*  x/y "top" and z "right"  */
#define TWAFL_LINE_TN      6
#define TWAFL_TICKS_TN     7
#define TWAFL_NUMBERS_TN   8
#define TWAFL_OUTWARDS_TN  9
/*#define TWAFL_???         10*/

/*  Mask for axe description (LINE ... OUTWARDS)  */
#define TWAFL_MASK         0x0f
#define TWAFL_MASK3D       0x07 /*  ignore outwards for 3D axes  */
#define TWAFL_SHIFT_B      2  /*  Shift to get to LINE_B ... OUTWARDS_B)  */
#define TWAFL_SHIFT_T      6  /*  Shift to get to LINE_T ... OUTWARDS_T)  */

/* ***  Default settings come here  *** */
static struct {
  unsigned long Mode;
  unsigned long AxeMode;
  int LineTypes[MAXLINEVALUES + 1],NoOfLevels;
  double LineWidth,SymbolSize;
  double SurfRotate,TextScale,ContHeight;
  char FontName[MAXLENGTH + 1];
  char PageLabel[MAXLENGTH + 1];
} TWGlobal,TWLocal = {
  /* ***  Settings for Mode  *** */
  (1UL << TWMODE_PLOTLINES_N) |
  (1UL << TWMODE_PAGELABEL_N) |
  (1UL << TWMODE_TITLE_N) |
  (1UL << TWMODE_CONTDESCR_N) |
  (1UL << TWMODE_LEGEND_N) |
#if 0
  /*  Do not use it as a default  */
  (1UL << TWMODE_USEBBEXT_N) |
#endif
  /*  PostScript fonts are used as a default  */
  (1UL << TWMODE_PSFONTS_N),

  /* ***  Settings for AxeMode  *** */
  /*  x axe  */
  (1UL << (TWAFL_XOFFSET_N + TWAFL_LABEL_N)) |
  (1UL << (TWAFL_XOFFSET_N + TWAFL_LINE_BN)) |
  (1UL << (TWAFL_XOFFSET_N + TWAFL_TICKS_BN)) |
  (1UL << (TWAFL_XOFFSET_N + TWAFL_NUMBERS_BN)) |
  (1UL << (TWAFL_XOFFSET_N + TWAFL_LINE_TN)) |
  (1UL << (TWAFL_XOFFSET_N + TWAFL_TICKS_TN)) |
  /*  y axe  */
  (1UL << (TWAFL_YOFFSET_N + TWAFL_LABEL_N)) |
  (1UL << (TWAFL_YOFFSET_N + TWAFL_LINE_BN)) |
  (1UL << (TWAFL_YOFFSET_N + TWAFL_TICKS_BN)) |
  (1UL << (TWAFL_YOFFSET_N + TWAFL_NUMBERS_BN)) |
  (1UL << (TWAFL_YOFFSET_N + TWAFL_LINE_TN)) |
  (1UL << (TWAFL_YOFFSET_N + TWAFL_TICKS_TN)) |
  /*  z axe  */
  (1UL << (TWAFL_ZOFFSET_N + TWAFL_LABEL_N)) |
  (1UL << (TWAFL_ZOFFSET_N + TWAFL_LINE_BN)) |
  (1UL << (TWAFL_ZOFFSET_N + TWAFL_TICKS_BN)) |
  (1UL << (TWAFL_ZOFFSET_N + TWAFL_NUMBERS_BN)) |
  (1UL << (TWAFL_ZOFFSET_N + TWAFL_LINE_TN)) |
  (1UL << (TWAFL_ZOFFSET_N + TWAFL_TICKS_TN)),

  /* ***  Other defaults  *** */
  { 10, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10 },
  NOOFLEVELS,
  LINEWIDTH, 1.0,
  ROTATEANGLE, 1.0, CONTOURHEIGHT,
  "",
  ""
};

ProgOpt TWAxeOptions[] = {
  /*  Used for all axes  */
  { "label",     PROPT_BOOL | TWAFL_LABEL_N,       &TWLocal.AxeMode },
  { "log",       PROPT_BOOL | TWAFL_LOG_N,         &TWLocal.AxeMode },

  { "botline",   PROPT_BOOL | TWAFL_LINE_BN,       &TWLocal.AxeMode },
  { "botticks",  PROPT_BOOL | TWAFL_TICKS_BN,      &TWLocal.AxeMode },
  { "botnumber", PROPT_BOOL | TWAFL_NUMBERS_BN ,   &TWLocal.AxeMode },
  { "botout",    PROPT_BOOL | TWAFL_OUTWARDS_BN,   &TWLocal.AxeMode },

  { "topline",   PROPT_BOOL | TWAFL_LINE_TN,       &TWLocal.AxeMode },
  { "topticks",  PROPT_BOOL | TWAFL_TICKS_TN,      &TWLocal.AxeMode },
  { "topnumber", PROPT_BOOL | TWAFL_NUMBERS_TN ,   &TWLocal.AxeMode },
  { "topout",    PROPT_BOOL | TWAFL_OUTWARDS_TN,   &TWLocal.AxeMode },

  /*   Abbreviations  */
  { "line",      PROPT_ABBREV,               "botline,topline"         },
  { "noline",    PROPT_ABBREV,               "nobotline,notopline"     },
  { "ticks",     PROPT_ABBREV,               "botticks,topticks"       },
  { "noticks",   PROPT_ABBREV,               "nobotticks,notopticks"   },
  { "number",    PROPT_ABBREV,               "botnumber,topnumber"     },
  { "nonumber",  PROPT_ABBREV,               "nobotnumber,notopnumber" },
  { "noaxe",     PROPT_ABBREV,               "noline,noticks,nonumber" },
  { NULL, 0, NULL }
};

/* ***  Options for TW Graph  *** */
ProgOpt TWOptions[] = {
  /*  General options  */
  { "verbose",      PROPT_BOOL | TWMODE_VERBOSE_N,      &TWLocal.Mode       },
  { "picsperpage",  PROPT_IVALUE,                       &TWPicsPerPage      },
#define NUM_OPT_PICWIDTH  2
  { "picwidth",     PROPT_DVALUE,                       &TWPicWidth         },
#define NUM_OPT_PICLEN    3
  { "piclen",       PROPT_DVALUE,                       &TWPicLen           },
  { "plotlines",    PROPT_BOOL | TWMODE_PLOTLINES_N,    &TWLocal.Mode       },
  { "linetype",     PROPT_IVALUE | MAXLINEVALUES,        TWLocal.LineTypes  },

  /*  Frame options  */
  { "title",        PROPT_BOOL | TWMODE_TITLE_N,        &TWLocal.Mode       },
  { "xaxeopt",      PROPT_SUBOPT | TWAFL_XOFFSET_N,     &TWAxeOptions[0]    },
  { "yaxeopt",      PROPT_SUBOPT | TWAFL_YOFFSET_N,     &TWAxeOptions[0]    },
  { "zaxeopt",      PROPT_SUBOPT | TWAFL_ZOFFSET_N,     &TWAxeOptions[0]    },
  { "pagelabel",    PROPT_BOOL | TWMODE_PAGELABEL_N,    &TWLocal.Mode       },
  { "legend",       PROPT_BOOL | TWMODE_LEGEND_N,       &TWLocal.Mode       },
  { "textscale",    PROPT_DVALUE,                       &TWLocal.TextScale  },

  /*  Options for contour plot  */
#define NUM_OPT_CONTOUR   13
  { "contour",      PROPT_IVALUE,                       &TWLocal.NoOfLevels },
  { "contdescr",    PROPT_BOOL | TWMODE_CONTDESCR_N,    &TWLocal.Mode       },

  /*  Options for surface plot and combined surface/contour plot  */
  { "hiddenlines",  PROPT_BOOL | TWMODE_HIDDENLINES_N,  &TWLocal.Mode       },
  { "linewidth",    PROPT_DVALUE,                       &TWLocal.LineWidth  },
  { "symbolsize",   PROPT_DVALUE,                       &TWLocal.SymbolSize },
  { "rotatesurf",   PROPT_DVALUE,                       &TWLocal.SurfRotate },
  { "contheight",   PROPT_DVALUE,                       &TWLocal.ContHeight },

  /*  Special options  */
  { "psfonts",      PROPT_BOOL | TWMODE_PSFONTS_N,      &TWLocal.Mode       },
  { "bbext",        PROPT_BOOL | TWMODE_USEBBEXT_N,     &TWLocal.Mode       },
  { "fontname",     PROPT_STRING | MAXLENGTH,            TWLocal.FontName   },
  { "tpagelabel",   PROPT_STRING | MAXLENGTH,            TWLocal.PageLabel  },
  { "color",        PROPT_BOOL | TWMODE_USECOLOR_N,     &TWLocal.Mode       },

  /*  Abbreviations  */
  { "nodescript",   PROPT_ABBREV, "notitle,nopagelabel,nolegend,nocontdescr" },
  { "noaxelabel",   PROPT_ABBREV,
    "xaxeopt=nolabel;yaxeopt=nolabel;zaxeopt=nolabel" },

  { NULL, 0, NULL }
};

#define STRINGIZE2(x)  #x
#define STRINGIZE(x)   STRINGIZE2(x)

const char *TWHelp[] = {
  "Show TW Graph calls",
  "Set number of pictures per page (1.."STRINGIZE(MAXPICS)")",
  "Set width of pictures (cm)",
  "Set height of pictures (cm)",
  "If unset, plot only data points",
  "Set line patterns (1.."STRINGIZE(MAXLINETYPES)")",

  "Plot pictures title",
  "Set axe options for x, y and z bottom and top axes: choose line, ticks \
and numbers, draw ticks outwards or inwards",
  NULL, NULL,
  "Print page label",
  "Print legend in case of multiple lines",
  "Set text scaling factor",

  "Plot 2D-data as contour plot and set number of intermediate contour lines",
  "Plot labels for contour lines",

  "Show hidden lines for surface plots",
  "Set line thickness",
  "Set size for point symbols",
  "Rotate 3D surface plot",

  "For combined surface and contour plot: Height where contour plot is shown",
  "Use PostScript font in TW Graph",
  "Use bounding box extension of TW Graph",
  "Set PostScript font for texts",
  "Set page label explicitly",
  "Use colors",
  "Inhibit all kind of global descriptions",
  "Inhibit all axe labels"
};

/*  ***  Internal auxiliary variables  *** */
/*  Map user axe mode to TW Graph values  */
static int TWMapAxeMode[TWAFL_MASK + 1] = {
    0,   1,   2,   3,  10,  11,  12,  13,
    0,  -1,  -2,  -3, -10, -11, -12, -13
};

/*   NxPics contains number of pictures in the horizontal direction	*/
/*    for a given total number of pictures.				*/
/*    1   2   3   4   5   6   7   8   9  10  11  12			*/
static int NxPics[MAXPICS + 1] = {
  0 , 1 , 2 , 3 , 2 , 3 , 3 , 4 , 4 , 3 , 4 , 4 , 4
};

static int PageNr,PicNr,PicNrX,Counter;
static double PicPosX,PicPosY;
static double XMin,XMax,YMin,YMax,ZMin,ZMax;
static char *OutputName;
static int MaxColors;
/*
 *  PlotFlag:
 *   Bit 0: Initialization done.
 *   Bit 1: Boundary values set.
 *   Bit 2: Legend drawn.
 *
 */
static int PlotFlag;

/* ***  Local prototypes  *** */
static int   twgInit(char *name);
static int   twgSetOpts(void);
static char *twgDeInit(void);

static int twgArea(PlotType type,double xMin,double xMax,
		   double yMin,double yMax,double zMin,double zMax);
static int twgNext(int what);

static int twgDraw2D(PlotMode mode,void *xValues,void *yValues,int nx);
static int twgDraw3D(PlotMode mode,void *xValues,void *yValues,
		     void *data,int nx,int ldx,int ny);

static int twgDrawLegend(PlotType type,PlotMode mode,char *title,
			 char *xAxe,char *yAxe,char *zAxe);

static OutputDevice DeviceTWG = {
  "twg",
  0,
  TWOptions,
  TWHelp,
  NULL,

  twgInit,twgDeInit,
  twgArea,twgNext,
  twgDraw2D,twgDraw3D,
  twgDrawLegend,

  0.0,0.0
};

/*JS*********************************************************************
*   TWGINIT
*************************************************************************/

static int twgInit(char *name)

/************************************************************************/
{
  OutputDevice *d = &DeviceTWG;
  int i;

  /*  Avoid multiple callings  */
  if(PlotFlag & 1) return(0);
  PlotFlag |= 1;

  /* ***  Some sanity checks  *** */
  if(TWLocal.NoOfLevels > MAXLEVELS)
    TWLocal.NoOfLevels = MAXLEVELS;
  else if(TWLocal.NoOfLevels < 1)
    TWLocal.NoOfLevels = 1;

  if(TWPicsPerPage > MAXPICS)
    TWPicsPerPage = MAXPICS;
  else if(TWPicsPerPage < 1)
    TWPicsPerPage = 1;

  while(TWLocal.SurfRotate >= 360)
    TWLocal.SurfRotate -= 360;

  /* ***  Copy to global options *** */
  TWGlobal = TWLocal;

  /*  Set up length and width of pictures	*/
  i = NxPics[TWPicsPerPage];
  if(TWOptions[NUM_OPT_PICLEN].PO_Flags & PROPT_FLAG_ISSET)
    d->picLen = (1.0 / PICSTRETCH) * TWPicLen;
  else
    d->picLen = (PAGELEN - (i - 1) * SPACEX) / i;

  if(TWOptions[NUM_OPT_PICWIDTH].PO_Flags & PROPT_FLAG_ISSET) {
    d->picWidth = (1.0 / PICSTRETCH) * TWPicWidth;
  } else {
    double x;

    x = (TWPicsPerPage + i - 1.0) / i;
    d->picWidth = (PAGEWIDTH - (x - 1.0) * SPACEY) / x;
  }

  PicNr = 0;
  PageNr = 0;

  /*  TWG exists only in single precision  */
  d->mode &= ~OUTPUTDEV_DOUBLE;

  /*  Set up work space for TW Graph and unit for errors (stderr)  */
  dwsize(LWORK);
  smsgu(6);

  /*  Use interface capabilities if available  */
  if(TWLocal.Mode & (1UL << TWMODE_PSFONTS_N))
    sgsum(1,1,1,1);

  /*  Enable bounding box extension  */
  if(TWLocal.Mode & (1UL << TWMODE_USEBBEXT_N))
    sbbcalc(1);

  /*  Set output name if required  */
  if(name) {
    OutputName = name;
    spsfile(name);
  }

#ifdef XDEBUG
  /*  For debugging: Show routine calls  */
  smsg(200,1000,0);
#endif

  /*  Set verbose option for TW Graph routines  */
  if(TWLocal.Mode & (1UL << TWMODE_VERBOSE_N)) smsg(200,200,0);

  /*  Set up len and width of picture   */
  if((TWOptions[NUM_OPT_CONTOUR].PO_Flags & PROPT_FLAG_ISSET) &&
     !(TWOptions[NUM_OPT_PICLEN].PO_Flags & PROPT_FLAG_ISSET) &&
     !(TWOptions[NUM_OPT_PICWIDTH].PO_Flags & PROPT_FLAG_ISSET)) {
    /*   Area should be quadratic for contour plots  */
    dvsize(PICSTRETCH * MIN(d->picLen,d->picWidth),
	   PICSTRETCH * MIN(d->picLen,d->picWidth));
  } else {
    dvsize(PICSTRETCH * d->picLen,PICSTRETCH * d->picWidth);
  }

  /*  No frame around page (for binding in LATEX files)   */
  dpbord(0);

  twgSetOpts();

  /*  Set start position  */
  d->Next(OUTPUTDEV_NEXTNOINC | OUTPUTDEV_NEXTNEWPAGE);

  return(0);
}

/*JS*********************************************************************
*   TWGSETOPTS
*************************************************************************/

static int twgSetOpts(void)

/************************************************************************/
{
  /*  Set font if it was changed  */
  if(TWLocal.FontName[0]) {
    spsfn(TWLocal.FontName,0.0);
    TWLocal.FontName[0] = '\0';
  }

  /*  Disable verbosity  */
  if(Verbose == 0) setmsg(0);

  /*  Set axe label options  */
  sframe(TWMapAxeMode[(TWLocal.AxeMode >> (TWAFL_XOFFSET_N + TWAFL_SHIFT_B)) &
		     TWAFL_MASK],
	 TWMapAxeMode[(TWLocal.AxeMode >> (TWAFL_XOFFSET_N + TWAFL_SHIFT_T)) &
		     TWAFL_MASK],
	 TWMapAxeMode[(TWLocal.AxeMode >> (TWAFL_YOFFSET_N + TWAFL_SHIFT_B)) &
		     TWAFL_MASK],
	 TWMapAxeMode[(TWLocal.AxeMode >> (TWAFL_YOFFSET_N + TWAFL_SHIFT_T)) &
		     TWAFL_MASK]);

  sfram3(TWMapAxeMode[(TWLocal.AxeMode >> (TWAFL_XOFFSET_N + TWAFL_SHIFT_B)) &
		     TWAFL_MASK3D],
	 0, /*  for compatibility  */
	 TWMapAxeMode[(TWLocal.AxeMode >> (TWAFL_YOFFSET_N + TWAFL_SHIFT_B)) &
		     TWAFL_MASK3D],
	 0, /*  for compatibility  */
	 TWMapAxeMode[(TWLocal.AxeMode >> (TWAFL_ZOFFSET_N + TWAFL_SHIFT_B)) &
		     TWAFL_MASK3D],
	 /* Plot lines for other four positions
	  *  only if lines are requested.
	  */
	 (TWLocal.AxeMode >> TWAFL_ZOFFSET_N) & (TWAFL_LINE_BN|TWAFL_LINE_TN)
	 ? 1 : 0,
	 (TWLocal.AxeMode >> TWAFL_ZOFFSET_N) & (TWAFL_LINE_BN|TWAFL_LINE_TN)
	 ? 1 : 0,
	 (TWLocal.AxeMode >> TWAFL_ZOFFSET_N) & (TWAFL_LINE_BN|TWAFL_LINE_TN)
	 ? 1 : 0,
	 (TWLocal.AxeMode >> TWAFL_ZOFFSET_N) & (TWAFL_LINE_BN|TWAFL_LINE_TN)
	 ? 1 : 0,
	 0);

  /*  Set size for axis and characters  */
  salscl(TWLocal.TextScale);
  scscal(TWLocal.TextScale);

  /*  Show hidden lines?  */
  if(TWLocal.Mode & (1UL << TWMODE_HIDDENLINES_N)) {
    /*   Show all lines   */
    shdlr3(0);
  } else {
    /*   Hide lines  */
    shdlr3(3);
  }

  /*  Set line thickness and scale factor for point symbols  */
  slwscl(TWLocal.LineWidth);
  smscal(TWLocal.SymbolSize);

  if(TWLocal.Mode & (1UL << TWMODE_USECOLOR_N)) {
    int idType,iCapbl,iColMx;
    TWG_PRECISION uMax,vMax,sLWMin,sLWMax;

    qdtype(&idType,&iCapbl,&uMax,&vMax,&iColMx,&sLWMin,&sLWMax);
    MaxColors = iColMx;
  } else
    MaxColors = 0;

  return(0);
}

/*JS*********************************************************************
*   TWGDEINIT
*************************************************************************/

static char *twgDeInit(void)

/************************************************************************/
{
  endplt();

  return(OutputName ? OutputName : OUTPUTFILE);
}

/*JS*********************************************************************
*   TWGAREA
*************************************************************************/

static int twgArea(PlotType type,double xMin,double xMax,
		   double yMin,double yMax,double zMin,double zMax)

/************************************************************************/
{
  OutputDevice *d = &DeviceTWG;

  /*  Avoid multiple callings  */
  if(PlotFlag & 2) return(0);
  PlotFlag |= 2;

  XMin = xMin; XMax = xMax;
  YMin = yMin; YMax = yMax;
  ZMin = zMin; ZMax = zMax;

  switch(type) {
  case D3:
    szscal((TWLocal.AxeMode & (1 << (TWAFL_ZOFFSET_N + TWAFL_LOG_N))) ? 1 : 0,
	   zMin,zMax);
  case D2:
    sxscal((TWLocal.AxeMode & (1 << (TWAFL_XOFFSET_N + TWAFL_LOG_N))) ? 1 : 0,
	   xMin,xMax);
    syscal((TWLocal.AxeMode & (1 << (TWAFL_YOFFSET_N + TWAFL_LOG_N))) ? 1 : 0,
	   yMin,yMax);
    break;
  }

  return(0);
}

/*JS*********************************************************************
*   TWGNEXT
*************************************************************************/

static int twgNext(int what)

/************************************************************************/
{
  OutputDevice *d = &DeviceTWG;
static int PrintTitle = TRUE;

  /* ***  Reset options  *** */
  PlotFlag &= ~(2|4);
  TWLocal = TWGlobal;

  /*
   * Increment only if something has been plotted,
   *  otherwise we'll get some empty pages.
   */
  if(!(what & OUTPUTDEV_NEXTNOINC)) PicNr++;

  /*  Reset counter for subpictures  */
  Counter = 0;

  /*  Show name of file on every page  */
  if(PrintTitle && d->fileName && !(what & OUTPUTDEV_NEXTNOINC) &&
     (TWLocal.Mode & (1UL << TWMODE_PAGELABEL_N))) {
    char buffer[30];
    TWG_PRECISION csize;

    qcsize(&csize); /*  Ask current size  */

    scsize(0.3 * TWLocal.TextScale); /*  a little bit smaller    */

    sprintf(buffer,"%d: ",PageNr + 1);
    ptextg(buffer,9.,0.9);

    if(TWLocal.PageLabel[0]) {
      ptextg(TWLocal.PageLabel,999.,999.);
    } else {
static time_t Tim = -1;
      struct tm *tp;

      ptextg("File:",999.,999.);
      ptextg(d->fileName,999.,999.);

#if 0
      ptextg("File:",12,0.9);
      ptextg(d->fileName,13.5,0.9);
#endif

      /*  Print date and time  */
      if(Tim == -1) Tim = time(NULL);
      tp = localtime(&Tim);
      strftime(buffer,30,"    Time: %H:%M:%S %d/%m/%y",tp);
      ptextg(buffer,999.,999.);
    }

    PrintTitle = FALSE;

    scsize(csize); /*  Default size  */
  }

  if((what & OUTPUTDEV_NEXTNEWPAGE) || PicNr == TWPicsPerPage) {
    if(what & OUTPUTDEV_NEXTVERBOSE) printf("New Page.\n");

    /*	 Go to next page   */
    PicNrX  = 0;
    PicPosX = STARTX;
    PicPosY = STARTY - d->picWidth;

    if(PicNr) {
      /*   Go to new page   */
      newpag(0);
      PageNr++;

      PrintTitle = TRUE;
      PicNr = 0;  /*   Start with first picture    */
    }

    newpos(PicPosX + .5 * (1. - PICSTRETCH) * d->picLen,
	   PicPosY + .5 * (1. - PICSTRETCH) * d->picWidth);
  } else {
    PicNrX++;

    if((PicNrX % NxPics[TWPicsPerPage]) == 0) {
      /*   One line down   */
      PicPosX = STARTX;
      PicPosY -= d->picWidth + SPACEY;
      PicNrX = 0;
    } else {
      /*   One picture right	*/
      PicPosX += d->picLen + SPACEX;
    }

    newpos(PicPosX + .5 * (1. - PICSTRETCH) * d->picLen,
	   PicPosY + .5 * (1. - PICSTRETCH) * d->picWidth);
  }

  return(0);
}

/*JS*********************************************************************
*   TWGDRAW2D
*************************************************************************/

static int twgDraw2D(PlotMode mode,void *xValues,void *yValues,int nx)

/************************************************************************/
{
  OutputDevice *d = &DeviceTWG;
  TWG_PRECISION *xx;
  int irep;

  /*  Change options if they changed meanwhile  */
  twgSetOpts();

  if(xValues == NULL && mode != SPECTRUM) {
    double x;
    int i;

    /*  Set them up  */
    if((xx = (TWG_PRECISION *)malloc(nx * sizeof(*xx)))
       == NULL) {
      fprintf(stderr,"\
ERROR plot: No Memory for x Values available -- Break.\n");
      return(1);
    }

    for(x = (XMax - XMin) / (nx - 1) , i = 0 ; i < nx ; i++)
      xx[i] = XMin + i * x;
  } else
    xx = NULL;

  /*  Choose type of plotting  */
  if(mode == SPECTRUM) {
    /*  Plot data for spectra  */
    TWG_PRECISION x[2],y[2];
    int i;

    /*  Lines are drawn from start of x axe    */
    y[0] = 0.0;

    for(i = 0 ; i < nx ; i++) {
      /*  Set twice the same value for x to draw vertical line   */
      x[0] = x[1] = ((TWG_PRECISION *)xValues)[i];

      /*  Skip data outside plot interval  */
      if(x[0] < XMin || x[0] > XMax) continue;

      /*  Vertical line from 0 to value	 */
      y[1] = ((TWG_PRECISION *)yValues)[i];

      /*  Draw simple vertical line   */
      pdata(-4,2,x,y);
    }
  } else {
    /*  Draw legend if more than one line  */
    if((TWLocal.Mode & (1UL<<TWMODE_LEGEND_N)) &&
       Counter > 0 && Counter < MAXLINETYPES) {
      TWG_PRECISION x[2],y[2];
      char s[4];
      int i;

      x[0] = 1.05;
      x[1] = 1.13;
      for(i = (Counter == 1 ? 0 : Counter) ; i <= Counter ; i++) {
	int save,lType;

	if(TWLocal.LineTypes[0] > 0) {
	  qltype(&lType);

	  sltype(1 + (TWLocal.LineTypes[1 + (i % TWLocal.LineTypes[0])] - 1)
		      % MAXLINETYPES);
	}

	if(MaxColors > 0) {
	  qcolor(&save);
	  scolor(1 + (i % MaxColors));
	}

	y[0] = y[1] = 0.9 - 0.08 * i;
	pdata(4,-2,x,y);

	if(MaxColors > 0) scolor(save);

	sprintf(s,"%d:",1 + i);
	ptext(s,1.01,y[0] - 0.018);

	if(TWLocal.LineTypes[0] > 0) sltype(lType);
      }
    }

    if(!(TWLocal.Mode & (1UL << TWMODE_PLOTLINES_N)))
      irep = 0;
    else {
      switch(mode) {
      case XYCURVE:  irep = -4;  break;
      default:       irep =  4;  break;
      }
    }

    {
      int save,lType;

      /*  Set line type for this plot  */
      if(TWLocal.LineTypes[0] > 0) {
	qltype(&lType);
	sltype(1 + (TWLocal.LineTypes[1 + (Counter % TWLocal.LineTypes[0])]-1)
		    % MAXLINETYPES);
      }

      if(MaxColors > 0) {
	qcolor(&save);
	scolor(1 + (Counter % MaxColors));
      }
      Counter++;

      /* Plot data  */
      if(xValues) { /*  with pre given x values  */
	pdata(irep,nx,xValues,yValues);
      } else { /*  with calculated x values  */
	pdata(irep,nx,xx,yValues);
      }

      if(MaxColors > 0) scolor(save);
      if(TWLocal.LineTypes[0] > 0) sltype(lType);
    }

    /*   Free memory   */
    if(xx) free(xx);
  }

  return(0);
}

/*JS*********************************************************************
*   TWGDRAW3D
*************************************************************************/

static int twgDraw3D(PlotMode mode,void *xValues,void *yValues,
		     void *data,int nx,int ldx,int ny)

/************************************************************************/
{
  OutputDevice *d = &DeviceTWG;

  /*  Change options if they changed meanwhile  */
  twgSetOpts();

  /*  Since TW Graph cannot handle mixing of axe data and non-axe data,
   *   we ignore it if one is not set.
   */
  if(xValues == NULL || yValues == NULL) {
    nx = -nx;
    ny = -ny;
  }

  if(mode == CONTOUR || mode == SURFCONTOUR ||
     (mode == PLAIN &&
      (TWOptions[NUM_OPT_CONTOUR].PO_Flags & PROPT_FLAG_ISSET))) {
    TWG_PRECISION levels[MAXLEVELS],dist;
    int i,nullLine;
    double zPos;

    /* ***  Contour plot  *** */
    /*	Set up values for contours  */
    nullLine = ZMin >= 0.0 ? 0 : TWLocal.NoOfLevels;
    dist = (ZMax - ZMin) / TWLocal.NoOfLevels;
    for(i = 0 ; i < TWLocal.NoOfLevels ; i++) {
      levels[i] = ZMin + i * dist;
      if(i > 0 && levels[i - 1] < 0.0 && levels[i] >= 0.0) nullLine = i;
    }

    /*   Set up labels for contour lines	   */
    if((TWLocal.Mode  & (1UL<<TWMODE_CONTDESCR_N)) && (mode != SURFCONTOUR)) {
      /*  If (max - min) > 0.1 then use e            */
      /*   representation of floating point numbers  */
      sclabl(MAX(TWLocal.NoOfLevels / 10,1),1,(ABS(ZMax-ZMin) > 0.1) ?
	     4 : -3);
    } else {
      sclabl(0,1,0);
    }

    if(mode == SURFCONTOUR) {
      /*  Set up rotating angle for surface plot  */
      sview3(60.0,TWLocal.SurfRotate,1.0,0.5);

      if(TWLocal.ContHeight > 0.0)
	zPos = ZMax + TWLocal.ContHeight * (ZMax - ZMin);
      else
	zPos = ZMin + TWLocal.ContHeight * (ZMax - ZMin);
    }

    /* ***  Plot positive (or all) contour lines  *** */
    if(mode == SURFCONTOUR) {
      if(nullLine < TWLocal.NoOfLevels) {
	/*   Plot contour lines    */
	pdata3c(13,nx,(TWG_PRECISION *)xValues,ny,(TWG_PRECISION *)yValues,
		ldx,(TWG_PRECISION *)data,TWLocal.NoOfLevels - nullLine,
		&levels[nullLine],zPos);
      } else {
	/*  Only plot surface  */
	pdata3(13,nx,(TWG_PRECISION *)xValues,ny,(TWG_PRECISION *)yValues,
	       ldx,(TWG_PRECISION *)data);
      }
    } else

    if(nullLine < TWLocal.NoOfLevels) {
      /*   Plot contour lines    */
      pdatac(nx,(TWG_PRECISION *)xValues,ny,(TWG_PRECISION *)yValues,
	     ldx,(TWG_PRECISION *)data,TWLocal.NoOfLevels - nullLine,
	     &levels[nullLine]);
    }

    /* ***  Plot negative contour lines  *** */
    if(nullLine > 0) {
      int lType;

      /*  Different line types for negative contour lines  */
      qltype(&lType);
      sltype(4);

      if(mode == SURFCONTOUR) {
	/*   Plot contour lines    */
	pdata3c(0,nx,(TWG_PRECISION *)xValues,ny,(TWG_PRECISION *)yValues,
		ldx,(TWG_PRECISION *)data,nullLine,levels,zPos);
      } else {
	/*   Plot contour lines    */
	pdatac(nx,(TWG_PRECISION *)xValues,ny,(TWG_PRECISION *)yValues,
	       ldx,(TWG_PRECISION *)data,nullLine,levels);
      }

      sltype(lType);
    }
  } else {
    /* ***  Surface plot  *** */
    /*  Set up rotating angle  */
    sview3(60.0,TWLocal.SurfRotate,1.0,0.5);

    /*  irep = 13: Lines in both directions  */
    pdata3(13,nx,(TWG_PRECISION *)xValues,ny,
	   (TWG_PRECISION *)yValues,ldx,(TWG_PRECISION *) data);
  }

  return(0);
}

/*JS*********************************************************************
*   TWGDRAWLEGEND
*************************************************************************/

static int twgDrawLegend(PlotType type,PlotMode mode,char *title,
			 char *xAxis,char *yAxis,char *zAxis)

/************************************************************************/
{
  OutputDevice *d = &DeviceTWG;

  /*  Change options if they changed meanwhile  */
  twgSetOpts();

  /*  Avoid multiple callings  */
  if(PlotFlag & 4) return(0);
  PlotFlag |= 4;

  /* ***  Show text for axes  *** */
  switch(type) {
  case D2:
    {
      TWG_PRECISION scalec;

      qcscal(&scalec); /*  Ask current size  */

      scscal(0.8 * TWLocal.TextScale); /*  a little bit smaller    */

      if((TWLocal.AxeMode & (1UL << (TWAFL_XOFFSET_N + TWAFL_LABEL_N))) &&
	 xAxis)
	pxtext(xAxis);

      if((TWLocal.AxeMode & (1UL << (TWAFL_YOFFSET_N + TWAFL_LABEL_N))) &&
	 yAxis)
	pytext(yAxis);

      scscal(scalec); /*  Set back default size  */
    }
    break;
  case D3:
    if(mode == CONTOUR ||
       (mode == PLAIN &&
	(TWOptions[NUM_OPT_CONTOUR].PO_Flags & PROPT_FLAG_ISSET))) {
      TWG_PRECISION scalec;

      qcscal(&scalec); /*  Ask current size  */

      /*   Text for axes     */
      scscal(0.8 * TWLocal.TextScale); /*  a little bit smaller    */
      if((TWLocal.AxeMode & (1UL << (TWAFL_XOFFSET_N + TWAFL_LABEL_N))) &&
	 xAxis)
	pxtext(xAxis);

      if((TWLocal.AxeMode & (1UL << (TWAFL_YOFFSET_N + TWAFL_LABEL_N))) &&
	 yAxis)
	pytext(yAxis);

      /*    Default size    */
      scscal(scalec);
    } else {
      double s,c;
      int rota = (int)(TWLocal.SurfRotate / 90.0) % 4;
      TWG_PRECISION scalec;

#define ROTX(x,y)  (0.5 + c * (x) + s * (y))
#define ROTY(x,y)  (0.5 - s * (x) + c * (y))

      c = cos((PI / 180.0) * (TWLocal.SurfRotate - 30.0));
      s = sin((PI / 180.0) * (TWLocal.SurfRotate - 30.0));

      qcscal(&scalec); /*  Ask current size  */

      /*  Plot text for axes  */
      scscal(1.5 * TWLocal.TextScale); /*  a little bigger   */

      if((TWLocal.AxeMode & (1UL << (TWAFL_XOFFSET_N + TWAFL_LABEL_N))) &&
	 xAxis) {
	stdir3(((rota + 1) & 2) ? -1.0 : 1.0,0.0,0.0,
	       0.0,((rota + 1) & 2) ? -1.0 : 1.0,0.0);
	ptext3(xAxis,0.5,((rota + 1) & 2) ? 1.2 : -0.2,0.0);
      }

      if((TWLocal.AxeMode & (1UL << (TWAFL_YOFFSET_N + TWAFL_LABEL_N))) &&
	 yAxis) {
	stdir3(0.0,(rota & 2) ? 1.0 : -1.0,0.0,
	       (rota & 2) ? -1.0 : 1.0,0.0,0.0);
	ptext3(yAxis,(rota & 2) ? 1.2 : -0.2,0.5,0.0);
      }

      if((TWLocal.AxeMode & (1UL << (TWAFL_ZOFFSET_N + TWAFL_LABEL_N))) &&
	 zAxis) {
	stdir3(c,-s,0.0, 0.0,0.0,1.0);
	ptext3(zAxis,ROTX(-0.5,0.5),ROTY(-0.5,0.5),1.3);
      }

      /*    Default size   */
      scscal(scalec);
    }

    break;
  }

  /* ***  Show title  *** */
  if(title && (TWLocal.Mode & (1UL << TWMODE_TITLE_N))) {
    char *title2;
    TWG_PRECISION ypos,scalec;
    char *string;

    /*	  Now print out Text	  */
    title2 = strdup(title);

    qcscal(&scalec); /*  Ask current size  */

    /*	 Characters not too big    */
    scscal(0.8 * TWLocal.TextScale);

    for(ypos = 1.1 , string = strtok(title2,"\n") ; string ;
	string = strtok(NULL,"\n") ) {
      ptitle(string,ypos);
      ypos -= 0.05;
    }

    /*	 Set default	*/
    scscal(scalec);
    free(title2);
  }

  return(0);
}
#endif
