/*JS*********************************************************************
*
*    Program : PLOT
*    Language: ANSI-C
*    Author  : Joerg Schoen
*    Purpose : Plots data
*
*    Part    : Device independent plot routines and auxiliary routines.
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id$";
#endif

/*********     INCLUDES 					*********/
#include "plot.h"

/*********     DEFINES						*********/

/*********     PROTOTYPES					*********/
Prototype int            plotPicture(FPPicture *picture,PlotMode mode,int part,
				     const BoundPair *bounds,int nBounds,
				     const BoundPair *scales,int nScales);

static FPPicture        *makeCopy(FPPicture *picture,int part,
				  const BoundPair *bounds,int nBounds,
				  const BoundPair *scales,int nScales,
				  double **pLimits,int mode);

static int               rescalePict(FPPicture *picture,
				     const BoundPair *scales,int nScales);
static double           *getLimits(FPPicture *picture,int part);
static void              correctValue(double *val,const BoundValue bVal);

#if 0
static BoundPair         correctBoundPair(BoundPair bp,double lo,double hi,
					  BoundPair scale);
#endif

Prototype void		 showpicture(FPPicture *picture);

Prototype int		 writePicture(FPPicture *picture,int mode,int part,
				      const BoundPair *bounds,int nBounds,
				      const BoundPair *scales,int nScales,
				      FPFile *fpOut);

/*JS*********************************************************************
*   PLOTPICTURE Plots the picture using the current device. Returns != 0
*    on error.
*************************************************************************/

int plotPicture(FPPicture *picture,PlotMode mode,int part,
		const BoundPair *bounds,int nBounds,
		const BoundPair *scales,int nScales)

/************************************************************************/
{
  FPPicture *picCopy;
  double *limits;
  FPData *picData;
  int type,flag;

  limits = NULL; /*  for error case  */

  /*  Make a local copy of the parts we want to plot  */
  if((picCopy = makeCopy(picture,part,bounds,nBounds,scales,nScales,
			 &limits,FPLOT_MODE_GENAXEDATA |
			 ((DEVICE->mode & OUTPUTDEV_DOUBLE) ?
			  FPLOT_MODE_CASTTODOUBLE : FPLOT_MODE_CASTTOFLOAT)))
     == NULL)
    goto error;

  /*  Set flag for plotted pictures   */
  flag = 0;

  switch(picCopy->FP_Content.FPD_NDim) {
  case 1:
    /*  Set up boundaries  */
    DEVICE->Area(D2,limits[0],limits[1],limits[2],limits[3],0.0,0.0);

    /*	Now scan through all pictures	*/
    for(picData = &picCopy->FP_Content ; picData ;
	picData = picData->FPD_Next) {
      switch(picData->FPD_Type) {
      case 0:
      case FPLOT_TYPE_XYCURVE:
	{
	  void *px,*pd;

	  if(picData->FPD_AxeValues && picData->FPD_AxeValues[0]) {
	    if(picData->FPD_Flags & FPLOT_FLAGS_SINGLE)
	      px = (float *)(picData->FPD_AxeValues[0]);
	    else
	      px = (double *)(picData->FPD_AxeValues[0]);
	  } else
	    px = NULL;

	  if(picData->FPD_Flags & FPLOT_FLAGS_SINGLE)
	    pd = (float *)picData->FPD_Data;
	  else
	    pd = (double *)picData->FPD_Data;

	  DEVICE->Draw2D((picData->FPD_Type == FPLOT_TYPE_XYCURVE) ?
			 XYCURVE : PLAIN,px,pd,picData->FPD_Dims[0]);
	}

	flag = 1; /*  set flag	 */
	break;
      case FPLOT_TYPE_SPECTRUM:
	DEVICE->Draw2D(SPECTRUM,picData->FPD_AxeValues ?
		       picData->FPD_AxeValues[0] : NULL,
		       picData->FPD_Data,picData->FPD_Dims[0]);
	flag = 1; /*	set flag   */
	break;
      default:
	fprintf(stderr,"ERROR plot: Unknown picture subtype %d -- Skipped.\n",
		picData->FPD_Type);
	break;
      }
    }

    DEVICE->DrawLegend(D2,mode,picCopy->FP_Title,
    		       picCopy->FP_NAxes > 0 ? picCopy->FP_Axes[0] : NULL,
		       picCopy->FP_NAxes > 1 ? picCopy->FP_Axes[1] : NULL,
		       NULL);
    break;
  case 2:
    DEVICE->Area(D3,limits[0],limits[1],limits[2],limits[3],
		 limits[4],limits[5]);

    /*	Get subtype (must be equal for all subpictures)   */
    type = picCopy->FP_Content.FPD_Type;

    /*	Now scan through all pictures	*/
    for(picData = &picCopy->FP_Content ; picData ;
	picData = picData->FPD_Next) {
      if(type != picData->FPD_Type) {
	fprintf(stderr,"\
ERROR plot: Incompatible picture types -- Subpicture ignored.\n");
	continue;
      }

      switch(type) {
      case 0:
	{
	  void *px,*py,*pd;

	  if(picData->FPD_AxeValues) {
	    if(picData->FPD_Flags & FPLOT_FLAGS_SINGLE) {
	      px = (float *)(picData->FPD_AxeValues[0]);
	      py = (float *)(picData->FPD_AxeValues[1]);
	    } else {
	      px = (double *)(picData->FPD_AxeValues[0]);
	      py = (double *)(picData->FPD_AxeValues[1]);
	    }
	  } else {
	    px = py = NULL;
	  }

	  if(picData->FPD_Flags & FPLOT_FLAGS_SINGLE)
	    pd = (float *)picData->FPD_Data;
	  else
	    pd = (double *)picData->FPD_Data;

	  DEVICE->Draw3D(mode,px,py,pd,picData->FPD_Dims[0],
			 picData->FPD_Dims[0],picData->FPD_Dims[1]);
	}

	DEVICE->DrawLegend(D3,mode,picCopy->FP_Title,
			   picCopy->FP_NAxes > 0 ? picCopy->FP_Axes[0] : NULL,
			   picCopy->FP_NAxes > 1 ? picCopy->FP_Axes[1] : NULL,
			   NULL);

	flag = 1; /*  set flag	 */
	break;
      default:
	fprintf(stderr,"ERROR plot: Unknown picture sub type %d!\n",type);
	break;
      }
    }

    break;
  default:
    fprintf(stderr,"ERROR plot: Cannot plot dimension %d -- Break\n",
	    picCopy->FP_Content.FPD_NDim);
    goto error;
  }

  if(picCopy != picture) fplotFreepic(picCopy);
  if(limits)  free(limits);

  if(!flag) fprintf(stderr,"ERROR plot: No data plotted!\n");

  return(0);
error:
  return(-1);
}

/*JS*********************************************************************
*   MAKECOPY
*************************************************************************/

static FPPicture *makeCopy(FPPicture *picture,int part,
			   const BoundPair *bounds,int nBounds,
			   const BoundPair *scales,int nScales,
			   double **pLimits,int mode)

/************************************************************************/
{
  FPPicture *picCopy1,*picCopy2;
  int i,nDim;

  if(nScales == 0) {
    /*  Default: use global scaling  */
    scales  = Scales;
    nScales = NScales;
  }

  /*  Get original dimension of problem  */
  nDim = picture->FP_Content.FPD_NDim;

  /*  Copying necessary?  */
  if(nScales > 0) {
    /*  Make a 1-1 copy and rescale copy  */
    if((picCopy1 = fplotCopy(picture,part,NULL,0,mode)) == NULL)
      goto error;

    /*  Delete task that have already been done  */
    part = 0;
    mode &= ~(FPLOT_MODE_CASTTODOUBLE|FPLOT_MODE_CASTTOFLOAT);

    if(rescalePict(picCopy1,scales,nScales)) goto error;

    /*  Resort in case a scale was negative  */
    if(fplotSortData(picCopy1)) goto error;
  } else
    picCopy1 = picture;

  if(part != 0 || nBounds != 0 ||
     (mode & (FPLOT_MODE_CASTTODOUBLE|FPLOT_MODE_CASTTOFLOAT))) {
    double *limits;

    if(nBounds) {
      /*  Get boundaries from picture parts we want to plot  */
      if((limits = getLimits(picCopy1,part)) == NULL) goto error;

      /*  Now correct pregiven values with parameters  */
      /*  Limits for data values are not used here  */
      for(i = 0 ; i < MIN(nBounds,nDim) ; i++) {
	if(bounds[i].BP_Type == BP_INTERVALL) {
	  correctValue(&limits[2*i],bounds[i].BP_Min);
	  correctValue(&limits[2*i+1],bounds[i].BP_Max);

	  /*  Still nothing set?  */
	  if(limits[2*i] == limits[2*i+1]) {
	    printf("\
WARNING: No interval given in dimension %d -- Set to [0,1].\n",
		   i);
	    limits[2*i]   = 0.0;
	    limits[2*i+1] = 1.0;
	  }
	} else {
	  double off;

	  correctValue(&limits[2*i],bounds[i].BP_Min);

	  /*  Set end value to something LOWER than start value  */
	  if((off = fabs(limits[2*i])) == 0.0) off = 1.0;
	  limits[2*i+1] = limits[2*i] - off;
	}
      }

#if defined(DEBUG)
      for(i = 0 ; i < nDim ; i++)
	printf("%d: %g - %g\n",i,limits[2*i],limits[2*i+1]);
#endif
    } else {
      limits = NULL; /*  nothing to cut  */
    }

    /*  Now copy the picture  */
    if((picCopy2 = fplotCopy(picCopy1,part,limits,nBounds,mode)) == NULL)
      goto error;

    if(picCopy1 != picture) fplotFreepic(picCopy1);
    if(limits) free(limits);

    picture = picCopy2;
  } else
    picture = picCopy1;

  /*  Wanna have limits?  */
  if(pLimits) {
    double *limits;
    int j;
    int newNDim = picture->FP_Content.FPD_NDim;

    if((limits = getLimits(picture,0)) == NULL) goto error;

    /*  Correct limits finally to allow wider intervalls
     *   to be shown than data values.
     */
    for(i = j = 0 ; i < nBounds ; i++) {
      /*  Skip removed dimensions  */
      if(bounds[i].BP_Type != BP_INTERVALL) continue;

      correctValue(&limits[2 * j],bounds[i].BP_Min);
      correctValue(&limits[2 * j + 1],bounds[i].BP_Max);

      /*  j runs from 0 to newNDim  */
      if(j++ >= newNDim) break;
    }

    *pLimits = limits;
  }

  return(picture);
error:
  return(NULL);
}

/*JS*********************************************************************
*   RESCALEPICT Rescale data.
*************************************************************************/

static int rescalePict(FPPicture *picture,const BoundPair *scales,int nScales)

/************************************************************************/
{
  double *s;
  int i;

  if(nScales == 0) return(0);

  if((s = (double *)malloc(2 * nScales * sizeof(*s))) == NULL) return(-1);

  for(i = 0 ; i < nScales ; i++) {
    if(scales[i].BP_Type == BP_INTERVALL) {
      if(scales[i].BP_Min.BV_Attr != BV_NSET) {
	s[2*i] = scales[i].BP_Min.BV_Value;

	s[2*i+1] = (scales[i].BP_Max.BV_Attr == BV_NSET) ?
	  HUGE_VAL : scales[i].BP_Max.BV_Value;
      } else if(scales[i].BP_Max.BV_Attr != BV_NSET) {
	s[2*i] = HUGE_VAL;

	s[2*i+1] = scales[i].BP_Max.BV_Value;
      } else {
	/*  Both not set, no scaling  */
	s[2*i] = s[2*i+1] = 1.0;
      }
    } else {
      s[2*i] = s[2*i+1] = (scales[i].BP_Min.BV_Attr != BV_NSET) ?
	scales[i].BP_Min.BV_Value : 1.0;
    }
    /*printf("%d: sc1 %g sc2 %g\n",i,s[2*i],s[2*i+1]);*/
  }

  if(fplotScaleData(picture,s,nScales)) return(-1);

  free(s);

  return(0);
}

/*JS*********************************************************************
*   GETLIMITS Get interval limits from all picture parts we want to plot.
*************************************************************************/

static double *getLimits(FPPicture *picture,int part)

/************************************************************************/
{
  double *limits,*preVals;
  FPData *picData;
  int i,nDim;

  /*  Get dimension of problem  */
  nDim = picture->FP_Content.FPD_NDim;

  if((limits = (double *)malloc(2 * (nDim + 1) * sizeof(*limits))) == NULL ||
     (preVals = (double *)malloc(2 * nDim * sizeof(*preVals))) == NULL)
    goto error;

  /*  Preset all values  */
  for(i = 0 ; i <= nDim ; i++)
    limits[2 * i] = limits[2 * i + 1] = 0.0;

  /*   Now scan through all parts to get our values    */
  for(picData = &picture->FP_Content ; picData ;
      picData = picData->FPD_Next, part >>= 1) {
    /*  Look at this part?  */
    if(part & 1) continue;

    /*  Cast values from data to double  */
    if(picData->FPD_Flags & FPLOT_FLAGS_SINGLE) {
      for(i = 0 ; i < picData->FPD_NDim ; i++) {
	preVals[2*i]   = ((float *)picData->FPD_Boundaries)[2*i];
	preVals[2*i+1] = ((float *)picData->FPD_Boundaries)[2*i+1];
      }
    } else {
      for(i = 0 ; i < picData->FPD_NDim ; i++) {
	preVals[2*i]   = ((double *)picData->FPD_Boundaries)[2*i];
	preVals[2*i+1] = ((double *)picData->FPD_Boundaries)[2*i+1];
      }
    }

    /*  Correct boundaries of every dimension  */
    for(i = 0 ; i < picData->FPD_NDim ; i++)
      if(limits[2*i] != limits[2*i+1]) {
	/*  Lower minimum and increase maximum	 */
	if(preVals[2*i]   < limits[2*i])   limits[2*i]   = preVals[2*i];
	if(preVals[2*i+1] > limits[2*i+1]) limits[2*i+1] = preVals[2*i+1];
      } else {
	/*  Just copy values  */
	limits[2*i]   = preVals[2*i];
	limits[2*i+1] = preVals[2*i+1];
      }

    /*  Correct boundaries of function values   */
    if(limits[2*nDim] != limits[2*nDim+1]) {
      /*  Lower minimum and increase maximum	 */
      if(picData->FPD_FMin < limits[2*nDim])
	limits[2*nDim] = picData->FPD_FMin;
      if(picData->FPD_FMax > limits[2*nDim+1])
	limits[2*nDim+1] = picData->FPD_FMax;
    } else {
      /*  Just copy the values    */
      limits[2*nDim]   = picData->FPD_FMin;
      limits[2*nDim+1] = picData->FPD_FMax;
    }
  }

  if(Verbose & 128) {
    printf("Intervals:");
    for(i = 0 ; i <= nDim ; i++)
      printf(" %c [%g,%g]",i == nDim ? '-' : 'x',limits[2*i],limits[2*i+1]);
    printf("\n");
  }

  free(preVals);

  return(limits);
error:
  return(NULL);
}

/*JS*********************************************************************
*   CORRECTVALUE
*************************************************************************/

static void correctValue(double *val,const BoundValue bVal)

/************************************************************************/
{
  switch(bVal.BV_Attr) {
  case BV_NSET: /*  Don't set  */
    break;
  case BV_EQUAL: /*  Use given value  */
    *val = bVal.BV_Value;
    break;
  case BV_LLIM:
    *val = MAX(bVal.BV_Value,*val);
    break;
  case BV_ULIM:
    *val = MIN(bVal.BV_Value,*val);
  case BV_FACTOR:
    *val *= bVal.BV_Value;
    break;
  }
}

#if 0
/*JS*********************************************************************
*   CORRECTBOUNDPAIR
*************************************************************************/

static BoundPair correctBoundPair(BoundPair bp,double lo,double hi,
				  BoundPair scale)

/************************************************************************/
{
  double m,b;

  if(scale.BP_Type == BP_POINT) {
    m = scale.BP_Min.BV_Value;
    b = 0.0;
  } else if(scale.BP_Min.BV_Attr == BV_NSET) {
    if(scale.BP_Max.BV_Attr == BV_NSET) /*  nothing to be done  */
      return(bp);

    m = scale.BP_Max.BV_Value / hi;
    b = 0.0;
  } else if(scale.BP_Max.BV_Attr == BV_NSET) {
    m = scale.BP_Min.BV_Value / lo;
    b = 0.0;
  } else {
    m = (scale.BP_Max.BV_Value - scale.BP_Min.BV_Value) / (hi - lo);
    b = (hi * scale.BP_Min.BV_Value - lo * scale.BP_Max.BV_Value) / (hi - lo);
  }

  /*printf("m %g b %g\n",m,b);*/

  /*  Use inverse linear transformation  */
  if(bp.BP_Min.BV_Attr != BV_NSET)
    bp.BP_Min.BV_Value = (bp.BP_Min.BV_Value - b) / m;

  if(bp.BP_Max.BV_Attr != BV_NSET) {
     bp.BP_Max.BV_Value = (bp.BP_Max.BV_Value - b) / m;

     /*  Exchange if negative scaling to restore order  */
     if(bp.BP_Min.BV_Attr != BV_NSET &&
	bp.BP_Min.BV_Value > bp.BP_Max.BV_Value) {
       BoundValue bp2;

       bp2 = bp.BP_Min;
       bp.BP_Min = bp.BP_Max;
       bp.BP_Max = bp2;
     }
  }

  return(bp);
}
#endif

/*JS*********************************************************************
*   SHOWPICTURE
*************************************************************************/

void showpicture(FPPicture *picture)

/************************************************************************/
{
  int i;
  char *string;

  /*   Print first line of picture title    */
  if(picture->FP_Title) {
    if( (string = strchr(picture->FP_Title,'\n')) )
      i = string - picture->FP_Title;
    else
      i = strlen(picture->FP_Title);

    if(Verbose)
      printf("Picture %d: \"%.*s\"\n",CurrentPicture,i,picture->FP_Title);
  } else {
    if(Verbose)
      printf("Picture %d: <NO TITLE>!\n",CurrentPicture);
  }

  return;
}

/*JS*********************************************************************
*   WRITEPICTURE Writes data to file
*************************************************************************/

int writePicture(FPPicture *picture,int mode,int part,
		 const BoundPair *bounds,int nBounds,
		 const BoundPair *scales,int nScales,FPFile *fpOut)

/************************************************************************/
{
  FPPicture *picCopy;
  FPData *picData;
  int i;

  /*  Make a local copy of the parts we want to write  */
  if((picCopy = makeCopy(picture,part,bounds,nBounds,scales,nScales,NULL,0))
     == NULL) goto error;

  /*  Now scan through all sub pictures  */
  for(picData = &picCopy->FP_Content ; picData ; picData = picData->FPD_Next) {
#ifdef OLDOLD /*  Separate code for NDim == 1 is no longer needed  */
    if(picData->FPD_NDim == 1) {
      if((picData->FPD_Flags & FPLOT_FLAGS_SINGLE))
	i = fplotSVector(fpOut,picData->FPD_Type,picData->FPD_AxeValues ?
			 (float *)picData->FPD_AxeValues[0] : NULL,
			 (float *)picData->FPD_Data,
			 picData->FPD_Dims[0],1,
			 ((float *)picData->FPD_Boundaries)[0],
			 ((float *)picData->FPD_Boundaries)[1],
			 1.0,1.0);
      else
	i = fplotVector(fpOut,picData->FPD_Type,picData->FPD_AxeValues ?
			(double *)picData->FPD_AxeValues[0] : NULL,
			(double *)picData->FPD_Data,
			picData->FPD_Dims[0],1,
			((double *)picData->FPD_Boundaries)[0],
			((double *)picData->FPD_Boundaries)[1],
			1.0,1.0);
    } else
#endif
    if((picData->FPD_Flags & FPLOT_FLAGS_SINGLE)) {
      i = fplotSTensor(fpOut,picData->FPD_Type,picData->FPD_NDim,
		       (const float**)picData->FPD_AxeValues,
		       (float *)picData->FPD_Data,picData->FPD_Dims,
		       NULL,picData->FPD_Boundaries,1.0,1.0);
    } else {
      i = fplotTensor(fpOut,picData->FPD_Type,picData->FPD_NDim,
		      (const double**)picData->FPD_AxeValues,
		      (double *)picData->FPD_Data,picData->FPD_Dims,
		      NULL,picData->FPD_Boundaries,1.0,1.0);
    }

    if(i) goto error;
  }

  if(picCopy != picture) fplotFreepic(picCopy);

  return(0);
error:
  return(1);
}
