/*JS*********************************************************************
*
*    Program : PLOT
*    Language: ANSI-C
*    Author  : Joerg Schoen
*    Purpose : Plots data
*
*    Part    : Main routines.
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id: plot.c,v 4.1 1995/10/30 09:34:47 joerg Stab joerg $";
#endif

/*********     INCLUDES 					*********/
#include <errno.h>
#include "plot.h"

/*********     DEFINES						*********/
#define OPTSTRING "h;i;o:p:v;qsS:D:d:"

/*  Environment variable for plot   */
#define ENVOPT    "PLOTOPTIONS"
#define ENVPATH   "PLOTPATH"

#define MODE_FPLOTREAD   FPLOT_MODE_SORTDATA

/*********     PROTOTYPES					*********/
static void		 help(int mode);
Prototype FPFile	*fplotOpenExt(char *fileName,int mode);

Prototype int		 Verbose,RemainingFiles,CurrentPicture;
Prototype BoundPair     *Scales;
Prototype int            NScales;
Prototype FPFile	*PlotFile,*PlotOutFile;
Prototype int            PlotOutState;

/*********     GLOBAL VARIABLES 				*********/
int   Verbose,RemainingFiles,CurrentPicture;
BoundPair *Scales;
int NScales;
FPFile *PlotFile,*PlotOutFile;
int PlotOutState;

static char  *FileName;

/*
 * Explanation of Verbose mode:
 * Bit 0: Show some small information about current doing.
 * Bit 1: Show title of plotted pictures.
 * Bit 2: Show found hunks.
 * Bit 3: Show text hunks.
 * Bit 4: Show vectors and arrays too.
 * Bit 5: Show position information.
 * Bit 6: Show plotted area and related information.
 */

/*JS*********************************************************************
*   HELP
*************************************************************************/

static void help(int mode)

/************************************************************************/
{
  if(mode != 2) {
    int i;

    printf("\
PLOT V"VERSION"                                      Joerg Schoen 1993-96\n\
Usage: plot [<opt>] <file1> ... <filen>\n\
    Plots the specially formatted data in the files <file1> .. <filen>.\n\
    If only one file is given, the output file (PostScript) is renamed\n\
    accordingly (for example \"plot test%s\" plots to file \"test%s\").\n\
    For further description of data format in \".out\" files see\n\
    documentation for library \"jssubs\".\n\
    Options are:\n\
	-h[<n>]	Show this information, '-h2' shows current device's options\n\
                and their default settings.\n\
	-v[<n>] Run Verbose with n:\n\
                Bit 10: Show position information.\n\
                Bit 11: Show found hunks.\n\
                Bit 12: Show text hunks.\n\
                Bit 13: Show vectors and arrays too.\n\
	-q	Run quietly. This is the default when output goes not to a\n\
                terminal. To get output in that case, use '-v1'.\n\
	-o<name> Rename output file to <name>.\n\
	-p<a>[,<e>] Plots only pictures from a to e.\n\
	-i[<commands>]	Interactive ask user for every picture. Starts\n\
			with commands if given.\n\
	-s	Show only contents, don't plot.\n\
        -S<xsc>,<ysc>,...  Set global scale values for x, y, ... axes.\n\
        -D<opts> Pass options to output device. Multiple options may be\n\
                 separated with ',', use ';' to mark end of suboptions list.\n\
                 Boolean options may be negated with prefix \"no\". To show\n\
                 available options, use '-h2'\n\
        -d<dev>  Set output device to one of",INPUTEXT,PSEXT);

    for(i = 0 ; i < NOutputDevices ; i++)
      printf(" \"%s\"",OutDev[i]->name);
    fputs(".\n",stdout);
  } else {
    printf("#  Options for device \"%s\" (only the defaults are shown):\n\n",
	   DEVICE->name);
    showProgOpts(stdout,DEVICE->devOpts,DEVICE->optHelp,"\n");
    putchar('\n');
  }

  return;
}

/*JS*********************************************************************
*   MAIN
*************************************************************************/

int main(int argc, char ** argv)

/************************************************************************/
{
  short plotflag,interactive;
  char	*output,*plotword,*string,*string2;
  char	*commandString;
  int i,startpic,endpic,flag;
  FPPicture *picture;

  /* *********	Set Defaults   ******* */
  plotflag = 1;  plotword = "Plotting";

  Verbose =
#ifndef CONFIG_NO_POSIX
    /*  If not writing to a terminal, suppress output  */
    !isatty(fileno(stdout)) ? 0 :
#endif
    1;
  interactive = 0;
  output = NULL;
  startpic = 1;
  endpic   = 4711; /*  This should be enough for all needs  */

  commandString = NULL;

  /*	Clear Picture	  */
  picture = NULL;

  /*  Get search path for input files from environment variable  */
  if((FPlotOpenPath = getenv(ENVPATH)) == NULL) {
    FPlotOpenPath = getenv(JSENVPATH);
    errno = 0;
  }

  /* *************************************************** */
  /* *******	Parse options (start with '-')    ****** */
  /* *************************************************** */
  for(flag = 2 ; flag >= 0 ; flag--) {
    char *env;

    if(flag == 2) {
      if((env = getenv(JSENVOPT)) == NULL) continue;
    } else if(flag == 1) {
      if((env = getenv(ENVOPT)) == NULL) continue;
    } else
      env = NULL;

    while((i = jsGetOpt(env ? NULL : &argc,env ? (void *) env :
			(void *)argv,OPTSTRING,&string)) != EOF) {
      switch(i) {
      default:
	goto ende;
      case 'h':
	if(string == NULL)
	  help(1);
	else
	  help(atoi(string));
	goto ende;
      case 'i':
	interactive = 1;
	/*   Any arguments passed in commandline?    */
	if(string) commandString = string;

	break;
      case 'o':
	output = string;
	break;
      case 'p':
	/*   Read start picture  */
	startpic = strtol(string,&string2,10);

	if(string2 == string) {
	  fprintf(stderr,"ERROR: Not a number after '-p' -- Break.\n");
	  goto error;
	}

	/*   One number means: Plot only this picture	 */
	if(!*string2) {
	  endpic = startpic;
	  break;
	}

	if(*string2 != ',') {
	  fprintf(stderr,"ERROR: Wrong delimiter '%c' at option '-p'\
 -- Break.\n",*string2);
	  goto error;
	}

	/*   Read end picture	 */
	endpic = strtol(++string2,NULL,10);

	break;
      case 'v':   /*  Set Verbose mode   */
	if(string) {
	  Verbose = strtol2(string,NULL,0);
	} else {
	  Verbose = 2;
	}
	break;
      case 'q':   /*  Be quiet   */
	Verbose = 0;
	break;
      case 's':   /*  Just showing, don't plot   */
	plotflag = 0;
	plotword = "Showing";
	break;
      case 'S':
	for(;;) {
	  double temp;

	  temp = strtod(string,&string2);
	  if(string == string2) break;
	  string = string2;

#if 0
	  if(temp == 0.0) {
	    fprintf(stderr,"WARNING: Scaling factor 0 not allowed, 1 assumed.\n");
	    temp = 1.0;
	  }
#endif

	  if((Scales = (BoundPair *)realloc(Scales,(NScales + 1) *
					    sizeof(*Scales))) == NULL) {
	    fprintf(stderr,"ERROR: No memory -- Break.\n");
	    goto error;
	  }

	  Scales[NScales].BP_Type = BP_POINT;
	  Scales[NScales].BP_Min.BV_Attr = Scales[NScales].BP_Max.BV_Attr =
	    BV_EQUAL;
	  Scales[NScales].BP_Min.BV_Value = Scales[NScales].BP_Max.BV_Value =
	    temp;
	  NScales++;

	  while(isspace(*(unsigned char *)string)) string++;
	  if(*string == ',')
	    string++;
	  else
	    break;
	}
	break;
      case 'd':
	/*  Search device  */
	for(i = 0 ; i < NOutputDevices ; i++)
	  if(strcmp(string,OutDev[i]->name) == 0) break;

	if(i == NOutputDevices) {
	  fprintf(stderr,"ERROR: Unknown device \"%s\" -- Break.\n",string);
	  goto error;
	}
	CurrentDevice = i;

	break;
      case 'D':
	if(DEVICE->devOpts == NULL) {
	  fprintf(stderr,"ERROR: No options for device -- Ignored.\n");
	} else if((i = parseProgOpts(string,DEVICE->devOpts)) < 0) {
	  goto error;
	} else if(string[i]) {
	  fprintf(stderr,"WARNING: Trailing garbage in option ignored.\n\
>>%s\n",string + i);
	}
	break;
      }
    }
  }

  /*  Set number of remaining arguments  */
  RemainingFiles = argc - 1;

  /*  Set flag to show text hunks   */
  if(!plotflag && Verbose < 2)
    Verbose |= 4096;

  if(plotflag) {
    /*  Init device  */
    DEVICE->Init(output);
  }

  if(Verbose & 16)
    printf("Plot area: length= %g, width= %g.\n",
	   DEVICE->picLen,DEVICE->picWidth);

  /*  If no arguments given and program is not used interactive, stop	*/
  if(RemainingFiles == 0) {
    if(interactive) {
      /*  Just call parser and we are ready    */
      if(parse(commandString,MODE_FPLOTREAD | (Verbose & ~1023)))
	goto error;
    } else {
      help(1);
      goto ende;
    }
  } else {
    /* *********   Now parse other arguments   ********* */
    for(argv++ ; --argc > 0 ; argv++) {
      RemainingFiles--;

      if((PlotFile = fplotOpenExt(*argv,FPLOT_READ)) == NULL) goto error;
      DEVICE->fileName = strdup(FPlotOpenName ? FPlotOpenName : *argv);

      if(Verbose)
	printf("#---------- %s file \"%s\" ----------\n",plotword,
	       DEVICE->fileName);

      CurrentPicture = 1;

      /*  Start new page with new file	*/
      if(plotflag)
	DEVICE->Next(OUTPUTDEV_NEXTNEWPAGE|OUTPUTDEV_NEXTNOINC);

      if(interactive) {
	/* ********	 Call parser	 ******** */
	if(parse(commandString,MODE_FPLOTREAD | (Verbose & ~1023)))
	  goto error;
	commandString = NULL; /*  Do not use again   */
      } else {
	/* ********	 Now read the pictures of this file   ******** */
	for( ; CurrentPicture <= endpic ; CurrentPicture++) {
	  /*	Set flag if next picture to be read or only seeked.   */
	  if(startpic <= CurrentPicture) {
	    i = (MODE_FPLOTREAD | FPLOT_MODE_READ) | (Verbose & ~1023);
	    if(Verbose) printf("# %d: ",CurrentPicture);
	  } else {
	    i = 0;
	  }

	  picture = fplotRead(PlotFile,i);

	  if((i & FPLOT_MODE_READ) && picture == NULL) {
	    if(!plotflag) printf("\n");

	    /*	 Test EOF    */
	    if(feof(PlotFile->FF_File)) break;  /*   seems to be EOF   */

	    if(!i) continue;
	    goto error;
	  }

	  /*	Already reached first picture?	 */
	  if(plotflag && startpic <= CurrentPicture) {
	    showpicture(picture);

	    if(plotPicture(picture,PLAIN,0,NULL,0,NULL,0)) {
	      fprintf(stderr,"ERROR in plotting picture -- Break.\n");
	      goto error;
	    }

	    DEVICE->Next(0);
	  }

	  /*	 Free Memory for reuse	  */
	  if(picture) {
	    fplotFreepic(picture);
	    picture = NULL;
	  }
	}
      }

      if(PlotFile) fplotClose(PlotFile);
      if(DEVICE->fileName) free(DEVICE->fileName);
      DEVICE->fileName = NULL;
    }
  }

  /*  Close output file if existant  */
  if(PlotOutFile) fplotClose(PlotOutFile);

  /*  Close up device and rename output file  */
  if(plotflag) {
    char *outputFile;

    if((outputFile = DEVICE->DeInit())) {
      /*  Rename output file  */
      if(output) {
	/*  Check if output file already ok  */
	if(strcmp(output,outputFile))
	  rename(outputFile,output);
      } else if(FileName) {
	char nameBuffer[JS_FILENAME_MAX + 1];

	/*   Jump behind last '/'     */
	if((string = strrchr(FileName,CONFIG_DIR_DELIMITER)) == NULL)
	  string = FileName;
	else
	  string++;

	if( (string2 = strrchr(string,'.')) ) {
	  i = string2 - string;
	} else {
	  i = strlen(string);
	}

	/*   Append extension   */
	sprintf(nameBuffer,"%.*s"PSEXT,i,string);

	rename(outputFile,nameBuffer);
	output = nameBuffer;
      } else
	output = outputFile;

      if(Verbose) printf("Plotted to file \"%s\".\n",output);
    }
  }

  if(Verbose && plotflag) printf("Done.\n");

ende:
  return(0);
error:
  jsperror("plot");
  return(30);
}

/*JS*********************************************************************
*   FPLOTOPENEXT Tries to open the plot file with default extension if
*    necessary.
*************************************************************************/

FPFile *fplotOpenExt(char *fileName,int mode)

/************************************************************************/
{
  FPFile *fp;
static char nameBuffer[JS_FILENAME_MAX + 1];
  char *string;

  /*  Special case: stdin/stdout  */
  if(fileName == NULL || strcmp(fileName,"-") == 0)
    return(fplotOpen(NULL,mode));

  /*  First copy name to permanent space   */
  strcpy(nameBuffer,fileName);

  /*  Save first output file for renaming of the output file at the end  */
  if(FileName == NULL) FileName = strdup(nameBuffer);

  if((fp = fplotOpen(nameBuffer,mode)) == NULL && mode != FPLOT_WRITE &&
     ((string = strrchr(nameBuffer,'.')) == NULL ||
      strchr(string,CONFIG_DIR_DELIMITER)) ) {
    /*	 Try with extension in case we do not want file for writing  */
    strcat(nameBuffer,INPUTEXT);

    fp = fplotOpen(nameBuffer,mode);
  }

  return(fp);
}
