/*JS*********************************************************************
*
*    Program : LCFFTN
*    Language: ANSI-C with call to numerical library routine (ESSL).
*    Author  : Joerg Schoen
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id: lcfftn.c,v 1.3 1996/04/18 08:49:09 joerg Stab joerg $";
#endif

/*********     INCLUDES 					*********/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include <jsconfig.h>

#ifdef CONFIG_LIB_ESSL
# include <essl.h>

 /*  Prevent jscmplx.h from redefining complex data types.  */
# define NO_DEFINE_COMPLEX
#endif

#ifdef CONFIG_LIB_COMPLIB
# include <jscmplx.h>

 /*  Prototypes for the routines I use  */
 dcmplx *zfft2di(int n1,int n2,dcmplx *workspace);
 int zfft2d(int job,int n1,int n2,dcmplx *sequence,
	    int lda,dcmplx *workspace);
#endif

#ifdef CONFIG_LIB_IMSL
# include <imsl.h>
#endif

#include <jssubs.h>

/*********     DEFINES                                          *********/
/*  START-DEFINITIONS */
#include "jscmplx.h" /*  For "dcmplx" definition  */
/*  END-DEFINITIONS */

#define printf logprintf  /*  Use logprintf for output  */

#define Prototype extern
/*********     PROTOTYPES					*********/
Prototype int            lcfftn(dcmplx data[],long nn[],int lDim,int nDim,
				int isign);
Prototype int            setLibFFTn(char *name);
Prototype char          *showLibFFTn(int mode);

#ifdef CONFIG_LIB_ESSL
static int fft2d_essl(dcmplx data[],long nn[],long lDim,int nDim,int isign);
#endif

#ifdef CONFIG_LIB_COMPLIB
static int fft2d_complib(dcmplx data[],long nn[],long lDim,int nDim,int isign);
#endif

#ifdef CONFIG_LIB_IMSL
static int fft2d_imsl(dcmplx data[],long nn[],long lDim,int nDim,int isign);
#endif

/*********     GLOBAL/ STATIC VARIABLES                         *********/
static int (*LibFFT)(dcmplx data[],long nn[],long lDim,int nDim,int isign) =
/*  If multiple libraries available, specify the precedence of use here  */
#ifdef CONFIG_LIB_ESSL
     fft2d_essl;
#elif defined(CONFIG_LIB_COMPLIB)
     fft2d_complib;
#else
     NULL;
#endif

#ifdef CONFIG_LIB_UMFFT
/*  Use Uwe Manthes FFT  */
int umfftn(dcmplx data[],long nn[],long lDim,int nDim,int isign);
#endif

static struct {
  char *LD_Name;
  int (*LD_Routine)(dcmplx data[],long nn[],long lDim,int nDim,int isign);
} LibraryDefs[] = {
#ifdef CONFIG_LIB_ESSL
  { "ESSL", fft2d_essl },
#endif
#ifdef CONFIG_LIB_COMPLIB
  { "COMPLIB",  fft2d_complib },
#endif
#ifdef CONFIG_LIB_IMSL
  { "IMSL", fft2d_imsl },
#endif
#ifdef CONFIG_LIB_UMFFT
  /*  Use Uwe Manthes FFT  */
  { "UMFFT", umfftn },
#endif
  { "INTRINSIC", NULL },
  { NULL, NULL }
};

/*JS*********************************************************************
*
*************************************************************************/

int lcfftn(dcmplx data[],long nn[],int lDim,int nDim,int isign)

/************************************************************************/
{
  int ret;

  if(lDim == 0) lDim = nn[0];

  if(LibFFT) {
    /*  Try library routine  */
    if((ret = (*LibFFT)(data,nn,lDim,nDim,isign)) <= 0)
      return(ret);

    /*  Library routine won't do the job, so disable for future calls  */
    printf("\
WARNING: Library routine for nDim = %d not available -- resort to own\n\
  FFT for future calls.\n",nDim);

    LibFFT = NULL;
  }

  /*  Use own routine  */
  return(cfftn(data,nn,lDim,nDim,isign));
}

/*JS*********************************************************************
*   Set the library FFT routine.
*************************************************************************/

int setLibFFTn(char *name)

/************************************************************************/
{
  int i;

  for(i = 0 ; LibraryDefs[i].LD_Name ; i++)
    if(strcmp(LibraryDefs[i].LD_Name,name) == 0) {
      LibFFT = LibraryDefs[i].LD_Routine;
      return(0);
    }

  return(-1);
}

/*JS*********************************************************************
*   Show available library FFT routines. Returns the current name. If
*    mode != 0, print available names.
*************************************************************************/

char *showLibFFTn(int mode)

/************************************************************************/
{
  int i;
  char *curr;

  /*  Find current setting  */
  for(i = 0 ; LibraryDefs[i].LD_Name ; i++)
    if(LibraryDefs[i].LD_Routine == LibFFT) break;

  curr = LibraryDefs[i].LD_Name;

  if(mode) {
    printf("Available library FFT's (default \"%s\"):",curr);

    for(i = 0 ; LibraryDefs[i].LD_Name ; i++)
      printf("%c %s",i > 0 ? ',' : ' ',LibraryDefs[i].LD_Name);
    printf("\n");
  }

  return(curr);
}

#ifdef CONFIG_LIB_ESSL
/*JS*********************************************************************
*   FFT with ESSL routine.
*************************************************************************/

static int fft2d_essl(dcmplx data[],long nn[],long lDim,int nDim,int isign)

/************************************************************************/
{
  switch(nDim) {
  case 2: /* ***  2D FFT  *** */
    {
#define NAUX1   200000
#define NAUX2   100000
      int direct;
static int Init[2] = { 1, 1 };
static double Aux1[2][NAUX1],Aux2[2][NAUX2];

      direct = isign > 0 ? 1 : 0;

      /*  To be consistent with own routine, reverse isign  */
      isign = -isign;
      if(Init[direct]) {
	/*  Fill up auxiliary storage  */
	dcft2(1,data,1,lDim,data,1,lDim,nn[0],nn[1],isign,1.0,
	      Aux1[direct],NAUX1,Aux2[direct],NAUX2);

	Init[direct] = 0;
      }

      /*  Call ESSL routine */
      dcft2(0,data,1,lDim,data,1,lDim,nn[0],nn[1],isign,1.0,
	    Aux1[direct],NAUX1,Aux2[direct],NAUX2);
#undef NAUX1
#undef NAUX2
    }
    break;
  default: /* ***  Not available  *** */
    return(1);
  }

  return(0);
}
#endif

#ifdef CONFIG_LIB_COMPLIB
/*JS*********************************************************************
*   FFT with COMPLIB routine.
*************************************************************************/

static int fft2d_complib(dcmplx data[],long nn[],long lDim,int nDim,int isign)

/************************************************************************/
{
  switch(nDim) {
  case 2: /* ***  2D FFT  *** */
    {
#define NAUX  (2 * (8192 + 15))
      static dcmplx WorkSpace[NAUX];
      static int Init = TRUE;

      if(Init) {
	/*  Initialize work space  */
	if((nn[0] + nn[1] + 30) > NAUX) {
	  printf("\
ERROR: Workspace to small -- Cannot call 'zfft2di' -- Break.\n");
	  return(-1);
	}

	zfft2di(nn[0],nn[1],WorkSpace);
	Init = FALSE;
      }

      /*  Do the two dimensional FFT  */
      zfft2d(isign,nn[0],nn[1],data,lDim,WorkSpace);
#undef NAUX
    }
    break;
  default: /* ***  Not available  *** */
    return(1);
  }

  return(0);
}
#endif

#ifdef CONFIG_LIB_IMSL
/*JS*********************************************************************
*   FFT with IMSL routine.
*************************************************************************/

static int fft2d_imsl(dcmplx data[],long nn[],long lDim,int nDim,int isign)

/************************************************************************/
{
  switch(nDim) {
  case 2: /* ***  2D FFT  *** */
    {
static double *WFF1,*WFF2,*CPY;
static dcmplx *WC;

      if(WFF1 == NULL) { /*  Initialize  */
	/*  The IMSL documentation is not clear: "15 +" or "15 *" ??  */
	if((WFF1 = (double *)malloc(60 * nn[0] * sizeof(*WFF1))) == NULL ||
	   (WFF2 = (double *)malloc(60 * nn[1] * sizeof(*WFF2))) == NULL ||
	   (WC = (dcmplx *)malloc(nn[0] * nn[1] * sizeof(*WC))) == NULL ||
	   (CPY = (double *)malloc(MAX(nn[0],nn[1]) * sizeof(*CPY))) == NULL)
	  return(-1);

	dfftci(nn[0],WFF1);
	dfftci(nn[1],WFF2);
      }

      if(isign == 1)
	/*  Forward transform  */
	df2t2b(nn[0],nn[1],data,lDim,data,lDim,WFF1,WFF2,WC,CPY);
      else
	/*  Backward transform  */
	df2t2d(nn[0],nn[1],data,lDim,data,lDim,WFF1,WFF2,WC,CPY);
    }
    break;
  default: /* ***  Not available  *** */
    return(1);
  }

  return(0);
}
#endif
