/*JS*********************************************************************
*
*    Program : JTSUBS
*    Language: ANSI-C
*    Author  : Joerg Schoen
*    Purpose : Auxiliary routines for Jahn-Teller programs.
*
*    Part    : Subroutines for reading dump files.
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id: jtsubs.c,v 1.1 94/03/30 14:43:45 joerg Exp Locker: joerg $";
#endif

/*********     INCLUDES 					*********/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <float.h>
#include <math.h>
#include <time.h>

#include "jtsubs.h"

/*********     PROTOTYPES					*********/
/* ****  General title for pictures   ***** */
Prototype char		   *PlotLabel[],*PlotAxeLabel[];

/* ****  General routines for reading dumps   **** */
Prototype JateDumpFile	   *JTReadDump(char *name,int mode,int nStart,
				       int nEnd);
Prototype void		    JTFreeDump(JateDumpFile *dumpFile);

Prototype SpecDumpFile	   *SPReadDump(char *name);
Prototype void		    SPFreeDump(SpecDumpFile *specDump);

Prototype const char	    ElecChar[],*JTildText[2][3];

/*********     GLOBAL VARIABLES 				*********/
/*   Explanation modes for JTild variable  */
const char *JTildText[2][3] = {
  { "3/2", "-1/2", "1/2" },
  { "0",   "1",    "2"   }
};

/*   Text for the Title of pictures   */
char *PlotLabel[2 * ELECFUNC_NUM] = {
  "Real part(-)", "Imaginary part(-)",
  "Real part(+)", "Imaginary part(+)",
  "Real part(A)", "Imaginary part(A)"
};

/*   Text for the Axes in the pictures	 */
char *PlotAxeLabel[2 * ELECFUNC_NUM] = {
  "Re(psi-)", "Im(psi-)",
  "Re(psi+)", "Im(psi+)",
  "Re(psiA)", "Im(psiA)"
};

/*  Corresponding character for Elecvalues -1 , 0 , +1 (used for output) */
const char ElecChar[] = {
  '-' , 'A' , '+'
};

/*JS*********************************************************************
*   JTREADDUMP Reads in a dump file and stores the contents in a structure
*    The structure is returned. On failure NULL is returned. Reads only
*    eigenvalues and vectors from nStart to nEnd. If nEnd is -1 it reads
*    all eigenvalues and -vector and if nEnd is -2, only the VMax vectors.
*************************************************************************/

JateDumpFile *JTReadDump(char *name,int mode,int nStart,int nEnd)

/************************************************************************/
{
  FILE *fp;
  JateDumpFile *dumpFile;
  int i,j,dim,nSaved; /*  "Number of SAVED data"  */

  fp = (mode & JTRD_USEPATH) ? fopenPath(name,"rb") : fopen(name,"rb");

  if(mode & JTRD_PRNAME)
    logprintf("Reading dump file \"%s\"...\n",
	      ((mode & JTRD_USEPATH) && FOpenName) ? FOpenName : name);

  if(fp == NULL) {
    logprintf("ERROR: Could not open dump file \"%s\"!\n",name);
    return(NULL);
  }

  /*  Allocate memory for structure    */
  if((dumpFile = (JateDumpFile *)malloc(sizeof(*dumpFile))) == NULL) {
    logprintf("ERROR: No memory for dump file!\n");
    goto error;
  }

  /*   Preset values for error case    */
  dumpFile->JDF_QNum = NULL;
  dumpFile->JDF_EigenValue = NULL;

  /*  Read in header	 */
  if( fread(&(dumpFile->JDF_Header),sizeof(dumpFile->JDF_Header),1,fp) != 1) {
    logprintf("ERROR: Could not read Header of dump file!\n");
    goto error;
  }

  /*   Test if dump file   */
  if(dumpFile->JDF_Header.DH_Magic != JATEDUMPMAGIC) {
    logprintf("ERROR: File is not a dump file (Magic = %#x)!\n",
	   dumpFile->JDF_Header.DH_Magic);
    goto error;
  }

  /*   Get most important values     */
  dim	= dumpFile->JDF_Header.DH_Dim;
  nSaved = dumpFile->JDF_Header.DH_NSave;

  if(nEnd == -1) {
    /*	Read all vectors    */
    nEnd = nSaved;
  } else if(nEnd == -2) {
    /*	Number of converged vectors   */
    nEnd = dumpFile->JDF_VMax;
  } else if(nEnd < 0) {
    logprintf("ERROR in 'JTReadDump': Wrong value %d for nEnd -- Break.\n",nEnd);
    goto error;
  }

  /*   Limit nEnd to maximum value   */
  nEnd = MIN(nSaved,nEnd);

  /*  Set up correct values   */
  if(nStart < 0) nStart = 0;

  if(nStart >= nEnd) {
    logprintf("ERROR in 'JTReadDump': Wrong value for nStart %d >= %d --\
 Break.\n",
	   nStart,nEnd);
    goto error;
  }

  /*  Set up correct values in header	*/
  dumpFile->JDF_NStart = nStart;
  dumpFile->JDF_NSave = nEnd - nStart;

  /*   Allocate memory for all needed stuff	*/
  i = dim * sizeof(*dumpFile->JDF_QNum);
  if((dumpFile->JDF_QNum = (QNumber *)malloc(i)) == NULL) {
    logprintf("ERROR: Couldn't allocate %d Bytes for quantum numbers!\n",i);
    goto error;
  }

  i = (nEnd - nStart) * (dim + 1) * sizeof(*dumpFile->JDF_EigenValue);
  if((dumpFile->JDF_EigenValue = (double *)malloc(i)) == NULL) {
    logprintf("ERROR: Couldn't allocate %d Bytes for dump file stuff!\n",i);
    goto error;
  }

  dumpFile->JDF_EigenVector = &dumpFile->JDF_EigenValue[nEnd - nStart];

  /*   Read in ALL quantum numbers  */
  if(fread(dumpFile->JDF_QNum,sizeof(*dumpFile->JDF_QNum),dim,fp) != dim) {
    logprintf("ERROR: Could not read quantum numbers -- Break.\n");
    goto error;
  }

  /*   Seek to first eigenvalue   */
  if(nStart > 0)
    if( fseek(fp,nStart * sizeof(*dumpFile->JDF_EigenValue),SEEK_CUR) ) {
      logprintf("ERROR: Could not fseek to eigenvalues -- Break.\n");
      goto error;
    }

  if( fread(dumpFile->JDF_EigenValue,sizeof(*dumpFile->JDF_EigenValue),
	    nEnd - nStart,fp) != (nEnd - nStart) ) {
    logprintf("ERROR: Could not read eigenvalues -- Break.\n");
    goto error;
  }

  /*   Skip intermediate values    */
  if( (i = (nSaved - nEnd) + nStart * dim) )
    if( fseek(fp,i * sizeof(*dumpFile->JDF_EigenValue),SEEK_CUR) ) {
      logprintf("ERROR: Could not skip intermediate values -- Break.\n");
      goto error;
    }

  /*   Read eigenvectors    */
  if( fread(dumpFile->JDF_EigenVector,sizeof(*dumpFile->JDF_EigenVector) * dim,
	    nEnd - nStart,fp) != (nEnd - nStart) ) {
    logprintf("ERROR: Could not read eigenvectors -- Break.\n");
    goto error;
  }

  if(mode & JTRD_PRHEADER) {
    /*	 Print contents of header  */
    logprintf("DumpFile:\n\
  Mode: %s   NMax: %d  VMax: %d  Dim: %d    NSave: %d\n\
  JTildp3h: %d	   Omega: %g  OmegaQuant: %g   E_E: %g	   E_A: %g\n\
  K: %g     G: %g     L: %g	F: %g    DeltaDiag: %g\n",
	      dumpFile->JDF_Header.DH_Mode ? "PJT" : "JT",
	      dumpFile->JDF_Header.DH_NMax,dumpFile->JDF_Header.DH_VMax,
	      dumpFile->JDF_Header.DH_Dim,nSaved,dumpFile->JDF_Header.DH_JTildp3h,
	      dumpFile->JDF_Header.DH_Omega,dumpFile->JDF_Header.DH_OmegaQuant,
	      dumpFile->JDF_Header.DH_E_E,dumpFile->JDF_Header.DH_E_A,
	      dumpFile->JDF_Header.DH_K,dumpFile->JDF_Header.DH_G,
	      dumpFile->JDF_Header.DH_L,dumpFile->JDF_Header.DH_F,
	      dumpFile->JDF_Header.DH_DeltaDiag);
  }

  if(mode & JTRD_PRQNUM) {
    logprintf("Oscillatorfunctions (%d):\n",dim);

    for(i = 0 ; i < dim ; i++) {
      if( (i % 25) == 0)
	logprintf("---------------------%d-------------------------------\n",
		  i);

      logprintf("|%d,%d,%c>%c",dumpFile->JDF_QNum[i].QN_N,
		dumpFile->JDF_QNum[i].QN_L,
		ElecChar[dumpFile->JDF_QNum[i].QN_Elec + 1],
		(i % 5) == 4 ? '\n' : ' ');
    }
    logprintf("\n\n");
  }

  if(mode & (JTRD_PREVAL|JTRD_PREVEC)) {
    int i2;

    if( !(mode & JTRD_PREVEC) )
      logprintf("Eigenvalues and Norm of eigenvectors:\n");

    for(i = nStart ; i < nEnd ; i++) {
      double norm;

      i2 = dim * i;

      /*  Calculate norm    */
      for(norm = 0.0 , j = 0 ; j < dim ; j++)
	norm += dumpFile->JDF_EigenVector[j + i2] *
	  dumpFile->JDF_EigenVector[j + i2];

      if(mode & JTRD_PREVEC) {
	logprintf("\nEigenvalue %3d is %24.17g. Vector has norm = %g.\n",
		  i,dumpFile->JDF_EigenValue[i],norm);

	logprintf("  Eigenvector:\n");
	for(j = 0 ; j < dim ; j++) {
	  if( (j % 25) == 0) logprintf("-----%d-----\n",j);

	  logprintf("%24.17g\n",dumpFile->JDF_EigenVector[j + i2]);
	}
      } else {
	if( (i % 25) == 0) logprintf("-----%d-----\n",i);

	logprintf("%24.17g %24.17g\n",dumpFile->JDF_EigenValue[i],norm);
      }
    }
    logprintf("\n");
  }

  fclose(fp);

  return(dumpFile);

error:
  fclose(fp);

  if(dumpFile) JTFreeDump(dumpFile);

  return(NULL);
}

/*JS*********************************************************************
*   JTFREEDUMP Frees the structure.
*************************************************************************/

void JTFreeDump(JateDumpFile *dumpFile)

/************************************************************************/
{
  if(dumpFile->JDF_EigenValue)
    free(dumpFile->JDF_EigenValue);

  if(dumpFile->JDF_QNum)
    free(dumpFile->JDF_QNum);

  /*   the memory for all other stuff  */
  /*	was simultaneously allocated   */

  free(dumpFile);

  return;
}

/*JS*********************************************************************
*   SPREADDUMP
*************************************************************************/

SpecDumpFile *SPReadDump(char *name)

/************************************************************************/
{
  SpecDumpFile *specDump = NULL;
  SpecDumpHeader specHeader;
  FILE *fp;
  int vecDim;

  if((fp = fopen(name,"r")) == NULL) {
    logprintf("ERROR: Could not open \"%s\" -- Break.\n");
    goto error;
  }

  /*  Read whole dump header  */
  if(fread(&specHeader,sizeof(specHeader),1,fp) != 1) {
    logprintf("ERROR: Could not read dump header -- Break.\n");
    goto error;
  }

  if(specHeader.DH_Magic != SPECDUMPMAGIC) {
    /*logprintf("ERROR: Not a spec dump file -- Break.\n");*/
    goto error;
  }

  /*  Read all that's there  */
  vecDim = (specHeader.DH_NrStates * (specHeader.DH_NMax + 1) *
	    (specHeader.DH_NMax + 2)) / 2;

  /*  Get enough memory  */
  if((specDump = (SpecDumpFile *)malloc(sizeof(*specDump))) == NULL ||
     (specDump->SDF_Vector = (dcmplx *)malloc(vecDim *
					      sizeof(*specDump->SDF_Vector)))
     == NULL) {
    logprintf("ERROR: No memory for spec dump file -- Break.\n");
    goto error;
  }
  specDump->SDF_Header = specHeader;
  specDump->SDF_Dim = vecDim;

  /*  Read state  */
  if(fread(specDump->SDF_Vector,sizeof(*specDump->SDF_Vector),vecDim,fp)
     != vecDim) {
    logprintf("ERROR: Could not read state from dump file -- Break.\n");
    goto error;
  }

  /*  Close dump file */
  fclose(fp);

  return(specDump);
error:
  if(specDump) {
    if(specDump->SDF_Vector) free(specDump->SDF_Vector);
    free(specDump);
  }
  return(NULL);
}

/*JS*********************************************************************
*   SPFREEDUMP
*************************************************************************/

void SPFreeDump(SpecDumpFile *specDump)

/************************************************************************/
{
  if(specDump) {
    if(specDump->SDF_Vector) free(specDump->SDF_Vector);
    free(specDump);
  }
}
