/*JS*********************************************************************
*
*    Program : JSEXPR
*    Language: ANSI-C
*    Author  : Joerg Schoen
*    Purpose : Allows to calculate in double precision.
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id: jsexpr.c,v 1.5 1997/03/06 09:48:23 joerg Stab joerg $";
#endif

/*********     INCLUDES                                         *********/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <float.h>
#include <math.h>
#include <errno.h>

#include <jssubs.h>

#include <fplot.h>

/*********     DEFINES                                          *********/
#define OPTSTRING     "hf:s:S:F:R:jpPi:z:Z:o:a:T:t:x:n"
#define DEFFORMAT     "%g"
#define DEFSEPARATOR0 "\n"
#define DEFSEPARATOR  " "

/*  Name of global profile in HOME directory  */
#define PROFILE       ".jsexprc"

#define MODE_IPN        (1<<0)
#define MODE_ALLZEROES  (1<<1)
#define MODE_JOINTLY    (1<<2)
#define MODE_FPLOTOUT   (1<<3)
#define MODE_FPLOTAPP   (1<<4)
#define MODE_NOEND      (1<<5)

/*  Allowed characters for identifier  */
#define isident(c)  ((c) == '_' || isalpha(c))

/*  Structure to store the values for a given variables range  */
struct Range {
  char  *R_Name;
  int    R_Max,R_Work;
  double R_Values[1];
};

/*  Precision for zero points  */
#define EPSILON   (10.0 * DBL_EPSILON)

#define Prototype extern
/*********     PROTOTYPES                                       *********/
Prototype int                newFunction(Expression *expr);
Prototype int                newExpression(char *name,Expression *expr);

Prototype struct Range      *getRange(char *input);

Prototype int                readFile(char *name,int mode);

Prototype int                findZero(struct Range *range,double *vars,
				      int index,Expression *expr);
Prototype int                bracketize(Expression *expr,double *vars,
					int index,double left,double right,
					double fLeft,double fRight);

/*********     GLOBAL VARIABLES                                 *********/
int Mode;

Expression **Fcts,**Exprs;
int FctCount,FctMax,ExprCount,ExprMax;
char *OutFormat,*OutSeparator0,*OutSeparator;

/*JS*********************************************************************
*   MAIN
*************************************************************************/

int main(int argc,char *argv[])

/************************************************************************/
{
  char *string;
  const char *string2;
  int i,c,flag;
  Expression *expr;
  char **rangeNames;
  struct Range **ranges;
  int rangeCount,rangeMax;
  int zeroSearch;
  double *vars,*array;
  char *outFile,*text,*axes;
  int type;
  unsigned long *dims,arrDim,arrOffset;

  /*  Preset defaults  */
  Mode = 0;
  OutFormat = DEFFORMAT;
  Fcts = Exprs = NULL;
  ExprCount = ExprMax = FctCount = FctMax = 0;
  rangeNames = NULL;
  ranges = NULL;
  rangeCount = rangeMax = 0;
  vars = NULL;
  zeroSearch = -1;
  text = axes = NULL;
  type = 0;

  /*  Read global profile  */
  if((string = getenv("HOME"))) {
    char name[JS_FILENAME_MAX];

    sprintf(name,"%s/%s",string,PROFILE);

    /*  Ignore non existant file and forbid expressions in file  */
    if(readFile(name,3)) goto error;
  }

  /*  Scan command line options  */
  while((c = jsGetOpt(&argc,argv,OPTSTRING,&string)) != EOF) {
    switch(c) {
    default:
      goto error;
    case 'h':
      goto help;
    case 'f':
      OutFormat = string;
      break;
    case 's':
      OutSeparator = string;
      break;
    case 'S':
      OutSeparator0 = string;
      break;
    case 'F':
      if((expr = CompDefinition(string,(const Expression **)Fcts,&string2))
	 == NULL) {
	fprintf(stderr,"\
ERROR jsexpr: Cannot compile function definition -- Break.\n");
	goto error;
      }

      if(*string2) {
        fprintf(stderr,"\
ERROR: Garbage in function definition -- Break.\n>>%s\n",string2);
	goto error;
      }

      /*  Check if function definition  */
      if(expr->EX_Name == NULL) {
	fprintf(stderr,"\
ERROR jsexpr: \"%s\" not a function definition -- Break.\n",
		string);
	goto error;
      }

      if(newFunction(expr)) goto error;

      break;
    case 'Z':
      Mode |= MODE_ALLZEROES;
    case 'z':
      if(zeroSearch >= 0) {
	fprintf(stderr,"\
ERROR jsexpr: Only one zero point search allowed -- Break.\n");
	goto error;
      }
      zeroSearch = rangeCount;
    case 'R':
      /*  Enough memory?  */
      if(rangeCount >= rangeMax) {
	rangeMax += 5;
	if((ranges = (struct Range **)realloc(ranges,rangeMax *
					      sizeof(*ranges))) == NULL ||
	   (rangeNames = (char **)realloc(rangeNames,(rangeMax + 1) *
					  sizeof(*rangeNames))) == NULL)
	  goto error;
      }

      if((ranges[rangeCount] = getRange(string)) == NULL) goto error;

      rangeNames[rangeCount] = ranges[rangeCount]->R_Name;
      rangeNames[rangeCount + 1] = NULL;

      /*  Check if range name appeared twice  */
      for(i = 0 ; i < rangeCount ; i++)
	if(strcmp(rangeNames[rangeCount],rangeNames[i]) == 0) {
	  fprintf(stderr,"\
ERROR jsexpr: Range name \"%s\" appeared twice -- Break.\n",
		  rangeNames[rangeCount]);
	  goto error;
	}

      rangeCount++;

      break;
    case 'P':
      ExpressionMode = FALSE;
    case 'p':
      Mode |= MODE_IPN;
      break;
    case 'i':
      if(readFile(string,0)) goto error;
      break;
    case 'j':
      Mode |= MODE_JOINTLY;
      break;
    case 'a':
      Mode |= MODE_FPLOTAPP;
    case 'o':
      Mode |= MODE_FPLOTOUT;
      outFile = string;
      break;
    case 'T':
      type = atoi(string);
      break;
    case 't':
      text = string;
      break;
    case 'x':
      axes = string;
      break;
    case 'n':
      Mode |= MODE_NOEND;
      break;
    }
  }

  if(Mode & MODE_FPLOTOUT) {
    if(rangeCount == 0)
      Mode &= ~MODE_FPLOTOUT;
    else if(Mode & MODE_JOINTLY)
      Mode &= ~MODE_JOINTLY;
  }

  /*  Scan remainder of command line  */
  for(i = 1 ; i < argc ; i++) {
    if((expr = CompExpression(argv[i],(const char **)rangeNames,
			      (const Expression **)Fcts,&string2)) == NULL) {
      fprintf(stderr,"\
ERROR jsexpr: Cannot compile expression \"%s\" -- Break.\n",
	      argv[i]);
      goto error;
    }

    if(*string2) {
      fprintf(stderr,"\
WARNING: Garbage in expression -- Break.\n>>%s\n",string2);
      goto error;
    }

    if(newExpression(argv[i],expr)) goto error;
  }

  /*  Handle '-p' case  */
  if(Mode & MODE_IPN) {
    /*  Show functions first  */
    for(i = 0 ; i < FctCount ; i++)
      ShowExpression(Fcts[i],NULL,(const Expression **)Fcts);

    for(i = 0 ; i < ExprCount ; i++) {
      ShowExpression(Exprs[i],(const char **)rangeNames,
		     (const Expression **)Fcts);
    }

    return(0);
  }

  /*  Now evaluate expressions  */
  array = NULL;
  if(rangeCount > 0) {
    if((vars = (double *)malloc(rangeCount * sizeof(*vars))) == NULL)
      goto error;

    if(Mode & MODE_FPLOTOUT) {
      /*  Calculate single and total dimension  */
      if((dims = (unsigned long *)malloc(rangeCount * sizeof(*dims))) == NULL)
	goto error;

      arrDim = 1;
      for(i = 0 ; i < rangeCount ; i++)
	arrDim *= dims[rangeCount - 1 - i] = ABS(ranges[i]->R_Max);

      /*  Use calloc to get elements defaulting to zero  */
      if((array = (double *)calloc(arrDim * ExprCount,sizeof(*array)))
	 == NULL)
	goto error;

      arrOffset = 0;
    }

    /*  Preset counters  */
    for(i = 0 ; i < rangeCount ; i++) ranges[i]->R_Work = 0;
  }

  flag = 0; /*  Flag for output  */
  do {
    /*  Set variables  */
    for(i = 0 ; i < rangeCount ; i++) {
      if(zeroSearch == i) continue;

      if(ranges[i]->R_Max >= 0)
	vars[i] = ranges[i]->R_Values[ranges[i]->R_Work];
      else
	vars[i] = ranges[i]->R_Values[0] + ranges[i]->R_Work *
	  ranges[i]->R_Values[2];
    }

    for(i = 0 ; i < ExprCount ; i++) {
      if(flag && !(Mode & MODE_FPLOTOUT)) {
	if(i == 0)
	  printf(OutSeparator0 ? OutSeparator0 : DEFSEPARATOR0,1 + flag);
	else
	  printf(OutSeparator ? OutSeparator : DEFSEPARATOR,1 + i);
      }

      /*  Zero points wanted?  */
      if(zeroSearch >= 0) {
	if((c = findZero(ranges[i],vars,zeroSearch,Exprs[i])) < 0)
	  goto error;
	if(c) flag++;
      } else if(Mode & MODE_FPLOTOUT) {
	array[arrOffset + i * arrDim] =
	  EvalExpression(Exprs[i],vars,NULL,(const Expression **)Fcts);
      } else {
	/*  Print result  */
	printf(OutFormat,EvalExpression(Exprs[i],vars,NULL,
					(const Expression **)Fcts));
	flag++;
      }
    }

    if(Mode & MODE_FPLOTOUT) arrOffset++;

    /*  Increment ranges  */
    for(i = rangeCount - 1 ; i >= 0 ; i--) {
      if(zeroSearch == i) continue;

      (ranges[i]->R_Work)++;
      if(Mode & MODE_JOINTLY) {
	if(ranges[i]->R_Work >= ABS(ranges[i]->R_Max)) break;
      } else {
	if(ranges[i]->R_Work < ABS(ranges[i]->R_Max)) break;
	ranges[i]->R_Work = 0; /*  Start value  */
      }
    }
  } while((Mode & MODE_JOINTLY) ? (i < 0 && rangeCount > 0) : i >= 0);

  if(Mode & MODE_FPLOTOUT) {
    const double **axeData;
    double *bounds;
    FPFile *fp;

    if((axeData = (const double **)malloc(rangeCount * sizeof(*axeData)))
       == NULL ||
       (bounds = (double *)malloc(2 * rangeCount * sizeof(*bounds))) == NULL)
      goto error;

    for(i = 0 ; i < rangeCount ; i++) {
      if(ranges[rangeCount - 1 - i]->R_Max >= 0) {
	bounds[2 * i] = bounds[2 * i + 1] = 0.0;
	axeData[i] = ranges[rangeCount - 1 - i]->R_Values;
      } else {
	bounds[2 * i]     = ranges[rangeCount - 1 - i]->R_Values[0];
	bounds[2 * i + 1] = ranges[rangeCount - 1 - i]->R_Values[1];
	axeData[i] = NULL;
      }
    }

    if((fp = fplotOpen(strcmp(outFile,"-") == 0 ? NULL : outFile,
		       (Mode & MODE_FPLOTAPP) ? FPLOT_APPEND : FPLOT_WRITE))
       == NULL ||
       fplotStart((Mode & MODE_NOEND) ? NULL : fp) ||
       (text && fplotText(fp,"%s",text)))
      goto error;

    if(arrOffset != arrDim)
      fprintf(stderr,"Problems with arithmetric comparison -- Remainder set to 0\n");

    for(i = 0 ; i < ExprCount ; i++)
      if(fplotTensor(fp,type,rangeCount,axeData,&array[arrDim * i],dims,NULL,
		     bounds,1.0,1.0))
	goto error;

    if((axes && fplotAxesS(fp,axes)) ||
       fplotEnd((Mode & MODE_NOEND) ? NULL : fp) ||
       fplotClose(fp))
      goto error;

    free(axeData);
    free(array);
    free(dims);
  } else if(flag) {
    printf("\n");
  } else if(zeroSearch >= 0) {
    /*  No zeros found, return non-zero  */
    return(2);
  }

  return(0);
error:
  jsperror("jsexpr");
  return(30);
help:
  fprintf(stderr,"\
Usage: jsexpr [<opt>] <expr> ...                         Joerg Schoen 1994\n\
  Evaluates the expressions in double precision and prints them on\n\
  stdout, line by line.\n\
  Options:\n\
    -h           Show this help information.\n\
    -f<format>   Choose output format for printf (Default: \"%s\").\n\
    -s<sep>      Set field separator. Is used to separate multiple expressions\n\
                 on one line. May contain \"%%d\" which will be substituted\n\
                 by the ordinal number of this expression. Defaults to\n\
                 \"%s\".\n\
    -S<sep>      Set field separator. Is used when having ranges to separate\n\
                 them. Defaults to newline.\n\
    -p|-P        Don't evaluate expression, print the compiled form (inverse\n\
                 polish notation). '-P' prints the original form which is not\n\
                 simplified.\n\
    -F<funcdef>  Define a function that can be used in expressions.\n\
    -R<var>=<val1>,<val2>,...   or   -R<var>=<start>-<end>[,<step>]\n\
                 Set a range where the expressions are evaluated. Multiple\n\
                 ranges are allowed, yielding nested loops where the last\n\
                 range runs first. All numbers may consist of expressions as\n\
                 well. Instead of ',' spaces may be used. If the expression\n\
                 for <start> itself contains a \'-\', enclosing the expression\n\
                 in parentheses protects the hyphen from a misinterpretation.\n\
    -j           In case of multiple ranges, increments them jointly. The\n\
                 total number of iterations is the minimum of all single\n\
                 ranges.\n\
    -z<var>=<val1>,<val2>,...   or   -z<var>=<start>-<end>[,<step>]   or\n\
    -Z<var>=<val1>,<val2>,...   or   -Z<var>=<start>-<end>[,<step>]\n\
                 Do a zero point search for every expression given, looking\n\
                 for a sign change between two successive values and than\n\
                 employing a simple interval bracketizing method. 'Z' finds\n\
                 all zeroes. If no zero point was found, the exit value is 2.\n\
                 The statements from the \'-R\' options about expressions\n\
                 apply here as well.\n\
    -i <file> or -i -\n\
                 Read expressions and function definitions from file or\n\
                 standard input.\n\
    -o <outfile> Print evaluated expressions to file in 'fplot' format.\n\
                 Works only when ranges are given to have dimensionality of\n\
                 data.\n\
    -a <outfile> Same as '-o', but appends to the file.\n\
    -T <type>    Set plot type.\n\
    -t <text>    Write text to output file.\n\
    -x <text>    Write axe description to output file. Text for different\n\
                 axes must be separated with ';'.\n\
    -n           Suppress end marker in output file.\n\
",DEFFORMAT,DEFSEPARATOR);
  return(5);
}

/*JS*********************************************************************
*   NEWFUNCTION Add a new function definition to global list.
*************************************************************************/

int newFunction(Expression *expr)

/************************************************************************/
{
  int i;

  for(i = 0 ; i < FctCount ; i++)
    if(strcmp(Fcts[i]->EX_Name,expr->EX_Name) == 0) {
      /*  Substitute expression  */
      FreeExpression(Fcts[i]);
      Fcts[i] = expr;
      return(0);
    }

  /*  Save definition  */
  if((FctCount + 1) >= FctMax) {
    FctMax += 5;
    if((Fcts = (Expression **)realloc(Fcts,FctMax * sizeof(*Fcts))) == NULL)
      return(-1);
  }

  Fcts[FctCount++] = expr;
  Fcts[FctCount] = NULL; /*  Mark last function  */

  return(0);
}

/*JS*********************************************************************
*   NEWEXPRESSION Add a new expression to global list.
*************************************************************************/

int newExpression(char *name,Expression *expr)

/************************************************************************/
{
  /*  Save expression  */
  if(ExprCount >= ExprMax) {
    ExprMax += 5;
    if((Exprs = (Expression **)realloc(Exprs,ExprMax * sizeof(*Exprs)))
       == NULL) return(-1);
  }

  Exprs[ExprCount] = expr;
  ExprCount++;

  return(0);
}

/*JS*********************************************************************
*   GETRANGE Read a range from input of the form "var=val1,val2,..." or
*    "var=start-end[,<step>]".
*************************************************************************/

struct Range *getRange(char *input)

/************************************************************************/
{
  struct Range *range;
  char *name,*string;
  int count,max,flag;

  /*  Skip spaces  */
  while(isspace(*(unsigned char *)input)) input++;

  /*  Read var name  */
  if((count = strtoid(name = input)) == 0) goto error;
  input += count;

  /*  Skip spaces  */
  while(isspace(*(unsigned char *)input)) input++;

  if(*input++ != '=') goto error;
  name[count] = '\0'; /*  Delimit name  */

  range = NULL;
  count = max = 0;

  /*  Skip heading spaces  */
  while(isspace(*(unsigned char *)input)) input++;

  for(flag = FALSE ; ; )  {
    if(count >= max) {
      max += 3; /*  We need at least 3 values!  */
      if((range = (struct Range *)
	  realloc(range,sizeof(*range) +
		  (max - 1) * sizeof(range->R_Values[0]))) == NULL) {
	fprintf(stderr,"ERROR jsexpr: No memory -- Break.\n");
	goto error2;
      }
    }

    /*  Try to read next value  */
#if 0
    /*  Old code: only numbers are allowed  */
    range->R_Values[count++] = strtod(input,&string);
#else
    /*  Problem when interpreting "1-10": Yields -9, but
     *   means 1,2,3 ... So we count brackets, since '-'
     *   within brackets is ok.
     *   I also like to have the interpreation of "1 10"
     *   as "1,10", not "1*10" (i. e. single number).
     */
    {
      int bCount;

      for(bCount = 0, string = input ; *string ; string++) {
	if((*string == '-' || isspace(*string)) && bCount == 0)
	  break;
	else if(*string == '(')
	  bCount++;
	else if(*string == ')')
	  bCount--;
      }

      if(*string) {
	char c;

	/*  We got a '-' or space, hide it when reading expression  */
	c = *string;
	*string = '\0';
	range->R_Values[count] = EvalString(input,NULL,NULL,
					    (const Expression **)Fcts,
					    (const char **)&string);
	if(*string) goto error;
	*string = c; /*  restore  */
      } else
	range->R_Values[count] = EvalString(input,NULL,NULL,
					    (const Expression **)Fcts,
					    (const char **)&string);

      /*  Any problems during evaluation  */
      if(range->R_Values[count] == HUGE_VAL) goto error;
      count++;
    }
#endif

    if(input == string) goto error;
    input = string;

    /*  Skip spaces  */
    while(isspace(*(unsigned char *)input)) input++;

    if(*input == '\0') break;

    if(*input == '-' && !flag) {
      flag = TRUE;
      input++;
    } else if(*input == ',') {
      input++;
    } else if(input == string)
      /*  Need a comma or at least one space as separator  */
      goto error;
  }

  range->R_Name = name;
  if(flag) {
    if(count > 3) goto error;

    /*  Default of stepsize is 1  */
    if(count == 2) range->R_Values[2] = 1.0;

    /*  Calculate number of iterations  */
    range->R_Max = -(int)(1.5 + (range->R_Values[1] - range->R_Values[0]) /
			  range->R_Values[2]);
  } else {
    range->R_Max = count;
  }

  return(range);
error:
  fprintf(stderr,"ERROR jsexpr: Syntax error in range -- Break.\n");
error2:
  return(NULL);
}

/*JS*********************************************************************
*   READFILE Read file 'name'. If bit 0 in mode is set, ignore if file
*    cannot be read. If bit 1 in mode is set, forbid expressions in file.
*************************************************************************/

int readFile(char *name,int mode)

/************************************************************************/
{
  FILE *fp;
  Expression *expr;
  char *string;

  if(strcmp(name,"-") == 0)
    fp = stdin;
  else if((fp = fopen(name,"r")) == NULL) {
    /*  Ignore non existant file?  */
    if(mode & 1) return(0);

    fprintf(stderr,"ERROR jsexpr: Cannot open \"%s\" -- Break.\n",
	    name);
    goto error;
  }

  while((string = jsgetline(fp,GETLINE_RMEOL|GETLINE_COMMENT|
			    GETLINE_CONCAT|GETLINE_RMCONCAT))) {
    const char *string2;

    /*  Skip empty lines  */
    while(isspace(*(unsigned char *)string)) string++;

    if(*string == '\0') continue;

    if((expr = CompDefinition(string,(const Expression **)Fcts,&string2))
       == NULL) {
      fprintf(stderr,"\
ERROR jsexpr: Cannot compile function definition -- Break.\n");
      goto error;
    }

    if(*string2)
      fprintf(stderr,"WARNING: Garbage \"%s\" in expression -- Ignored.\n",
	      string2);

    /*  Check if function definition  */
    if(expr->EX_Name) {
      if(newFunction(expr)) goto error;
    } else if(mode & 2) {
      fprintf(stderr,"WARNING: Expressions not permitted in file -- Ignored.\n\
  >>%s\n",string);
      FreeExpression(expr);
    } else if(newExpression(NULL,expr))
      goto error;
  }

  if(fp != stdin) fclose(fp);

  return(0);
error:
  return(-1);
}

/*JS*********************************************************************
*   FINDZERO Find all or first zero point of the expression in the range
*    given by 'range' for variable 'vars[index]'. All other variables
*    have their fixed values.
*************************************************************************/

int findZero(struct Range *range,double *vars,int index,Expression *expr)

/************************************************************************/
{
  double oldVar,oldRes,res;
  int j,count;

  for(vars[index] = range->R_Values[0], j = 0, count = 0 ; ; ) {
    double zPoint;
    int flag = FALSE;

    res = EvalExpression(expr,vars,NULL,(const Expression **)Fcts);

    /*printf("%g %g: %g %g\n",oldVar,vars[index],oldRes,res);*/

    if(res == 0.0) {
      zPoint = vars[index];
      flag = TRUE;
    } else if(j > 0 &&
	      ((oldRes < 0.0 && res > 0.0) || (oldRes > 0.0 && res < 0.0))) {
      double save = vars[index];

      if(bracketize(expr,vars,index,oldVar,vars[index],oldRes,res))
	goto error;

      zPoint = vars[index];
      vars[index] = save;
      flag = TRUE;
    }

    /*  Zero point found  */
    if(flag) {
      if(count)
	printf(OutSeparator ? OutSeparator : DEFSEPARATOR,1 + count);

      printf(OutFormat,zPoint);
      count++;

      if(!(Mode & MODE_ALLZEROES)) break;
    }

    /*  Next value  */
    oldVar = vars[index];
    oldRes = res;
    j++;
    if(range->R_Max >= 0) {
      if(j >= range->R_Max) break;
      vars[index] = range->R_Values[j];
    } else {
      if(j >= -range->R_Max) break;
      vars[index] = range->R_Values[0] + j * range->R_Values[2];
    }
  }

  return(count);
error:
  return(-1);
}

/*JS*********************************************************************
*   BRACKETIZE Find the zero of the expression in the interval
*    [left,right] (fLeft, fRight being the values of the expression at
*    these points) by interval bracketizing. Is stupid, but works quite
*    fine.
*************************************************************************/

int bracketize(Expression *expr,double *vars,int index,
	       double left,double right,double fLeft,double fRight)

/************************************************************************/
{
  while(fabs(left - right) > (EPSILON * fabs(left + right))) {
    double fm;

    vars[index] = 0.5 * (left + right);
    fm = EvalExpression(expr,vars,NULL,(const Expression **)Fcts);

#ifdef DEBUG
    printf("f(%g)=%g,f(%g)=%g  => f(%g)=%g\n",left,fLeft,right,fRight,
           vars[index],fm);
#endif

    if((fLeft < 0.0) ? fm < 0.0 : fm > 0.0) {
      left  = vars[index];
      fLeft = fm;
    } else {
      right  = vars[index];
      fRight = fm;
    }

    if(fm == 0.0) break;
  }

  vars[index] = 0.5 * (left + right);

  return(0);
}
