/*JS*********************************************************************
*
*    Program : JSCOMPLEX
*    Language: ANSI-C / ANSI-C++
*    Author  : Joerg Schoen
*    Purpose : Definitions for complex numbers.
*
*    $Id: jscmplx.h,v 1.1 1995/02/01 10:26:49 joerg Stab joerg $
*
*************************************************************************/
#ifndef __JSCOMPLEX_H__ /*   File must not be included more than once!   */
#define __JSCOMPLEX_H__

#ifndef __cplusplus
/* **************************************************************** */
/* ****  Definition of complex data types. Taken from essl.h   **** */
/* **************************************************************** */
/*  To mask out when using essl, which defines the same types   */
#ifndef NO_DEFINE_COMPLEX
typedef union {
  struct {
    float  __re, __im;
  } __data;
  double __align;
} cmplx;

typedef union {
  struct {
    double __re, __im;
  } __data;
  double __align;
} dcmplx;

#define RE(x)  ((x).__data.__re)
#define IM(x)  ((x).__data.__im)
#endif

/* ***  Macros to do complex arithmetric more easier  *** */
/*  Set complex number  */
#define SETC(z,re,im) (RE(z) = (re) , IM(z) = (im))

/*  Calculate |z|^2  */
#define ABS2C(z)      (RE(z) * RE(z) + IM(z) * IM(z))

/* **************************************************** */
/*  Multiply with a real number  */
#define MULRE(z,r)  (RE(z) *= r,  IM(z) *= r)

/*  Set to real times complex  */
#define SETRECO(z,r,c) \
  (RE(z) = (r) * RE(c), \
   IM(z) = (r) * IM(c) )

/*  Set to complex times complex  */
#define SETCOCO(z,c1,c2) \
  (RE(z) = RE(c1) * RE(c2) - IM(c1) * IM(c2), \
   IM(z) = IM(c1) * RE(c2) + RE(c1) * IM(c2) )

/*  Set to complex times complex-conjugated  */
#define SETCOCCO(z,c1,c2) \
  (RE(z) = RE(c1) * RE(c2) + IM(c1) * IM(c2), \
   IM(z) = IM(c1) * RE(c2) - RE(c1) * IM(c2) )

/*  Set to real times complex times complex  */
#define SETRECOCO(z,r,c1,c2) \
  (RE(z) = (r) * (RE(c1) * RE(c2) - IM(c1) * IM(c2)), \
   IM(z) = (r) * (IM(c1) * RE(c2) + RE(c1) * IM(c2)) )

/*  Set to real times complex times complex-conjugated  */
#define SETRECOCCO(z,r,c1,c2) \
  (RE(z) = (r) * (RE(c1) * RE(c2) + IM(c1) * IM(c2)), \
   IM(z) = (r) * (IM(c1) * RE(c2) - RE(c1) * IM(c2)) )

/* **************************************************** */
/*  Add real times complex  */
#define ADDRECO(z,r,c) \
  (RE(z) += (r) * RE(c), \
   IM(z) += (r) * IM(c) )

/*  Add complex times complex  */
#define ADDCOCO(z,c1,c2) \
  (RE(z) += RE(c1) * RE(c2) - IM(c1) * IM(c2), \
   IM(z) += IM(c1) * RE(c2) + RE(c1) * IM(c2) )

/*  Add complex times complex-conjugated  */
#define ADDCOCCO(z,c1,c2) \
  (RE(z) += RE(c1) * RE(c2) + IM(c1) * IM(c2), \
   IM(z) += IM(c1) * RE(c2) - RE(c1) * IM(c2) )

/*  Add real times complex times complex  */
#define ADDRECOCO(z,r,c1,c2) \
  (RE(z) += (r) * (RE(c1) * RE(c2) - IM(c1) * IM(c2)), \
   IM(z) += (r) * (IM(c1) * RE(c2) + RE(c1) * IM(c2)) )

/*  Add real times complex times complex-conjugated  */
#define ADDRECOCCO(z,r,c1,c2) \
  (RE(z) += (r) * (RE(c1) * RE(c2) + IM(c1) * IM(c2)), \
   IM(z) += (r) * (IM(c1) * RE(c2) - RE(c1) * IM(c2)) )

#else
/***  Definition of complex data types in C++  ****/
#include <math.h>

class dcmplx
{
  double re,im;
public:
  /*   Basic function to get real and imaginary part  */
  double real() const;
  double imag() const;

  dcmplx();
  dcmplx(const dcmplx& y);
  dcmplx(double r,double i = 0.0);
  dcmplx& operator=(const dcmplx& y);

  dcmplx& operator+=(const dcmplx& y);
  dcmplx& operator+=(double y);
  dcmplx& operator-=(const dcmplx& y);
  dcmplx& operator-=(double y);
  dcmplx& operator*=(const dcmplx& y);
  dcmplx& operator*=(double y);
  dcmplx& operator/=(const dcmplx& y);
  dcmplx& operator/=(double y);
};

/*  General inline functions  */
inline double dcmplx::real() const { return re; }
inline double dcmplx::imag() const { return im; }
inline double real(const dcmplx& x) { return x.real(); } /* For convenience */
inline double imag(const dcmplx& x) { return x.imag(); } /* For convenience */

inline dcmplx::dcmplx() {}  /*  Default constructor is empty  */
inline dcmplx::dcmplx(const dcmplx& y): re(y.real()), im(y.imag()) { }
inline dcmplx::dcmplx(double r, double i): re(r), im(i) { }
inline dcmplx& dcmplx::operator=(const dcmplx& y)
{  re = y.real(); im = y.imag(); return *this; }

/* ***  Standard mathematical operations  *** */
inline dcmplx operator-(const dcmplx& x)  /*  unary "-"  */
{ return dcmplx(-x.real(),-x.imag()); }

inline dcmplx& dcmplx::operator+=(const dcmplx& y) /*  "+="  */
{  re += y.real();  im += y.imag(); return *this; }
inline dcmplx& dcmplx::operator+=(double y)
{  re += y; return *this; }
inline dcmplx operator+(const dcmplx& x,const dcmplx& y) /*  "+"  */
{ return dcmplx(x.real() + y.real(),x.imag() + y.imag()); }
inline dcmplx operator+(const dcmplx& x,double y)
{ return dcmplx(x.real() + y, x.imag()); }
inline dcmplx operator+(double x,const dcmplx& y)
{ return dcmplx(x + y.real(), y.imag()); }

inline dcmplx& dcmplx::operator-=(const dcmplx& y) /*  "-="  */
{  re -= y.real();  im -= y.imag(); return *this; }
inline dcmplx&  dcmplx::operator-=(double y)
{  re -= y; return *this; }
inline dcmplx operator-(const dcmplx& x,const dcmplx& y) /*  "-"  */
{ return dcmplx(x.real() - y.real(), x.imag() - y.imag()); }
inline dcmplx operator-(const dcmplx& x,double y)
{ return dcmplx(x.real() - y, x.imag()); }
inline dcmplx operator-(double x,const dcmplx& y)
{ return dcmplx(x - y.real(),-y.imag()); }

inline dcmplx& dcmplx::operator*=(const dcmplx& y) /*  "*="  */
{
  double r = re * y.real() - im * y.imag();
  im = re * y.imag() + im * y.real();
  re = r;
  return *this;
}
inline dcmplx& dcmplx::operator*=(double y)
{  re *=  y; im *=  y; return *this; }
inline dcmplx operator*(const dcmplx& x,const dcmplx& y) /*  "*"  */
{ return dcmplx(x.real() * y.real() - x.imag() * y.imag(),
		x.real() * y.imag() + x.imag() * y.real());
}
inline dcmplx operator*(const dcmplx& x,double y)
{ return dcmplx(x.real() * y,x.imag() * y); }
inline dcmplx operator*(double x,const dcmplx& y)
{ return dcmplx(x * y.real(),x * y.imag()); }

inline dcmplx& dcmplx::operator/=(const dcmplx& y) /*  "/="  */
{
  double f,r,den;

  if(fabs(y.real()) < fabs(y.imag())) {
    f = y.real() / y.imag();
    den = 1.0 / (y.imag() + f * y.real());

    r  = den * (re * f + im);
    im = den * (im * f - re);
    re = r;
    return *this;
  } else {
    f = y.imag() / y.real();
    den = 1.0 / (y.real() + f * y.imag());

    r  = den * (re + f * im),
    im = den * (im - f * re);
    re = r;
    return *this;
  }
}
inline dcmplx& dcmplx::operator/=(double y)
{
  double temp = 1.0 / y;
  re *= temp;
  im *= temp;
  return *this;
}
inline dcmplx operator/(const dcmplx& x,const dcmplx& y) /*  "/"  */
{
  double f,den;

  if(fabs(y.real()) < fabs(y.imag())) {
    f = y.real() / y.imag();
    den = 1.0 / (y.imag() + f * y.real());
    return dcmplx(den * (x.real() * f + x.imag()),
		  den * (x.imag() * f - x.real()));
  } else {
    f = y.imag() / y.real();
    den = 1.0 / (y.real() + f * y.imag());
    return dcmplx(den * (x.real() + f * x.imag()),
		  den * (x.imag() - f * x.real()));
  }
}
inline dcmplx operator/(dcmplx& x,double y)
{
  double temp = 1.0 / y;
  return dcmplx(x.real() * temp,x.imag() * temp);
}

/*  For logical operations  */
inline int operator==(const dcmplx& x,const dcmplx& y)
{ return x.real() == y.real() && x.imag() == y.imag(); }
inline int operator==(const dcmplx& x,double y)
{ return x.imag() == 0.0 && x.real() == y; }
inline int operator!=(const dcmplx& x,const dcmplx& y)
{ return x.real() != y.real() || x.imag() != y.imag(); }
inline int operator!=(const dcmplx& x,double y)
{ return x.imag() != 0.0 || x.real() != y; }

/*  Various operations  */
inline dcmplx conj(const dcmplx& x)
{ return dcmplx(x.real(), -x.imag()); }
inline double norm(const dcmplx& x)
{ return (x.real() * x.real() + x.imag() * x.imag()); }
inline double arg(const dcmplx& x)
{ return atan2(x.imag(), x.real()); }
inline dcmplx polar(double r, double t)
{ return dcmplx(r * cos(t), r * sin(t)); }

/* ********************************************************** */
/* ***  Macros for compatibility. Won't work completely!  *** */
#define RE(z)                      (::real(z))
#define IM(z)                      (::imag(z))
#define SETC(z,re,im)              (z = dcmplx(re,im))
#define ABS2C(z)                   (::norm(z))

#define MULRE(z,r)                 (z *= (r))
#define SETRECO(z,r,c)             (z = (r) * (c))
#define SETCOCO(z,c1,c2)           (z = (c1) * (c2))
#define SETCOCCO(z,c1,c2)          (z = (c1) * conj(c2))
#define SETRECOCO(z,r,c1,c2)       (z = (r) * (c1) * (c2))
#define SETRECOCCO(z,r,c1,c2)      (z = (r) * (c1) * conj(c2))

#define ADDRECO(z,r,c)             (z += (r) * (c))
#define ADDCOCO(z,c1,c2)           (z += (c1) * (c2))
#define ADDCOCCO(z,c1,c2)          (z += (c1) * conj(c2))
#define ADDRECOCO(z,r,c1,c2)       (z += (r) * (c1) * (c2))
#define ADDRECOCCO(z,r,c1,c2)      (z += (r) * (c1) * conj(c2))

#endif
#endif
