/*JS*********************************************************************
*
*    Program : FPLOTTENSOR
*    Language: ANSI-C
*    Author  : Joerg Schoen
*    Purpose : Routines for plotting data to a file.
*    Part    : Plot multidimensional tensor data - double precision.
*
*      Format of hunks:
*       HUNK_VECTOR(XY):
*        hunk header:
*                ulong  hunk_type   (== HUNK_VECTOR or HUNK_VECTORXY)
*                ulong  hunk_length (in "sizeof(long)" without hunk header).
*        hunk content:
*                int    type
*                ulong  dim
*                double xMin, xMax
*                double fMin, fMax
*         dim  * double data
*         dim  * double xData   Only in case of HUNK_VECTORXY
*
*       HUNK_TENSOR(XY):
*        hunk header:
*                ulong  hunk_type   (== HUNK_TENSOR or HUNK_TENSORXY)
*                ulong  hunk_length (in "sizeof(long)" without hunk header).
*        hunk content:
*                int    type
*                int    nDim
*         nDim * ulong  dim[i]
*         [      ulong  zero ]  only if nDim is odd
*         nDim * double xMin[i], xMax[i]
*                double fMin, fMax
*         fDim * double data    (fDim = dim[0] * dim[1] * .. * dim[nDim])
*         dim[0]*double x0Data   Only in case of HUNK_TENSORXY
*           ...
*         dim[nDim-1]*
*                double xnData   Only in case of HUNK_TENSORXY
*
*       HUNK_AXEDATA:
*        hunk header:
*                ulong  hunk_type   (== HUNK_AXEDATA)
*                ulong  hunk_length (in "sizeof(long)" without hunk header).
*        hunk content:
*                int    index
*                ulong  len
*         len  * double axeData
*
*      Comments:
*       If no axe values are given, only a single hunk with type HUNK_VECTOR
*       or HUNK_TENSOR is written. If all axes have values, a single hunk
*       with type HUNK_VECTORXY or HUNK_TENSORXY is written. In all other
*       cases a hunk with type HUNK_TENSOR is written followed by separate
*       hunks with type HUNK_AXEDATA for every axe that has values.
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id: fplottensor.c,v 1.4 1996/10/21 16:02:58 joerg Stab joerg $";
#endif

/*********     INCLUDES 					*********/
#include <jsalloca.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

#include <fplot.h>

/*********     DEFINES						*********/
#ifndef THISNAME
# define THISNAME    fplotTensor
#endif

#ifndef THISNAME2
# define THISNAME2   fplotTensorV
#endif

#ifndef THISHUNK
# define THISHUNK	 HUNK_TENSOR
#endif

#ifndef THIS1DIMHUNK
# define THIS1DIMHUNK	 HUNK_VECTOR
#endif

#ifndef THISHUNKXY
# define THISHUNKXY	 HUNK_TENSORXY
#endif

#ifndef THIS1DIMHUNKXY
# define THIS1DIMHUNKXY  HUNK_VECTORXY
#endif

#ifndef THISAXEHUNK
# define THISAXEHUNK     HUNK_AXEDATA
#endif

#ifndef PRECISION
# define PRECISION double
#endif

#define Prototype extern
/*********     PROTOTYPES					*********/
Prototype int		 fplotTensor(FPFile *fp,int type,int nDim,
				     const double **axeData,
				     const double *data,
				     unsigned long *dims,long *strides,
				     const double *boundaries,
				     double fMin,double fMax);

Prototype int		 fplotTensorV(FPFile *fp,int type,int nDim, ...);

/*JS*********************************************************************
*   Writes multidimensional data to file with optional values for axes.
*************************************************************************/

int THISNAME(FPFile *fp,int type,int nDim,
	     const PRECISION **axeData,const PRECISION *data,
	     unsigned long *dims,long *strides,
	     const PRECISION *boundaries,PRECISION fMin,PRECISION fMax)

/************************************************************************/
{
  OutputHeader oheader;
static unsigned long Zero = 0;
  int i,axeFlag;
  long len;
  unsigned long dim;

  /*  Calculate dimension  */
  for(dim = dims[0], i = 1 ; i < nDim ; i++) dim *= dims[i];

  /*   Set up header length and type  */
  if(nDim == 1)
    len = sizeof(int) + sizeof(unsigned long);
  else
    len = 2 * sizeof(int) + ((nDim + 1) & ~1) * sizeof(unsigned long);
  len += 2 * (nDim + 1) * sizeof(*boundaries) + dim * sizeof(*data);

  if(axeData) {
    int nAxe;

    /*  Check if all axes have values or only some  */
    for(axeFlag = 0, i = 0 ; i < nDim ; i++)
      axeFlag |= (axeData[i] != NULL) ? 2 : 1;

    if(axeFlag == 2) { /*  All axe values given, write special hunk  */
      oheader.OH_Type = (nDim != 1) ? THISHUNKXY : THIS1DIMHUNKXY;

      for(nAxe = 0, i = 0 ; i < nDim ; i++) nAxe += dims[i];
      len += nAxe * sizeof(**axeData);
    } else
      oheader.OH_Type = (nDim != 1) ? THISHUNK : THIS1DIMHUNK;
  } else {
    oheader.OH_Type = (nDim != 1) ? THISHUNK : THIS1DIMHUNK;
  }
  oheader.OH_Length = HUNKLENGTH(len);

  /*  Write out Header and data. Data comes first, then values	*/
  /*   for axes. A special form is used if nDim is 1.		*/
  if(fwrite(&oheader,sizeof(oheader),1,fp->FF_File) != 1 ||
     fwrite(&type,sizeof(type),1,fp->FF_File) != 1 ||
     (nDim != 1 && fwrite(&nDim,sizeof(nDim),1,fp->FF_File) != 1) ||
     fwrite(dims,sizeof(*dims),nDim,fp->FF_File) != nDim ||
     ((nDim & 1) && nDim != 1 &&
      fwrite(&Zero,sizeof(Zero),1,fp->FF_File) != 1) ||
     fwrite(boundaries,sizeof(*boundaries),2 * nDim,fp->FF_File) !=
     (2 * nDim) ||
     fwrite(&fMin,sizeof(fMin),1,fp->FF_File) != 1 ||
     fwrite(&fMax,sizeof(fMax),1,fp->FF_File) != 1)
    goto ioerror;

  if(strides) {
    int *ind,*offs;

#ifdef CONFIG_USE_ALLOCA
    if((ind = (int *)alloca(2 * nDim * sizeof(*ind))) == NULL) goto error;
#else
    if((ind = (int *)malloc(2 * nDim * sizeof(*ind))) == NULL) goto error;
#endif
    offs = &ind[nDim];

    /*	Preset values for loop and set strides if not done  */
    for(i = 0 ; i < nDim ; i++) {
      ind[i] = offs[i] = 0;

      if(strides[i] < 0) /*  Negative value means preset  */
	strides[i] = (i > 0) ? (strides[i - 1] * dims[i - 1]) : 1;
    }
    for(;;) {
      /*  Write data  */
      if(fwrite(&data[offs[0]],sizeof(*data),1,fp->FF_File) != 1) goto ioerror;

      for(i = 0 ; i < nDim ; i++) {
	ind[i]++;

	if(ind[i] < dims[i]) {
	  offs[i] += strides[i];
	  break;
	}
      }
      if(i == nDim) break;
      while(i-- > 0) {
	ind[i] = 0;
	offs[i] = offs[i + 1];
      }
    }

#ifndef CONFIG_USE_ALLOCA
    free(ind);
#endif
  } else {
    /*	Write in one call  */
    if(fwrite(data,sizeof(*data),dim,fp->FF_File) != dim) goto ioerror;
  }

  /*  Write axe values	*/
  if(axeData) {
    if(axeFlag == 2) {
      /*  Axe data appended to hunk  */
      for(i = 0 ; i < nDim ; i++)
	if(fwrite(axeData[i],sizeof(*axeData[0]),dims[i],fp->FF_File) !=
	   dims[i])
	  goto ioerror;
    } else {
      /*  Write hunk for every axe data given  */
      oheader.OH_Type = THISAXEHUNK;
      for(i = 0 ; i < nDim ; i++) {
	if(axeData[i] == NULL) continue;

	/*   Write out padding bytes from previous hunk  */
	if((len = oheader.OH_Length * HUNKROUND - len) > 0 &&
	   fwrite("\0\0\0",1,len,fp->FF_File) != len) goto ioerror;

	len = sizeof(i) + sizeof(dims[i]) + dims[i] * sizeof(*axeData[i]);
	oheader.OH_Length = HUNKLENGTH(len);

	if(fwrite(&oheader,sizeof(oheader),1,fp->FF_File) != 1 ||
	   fwrite(&i,sizeof(i),1,fp->FF_File) != 1 ||
	   fwrite(&dims[i],sizeof(dims[i]),1,fp->FF_File) != 1 ||
	   fwrite(axeData[i],sizeof(*axeData[0]),dims[i],fp->FF_File) !=
	   dims[i])
	  goto ioerror;
      }
    }
  }

  /*   Write out padding bytes	  */
  if((len = oheader.OH_Length * HUNKROUND - len) > 0 &&
     fwrite("\0\0\0",1,len,fp->FF_File) != len) goto ioerror;

  return(0);
ioerror:
error:
  return(-1);
}

/*JS*********************************************************************
*   For convenience: All values passed directly, not in vectors. Called
*    via:
*      fplot(S)TensorV(fp,type,nDim,axeData1,..,axeDatan,data,
*		       dim1,..,dimn,stride1,..,striden,
*		       x1min,x1max,..,xnmin,xnmax,fmin,fmax);
*************************************************************************/

int THISNAME2(FPFile *fp,int type,int nDim, ...)

/************************************************************************/
{
  va_list ap;
  const PRECISION **axeData,*data;
  unsigned long *dims;
  long *strides;
  PRECISION *boundaries,fMin,fMax;
  int i;

  if((boundaries = (PRECISION *)
#ifdef CONFIG_USE_ALLOCA
      alloca(nDim * (2 * sizeof(PRECISION) + sizeof(const PRECISION *) +
		     2 * sizeof(long *)))
#else
      malloc(nDim * (2 * sizeof(PRECISION) + sizeof(const PRECISION *) +
		     2 * sizeof(long *)))
#endif
      ) == NULL) goto error;
  axeData = (const PRECISION **) &boundaries[2 * nDim];
  dims	  = (unsigned long *) &axeData[nDim];
  strides = (long *) &dims[nDim];

  /*  Now collect all arguments  */
  va_start(ap,nDim);
  for(i = 0 ; i < nDim ; i++) axeData[i] = va_arg(ap,const PRECISION *);
  data = va_arg(ap,const PRECISION *);

  for(i = 0 ; i < nDim ; i++) dims[i]    = va_arg(ap,unsigned long);
  for(i = 0 ; i < nDim ; i++) strides[i] = va_arg(ap,long);

  for(i = 0 ; i < nDim ; i++) {
    boundaries[2 * i]	  = va_arg(ap,PRECISION);
    boundaries[2 * i + 1] = va_arg(ap,PRECISION);
  }
  fMin = va_arg(ap,PRECISION);
  fMax = va_arg(ap,PRECISION);
  va_end(ap);

  i = THISNAME(fp,type,nDim,axeData,data,dims,strides,boundaries,fMin,fMax);

#ifndef CONFIG_USE_ALLOCA
  free(boundaries);
#endif

  return(i);
error:
  return(-1);
}
