/*JS*********************************************************************
*
*    Program : FPLOTREAD
*    Language: ANSI-C
*    Author  : Joerg Schoen
*    Purpose : Routines for reading plot data from a file.
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id: fplotread.c,v 1.11 1996/10/21 16:01:58 joerg Stab joerg $";
#endif

/*********     INCLUDES 					*********/
#include <jsalloca.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <jsconfig.h>
#include <jssubs.h>

#include <fplot.h>

/*********     DEFINES						*********/
#ifdef CONFIG_NO_STRDUP
# define strdup  mystrdup  /*  strdup is not POSIX conformant  */
#endif

#define VERBOSE

#ifndef FPLOT_MODE_READ
/*  START-DEFINITIONS  */
/*  Defines for routine 'fplotRead'  */
#define FPLOT_MODE_READ 	 (1<<0)
#define FPLOT_MODE_NOGETMINMAX	 (1<<1)
#define FPLOT_MODE_CASTTOFLOAT	 (1<<2)
#define FPLOT_MODE_CASTTODOUBLE  (1<<3)
#define FPLOT_MODE_GENAXEDATA    (1<<4)
#define FPLOT_MODE_SORTDATA      (1<<5)

#define FPLOT_MODE_USER          (1<<8)

/*  Set from routine 'fplotRead' in FPD_Flags field  */
#define FPLOT_FLAGS_MASK    0xff
#define FPLOT_FLAGS_SINGLE  (1<<8)  /*  Values are single precision  */

/*  Structure to store the contents of one hunk (double precision)  */
typedef struct FPData FPData;
struct FPData {
  FPData        *FPD_Next;
  unsigned long  FPD_Flags;
  int            FPD_Type,FPD_NDim;
  unsigned long *FPD_Dims;
  void          *FPD_Boundaries;
  double         FPD_FMin,FPD_FMax;
  void         **FPD_AxeValues;
  void          *FPD_Data;
};

/*  Structure to store the contents of one sequence    */
/*   of hunks (stopped with a END hunk).               */
typedef struct {
  int	   FP_User;
  char	  *FP_Title;
  FPData   FP_Content;
  int	   FP_NAxes;
  char	 **FP_Axes;
} FPPicture;

/*  Some useful abbreviations	 */
#define FP_Next    FP_Content.FPD_Next
#define FP_Flags   FP_Content.FPD_Flags
/*  END-DEFINITIONS  */
#endif

/* ERROR-DEFINITIONS from fplotRead label _ERR_FPLTREAD ord 2
   0-dimensional data found
   Not a valid hunk
   Inconsistent Hunk length
   Axe values without data found
   Axe data given twice
   Mixed single and double precision values
*/

#define Prototype extern
/*********     PROTOTYPES					*********/
Prototype FPPicture	*fplotRead(FPFile *ffp,int mode);
Prototype void		 fplotFreepic(FPPicture *picture);

static int               dummyFseek(FILE *fp,long offset);

/*
 * Explanation of variable mode in fplotRead:
 *   Bit 10: Show position information.
 *   Bit 11: Show found hunks.
 *   Bit 12: Show text hunks.
 *   Bit 13: Show vectors and arrays too.
 */

/*JS*********************************************************************
*   Read next hunks and stores contents in a structure until end hunk is
*    found. On error, NULL is returned.
*************************************************************************/

FPPicture *fplotRead(FPFile *ffp,int mode)

/************************************************************************/
{
  OutputHeader oheader;
  FPPicture *picture;
  FPData *picData;
  char	*text;
  int countHunks;
  unsigned long len;
  long fPos;

  /*	Get a picture structure and initialize it   */
  if((picture = (FPPicture *)malloc(sizeof(*picture))) == NULL) goto nomem;
  picture->FP_Title = NULL;
  picture->FP_NAxes = 0;
  picture->FP_Axes = NULL;
  picture->FP_Content.FPD_Next = NULL;
  picture->FP_Content.FPD_NDim = 0;
  picture->FP_Content.FPD_Dims = NULL;
  picture->FP_Content.FPD_Boundaries = NULL;
  picture->FP_Content.FPD_AxeValues = NULL;
  picture->FP_Content.FPD_Data = NULL;

  /*  Read hunks until end hunk  */
  if(!(ffp->FF_Flags & FPLOT_FLAG_NOFSEEK) && (fPos = ftell(ffp->FF_File)) < 0)
    goto ioerror;

  for(countHunks = 0 ; ; countHunks++) {
    int flag,nDim,type;

#ifdef VERBOSE
    /*	 Print out current position    */
    if((mode & FPLOT_MODE_READ) && !(ffp->FF_Flags & FPLOT_FLAG_NOFSEEK) &&
       (mode & 1024))
      printf("0x%lx: \n",fPos);
#endif

    /*  Read hunk header  */
    if(fread(&oheader,sizeof(oheader),1,ffp->FF_File) != 1) {
      /*  If user forgot end hunk, don't worry  */
      if(feof(ffp->FF_File) && countHunks) goto endfound;

      goto error;
    }

    /*  Check for correct hunks  */
redo:
    if(oheader.OH_Type < HUNK_MIN || oheader.OH_Type > HUNK_MAX) {
      /*  Accept garbage due to concatenation of two fplot files  */
      if(oheader.OH_Type == MAGIC_PLOT) {
	oheader.OH_Type = oheader.OH_Length;
	if(fread(&oheader.OH_Length,sizeof(oheader.OH_Length),1,ffp->FF_File)
	   != 1) {
	  if(feof(ffp->FF_File) && countHunks) goto endfound;
	  goto error;
	}
	goto redo;
      }

      JSErrNo = _ERR_FPLTREAD + 1;
      goto error;
    }

    len = oheader.OH_Length * HUNKROUND;

    /*  Set up end of current hunk  */
    if(!(ffp->FF_Flags & FPLOT_FLAG_NOFSEEK)) fPos += sizeof(oheader) + len;

    /*	Reading or seeking?  */
    if(!(mode & FPLOT_MODE_READ)) {
      /*  Just fseek this hunk	*/
      if(len &&
	 ((ffp->FF_Flags & FPLOT_FLAG_NOFSEEK) ? dummyFseek(ffp->FF_File,len) :
	  fseek(ffp->FF_File,len,SEEK_CUR))) goto ioerror;

      /*  Test if Hunk end found  */
      if(oheader.OH_Type == HUNK_END) break;

      /*  Do not read contents	*/
      continue;
    }

    /*	Now read Hunks	*/
    nDim = 0;
    flag = 0;
#define FLAG_AXEVALUES	 (1<<4)
#define FLAG_SINGLEPREC  (1<<5)

    switch(oheader.OH_Type) {
    case HUNK_END:
#ifdef VERBOSE
      if(mode & (1024 + 2048)) printf("Hunk End found.\n");
#endif

      /*   Stop reading hunks	 */
      goto endfound;
    case HUNK_AXES:
    case HUNK_TEXT:
#ifdef VERBOSE
      if(mode & (1024 + 2048))
	printf("Hunk %s found.\n",
	       oheader.OH_Type == HUNK_TEXT ? "Text" : "Axes");
#endif

      if(len == 0) {
	fprintf(stderr,"\
WARNING fplotRead: Hunk contains no data -- Skipped.\n");
	break;
      }

      /*  Get memory for text	*/
      if((text = (char *)malloc(len + 1)) == NULL) goto nomem;

      /*   Read in contents	*/
      if(fread(text,1,len,ffp->FF_File) != len) goto error;
      text[len] = '\0'; /*  trailing zero might have got lost  */

      if(oheader.OH_Type == HUNK_TEXT) {
	if(picture->FP_Title) {
	  fprintf(stderr,"\
WARNING fplotRead: Text Hunk found twice -- second ignored:\n%s\n",
		 text);
	  free(text);
	} else {
	  /*   Save in picture structure   */
	  picture->FP_Title = text;
	}
      } else if(picture->FP_Axes) {
	char *string;

	fprintf(stderr,"\
WARNING fplotRead: Hunk Axes found twice -- ignored:\n");

	for(string = text ; string < text + len ;
	    string += strlen(string) + 1) {
	  printf("%s\n",string);
	}

	free(text);
      } else {
	/*  HUNK_AXES: Scan through string and set values for all axes  */
	char *string;
	int i;

	for(picture->FP_NAxes = 0, string = text ; string < text + len ;
	    string += strlen(string) + 1)
	  picture->FP_NAxes++;

	if((picture->FP_Axes = (char **)
	    malloc(picture->FP_NAxes * sizeof(*(picture->FP_Axes)))) == NULL)
	  goto nomem;

	for(i = 0, string = text ; string < text + len ; i++,
	    string += strlen(string) + 1) {
	  /*  Make every axe a single copy  */
	  picture->FP_Axes[i] = *string ? strdup(string) : (char *)NULL;
	}

	/*  Delete trailing empty axes  */
	while(picture->FP_NAxes > 0 &&
	      (picture->FP_Axes[picture->FP_NAxes - 1] == NULL ||
	       *picture->FP_Axes[picture->FP_NAxes - 1] == '\0'))
	  picture->FP_NAxes--;

	free(text);
      }
      break;
    case HUNK_VECTORXY:
    case HUNK_SVECTORXY:
      flag |= FLAG_AXEVALUES;
    case HUNK_VECTOR:
    case HUNK_SVECTOR:
      if(oheader.OH_Type == HUNK_SVECTORXY || oheader.OH_Type == HUNK_SVECTOR)
	flag |= FLAG_SINGLEPREC;

      /*  Only type field to be read  */
      if(fread(&type,sizeof(type),1,ffp->FF_File) != 1) goto ioerror;
      nDim = 1;
      break;
    case HUNK_TENSORXY:
    case HUNK_STENSORXY:
      flag |= FLAG_AXEVALUES;
    case HUNK_TENSOR:
    case HUNK_STENSOR:
      if(oheader.OH_Type == HUNK_STENSORXY || oheader.OH_Type == HUNK_STENSOR)
	flag |= FLAG_SINGLEPREC;

      /*  Read type and nDim field  */
      if(fread(&type,sizeof(type),1,ffp->FF_File) != 1 ||
	 fread(&nDim,sizeof(nDim),1,ffp->FF_File) != 1)
	goto ioerror;

      if(nDim == 0) {
	JSErrNo = _ERR_FPLTREAD + 0;
	goto error;
      }

      break;
    case HUNK_SAXEDATA:
      flag |= FLAG_SINGLEPREC;
    case HUNK_AXEDATA:
      /*  Axe data already allocated?  */
      if(picData->FPD_AxeValues == NULL) {
	int i;

	if(picData->FPD_Data == NULL) {
	  JSErrNo = _ERR_FPLTREAD + 3;
	  goto error;
	}

	if((picData->FPD_AxeValues = (void **)
	    malloc(picData->FPD_NDim * sizeof(*picData->FPD_AxeValues)))
	   == NULL) goto nomem;

	for(i = 0 ; i < picData->FPD_NDim ; i++)
	  picData->FPD_AxeValues[i] = NULL;
      }

      {
	int index,n,size;

	/*  Read index and number of this axe data  */
	if(fread(&index,sizeof(index),1,ffp->FF_File) != 1 ||
	   fread(&n,sizeof(n),1,ffp->FF_File) != 1) goto ioerror;

	/*  Make some consistency checks  */
	if(picData->FPD_AxeValues[index] != NULL) {
	  JSErrNo = _ERR_FPLTREAD + 4;
	  goto error;
	}

	if(flag & FLAG_SINGLEPREC) {
	  size = sizeof(float);
	  if(!(picData->FPD_Flags & FPLOT_FLAGS_SINGLE)) {
	    JSErrNo = _ERR_FPLTREAD + 5;
	    goto error;
	  }
	} else {
	  size = sizeof(double);
	  if(picData->FPD_Flags & FPLOT_FLAGS_SINGLE) {
	    JSErrNo = _ERR_FPLTREAD + 5;
	    goto error;
	  }
	}

	/*  Get data  */
	if((picData->FPD_AxeValues[index] = (void *)
	    malloc(picData->FPD_Dims[index] * size)) == NULL)
	  goto nomem;

	if(fread(picData->FPD_AxeValues[index],size,
		 picData->FPD_Dims[index],ffp->FF_File) !=
	   picData->FPD_Dims[index])
	  goto ioerror;
      }

      break;
    default:
      fprintf(stderr,"\
WARNING fplotRead: Unknown Hunk Type '%#lx' -- Skipped.\n",oheader.OH_Type);

      /*   Just fseek this hunk    */
      if((ffp->FF_Flags & FPLOT_FLAG_NOFSEEK) ?
	 dummyFseek(ffp->FF_File,oheader.OH_Length * HUNKROUND) :
	 fseek(ffp->FF_File,oheader.OH_Length * HUNKROUND,SEEK_CUR)) goto ioerror;
      break;
    }

    if(nDim > 0) {
      unsigned long dim;
      int i,size;

      /*  Set up current picture data structure    */
      picData = &picture->FP_Content;

      if(picture->FP_Content.FPD_Data) {
	/*  Check compatibility of data (same dimension)  */
	if(picture->FP_Content.FPD_NDim != nDim) {
	  fprintf(stderr,"\
WARNING fplotRead: Extra data not compatible with previous (nDim=%d != %d) -- ignored.\n",
		 nDim,picture->FP_Content.FPD_NDim);
	  break;
	}

	/*   Go to end of linked list	*/
	while(picData->FPD_Next) picData = picData->FPD_Next;

	/*   Allocate memory for new structure	 */
	if((picData->FPD_Next = (FPData *)malloc(sizeof(*picData))) == NULL)
	  goto nomem;

	/*  Link at end of list   */
	picData = picData->FPD_Next;
      }
      picData->FPD_Next = NULL;
      picData->FPD_Type = type;
      picData->FPD_NDim = nDim;

      if(flag & FLAG_SINGLEPREC) {
	size = sizeof(float);
	picData->FPD_Flags = FPLOT_FLAGS_SINGLE;
      } else {
	size = sizeof(double);
        picData->FPD_Flags = 0;
      }

      if((picData->FPD_Dims = (unsigned long *)
	  malloc(nDim * sizeof(*picData->FPD_Dims))) == NULL) goto nomem;
      if((picData->FPD_Boundaries = malloc(nDim * 2 * size)) == NULL)
	goto nomem;

      if(fread(picData->FPD_Dims,sizeof(*picData->FPD_Dims),nDim,ffp->FF_File)
	 != nDim) goto ioerror;

      if((nDim & 1) && nDim != 1) {
	unsigned long dummy;

	if(fread(&dummy,sizeof(dummy),1,ffp->FF_File) != 1) goto ioerror;
      }

      if(fread(picData->FPD_Boundaries,size,2 * nDim,ffp->FF_File)
	 != 2 * nDim ||
	 fread(&picData->FPD_FMin,size,1,ffp->FF_File) != 1 ||
	 fread(&picData->FPD_FMax,size,1,ffp->FF_File) != 1)
	goto ioerror;

      if(picData->FPD_Flags & FPLOT_FLAGS_SINGLE) {
        /*  Cast fmin and fmax to double precision  */
	picData->FPD_FMin = *(float *)&picData->FPD_FMin;
	picData->FPD_FMax = *(float *)&picData->FPD_FMax;
      }

      /*  Calculate needed space  */
      for(dim = 1, i = 0 ; i < nDim ; i++) dim *= picData->FPD_Dims[i];

      /*  Allocate and read data  */
      if((picData->FPD_Data = malloc(dim * size)) == NULL) goto nomem;
      if(fread(picData->FPD_Data,size,dim,ffp->FF_File) != dim)
	goto ioerror;

      if(flag & FLAG_AXEVALUES) {
	if((picData->FPD_AxeValues = (void **)
	    malloc(nDim * sizeof(*picData->FPD_AxeValues))) == NULL)
	  goto nomem;

	/*  Set up axe values pointer and read axe values  */
	for(i = 0 ; i < nDim ; i++) {
	  if((picData->FPD_AxeValues[i] = (void *)
	     malloc(picData->FPD_Dims[i] * size)) == NULL)
	    goto nomem;

	  if(fread(picData->FPD_AxeValues[i],size,picData->FPD_Dims[i],ffp->FF_File) !=
	     picData->FPD_Dims[i])
	    goto ioerror;
	}
      } else {
	picData->FPD_AxeValues = NULL;
      }

      /*  Set up boundaries for function values  */
      if(picData->FPD_FMin == picData->FPD_FMax &&
	 !(mode & FPLOT_MODE_NOGETMINMAX)) {
	if(picData->FPD_Flags & FPLOT_FLAGS_SINGLE) {
	  float fMin,fMax;

	  getMinMaxS((float *)picData->FPD_Data,dim,&fMin,&fMax);
	  picData->FPD_FMin = fMin;
	  picData->FPD_FMax = fMax;
	} else {
	  getMinMax((double *)picData->FPD_Data,dim,&picData->FPD_FMin,
		    &picData->FPD_FMax);
	}
      }
    }

    /*  Check if hunk length compatible with read data  */
    if(!(ffp->FF_Flags & FPLOT_FLAG_NOFSEEK)) {
      long fPos2;

      if((fPos2 = ftell(ffp->FF_File)) < 0) goto ioerror;

      if(fPos2 > fPos) {
	JSErrNo = _ERR_FPLTREAD + 2;
	goto error;
      } else if(fPos2 < fPos) {
	fprintf(stderr,"\
WARNING fplotRead: Hunk contains %ld garbage bytes -- Skipped.\n",
		fPos - fPos2);
	if(fseek(ffp->FF_File,fPos,SEEK_SET)) goto ioerror;
      }
    }
  }

endfound:
  if(!(mode & FPLOT_MODE_READ)) {
    free(picture);
    return(NULL);
  }

  /*  Any axe boundary values still to determine?  */
  if(!(mode & FPLOT_MODE_NOGETMINMAX)) {
    for(picData = &picture->FP_Content ; picData ;
	picData = picData->FPD_Next) {
      int i;

      for(i = 0 ; i < picData->FPD_NDim ; i++) {
	if(picData->FPD_AxeValues == NULL || picData->FPD_AxeValues[i] == NULL)
	  continue;

	if(picData->FPD_Flags & FPLOT_FLAGS_SINGLE) {
	  float *pMin,*pMax;

	  pMin = &((float *)picData->FPD_Boundaries)[2 * i];
	  pMax = &((float *)picData->FPD_Boundaries)[2 * i + 1];
	  if(*pMin == *pMax)
	    getMinMaxS((float *)(picData->FPD_AxeValues[i]),
		       picData->FPD_Dims[i],pMin,pMax);
	} else {
	  double *pMin,*pMax;

	  pMin = &((double *)picData->FPD_Boundaries)[2 * i];
	  pMax = &((double *)picData->FPD_Boundaries)[2 * i + 1];
	  if(*pMin == *pMax)
	    getMinMax((double *)(picData->FPD_AxeValues[i]),
		      picData->FPD_Dims[i],pMin,pMax);
	}
      }
    }
  }

#ifdef VERBOSE
  if((mode & (4096 + 8192))) {
    /*	 Print out title    */
    if(picture->FP_Title) {
      char *ptr;

      printf("# Title:\n#");
      for(ptr = picture->FP_Title ; *ptr ; ptr++) {
	putchar(*ptr);
	if(*ptr == '\n' && ptr[1]) putchar('#');
      }
      printf("\n");
    }

    /*	 Print out Data     */
    for(picData = &picture->FP_Content ; picData ;
	picData = picData->FPD_Next) {
      int i;

      if(picData->FPD_Data == NULL) {
	printf("  WARNING fplotRead: Block without data found.\n");
	continue;
      }

      printf("#  Content: %ld",picData->FPD_Dims[0]);
      for(i = 1 ; i < picData->FPD_NDim ; i++)
	printf(" x %ld",picData->FPD_Dims[i]);

      if(picData->FPD_Flags & FPLOT_FLAGS_SINGLE) {
	printf(" floats %s axe values.\n",
	       picData->FPD_AxeValues ? "with" : "without");
	printf("#    Interval: [%g,%g]",((float *)picData->FPD_Boundaries)[0],
	       ((float *)picData->FPD_Boundaries)[1]);
	for(i = 1 ; i < picData->FPD_NDim ; i++)
	  printf(" x [%g,%g]",((float *)picData->FPD_Boundaries)[2 * i],
	       ((float *)picData->FPD_Boundaries)[2 * i + 1]);
      } else {
	printf(" doubles %s axe values.\n",
	       picData->FPD_AxeValues ? "with" : "without");
	printf("#    Interval: [%g,%g]",((double *)picData->FPD_Boundaries)[0],
	       ((double *)picData->FPD_Boundaries)[1]);
	for(i = 1 ; i < picData->FPD_NDim ; i++)
	  printf(" x [%g,%g]",((double *)picData->FPD_Boundaries)[2 * i],
		 ((double *)picData->FPD_Boundaries)[2 * i + 1]);
      }
      printf(" -- [%g,%g]\n",picData->FPD_FMin,picData->FPD_FMax);

      if(mode & 8192) {
	if(picData->FPD_AxeValues) {
	  long *index,arrayIndex;

	  if((index = (long *)
#ifdef CONFIG_USE_ALLOCA
	      alloca(picData->FPD_NDim * sizeof(*index))
#else
	      malloc(picData->FPD_NDim * sizeof(*index))
#endif
	      ) == NULL) goto nomem;

	  /*  Clear indices  */
	  for(i = 0 ; i < picData->FPD_NDim ; i++) index[i] = 0;
	  arrayIndex = 0;
	  do {
	    /*  Print axe values  */
	    if(picData->FPD_Flags & FPLOT_FLAGS_SINGLE) {
	      for(i = 0 ; i < picData->FPD_NDim ; i++)
		if(picData->FPD_AxeValues[i])
		  printf("%g ",((float *)(picData->FPD_AxeValues[i]))
			 [index[i]]);
		else
		  printf("---------------- ");

	      printf("   %g\n",((float *)picData->FPD_Data)[arrayIndex]);
	    } else {
	      for(i = 0 ; i < picData->FPD_NDim ; i++)
		if(picData->FPD_AxeValues[i])
		  printf("%g ",((double *)(picData->FPD_AxeValues[i]))
			 [index[i]]);
		else
		  printf("------------------------ ");

	      printf("   %g\n",((double *)picData->FPD_Data)[arrayIndex]);
	    }

	    /*  Increment  */
	    arrayIndex++;
	    for(i = 0 ; i < picData->FPD_NDim ; i++) {
	      if(++index[i] < picData->FPD_Dims[i]) break;
	      index[i] = 0;
	    }
	  } while(i < picData->FPD_NDim);

#ifndef CONFIG_USE_ALLOCA
	  free(index);
#endif
	} else {
	  int dim;

	  /*  Get full dimension  */
	  for(dim = 1, i = 0 ; i < picData->FPD_NDim ; i++)
	    dim *= picData->FPD_Dims[i];

	  if(picData->FPD_Flags & FPLOT_FLAGS_SINGLE) {
	    for(i = 0 ; i < dim ; i++)
	      printf("%g\n",((float *)picData->FPD_Data)[i]);
	  } else {
	    for(i = 0 ; i < dim ; i++)
	      printf("%g\n",((double *)picData->FPD_Data)[i]);
	  }
	}
      }
    }
  }
#endif

  /*  Any casts of data required?  */
  if(mode & FPLOT_MODE_CASTTOFLOAT) {
    if(fplotPicD2F(picture) < 0)
      goto error;
  } else if(mode & FPLOT_MODE_CASTTODOUBLE) {
    if(fplotPicF2D(picture) < 0)
      goto error;
  }

  if((mode & FPLOT_MODE_SORTDATA) && fplotSortData(picture) < 0)
    goto error;
  if((mode & FPLOT_MODE_GENAXEDATA) && fplotGenAxeData(picture) < 0)
    goto error;

  return(picture);
nomem:
  return(NULL);
ioerror:
error:
  return(NULL);
}
#undef FLAG_AXEVALUES
#undef FLAG_SINGLEPREC

/*JS*********************************************************************
*   Frees the structure.
*************************************************************************/

void fplotFreepic(FPPicture *picture)

/************************************************************************/
{
  FPData *picData,*picNext;

  /*  Allow user to pass null pointer  */
  if(picture == NULL) return;

  /*  Free Title and axes    */
  if(picture->FP_Title) free(picture->FP_Title);

  if(picture->FP_Axes) {
    int i;

    for(i = 0 ; i < picture->FP_NAxes ; i++)
      if(picture->FP_Axes[i]) free(picture->FP_Axes[i]);

    free(picture->FP_Axes);
  }

  /*   Now clear the contents of the structure	 */
  for(picData = &picture->FP_Content ; picData ; picData = picNext) {
    /*  Free all allocated memory  */
    if(picData->FPD_Dims) free(picData->FPD_Dims);
    if(picData->FPD_Boundaries) free(picData->FPD_Boundaries);
    if(picData->FPD_Data) free(picData->FPD_Data);

    if(picData->FPD_AxeValues) {
      int i;

      for(i = 0 ; i < picData->FPD_NDim ; i++)
	if(picData->FPD_AxeValues[i]) free(picData->FPD_AxeValues[i]);

      free(picData->FPD_AxeValues);
    }

    /*	Save next value  */
    picNext = picData->FPD_Next;

    /*	Clear every substructure except the first one	*/
    if(picData != &picture->FP_Content) free(picData);
  }

  /*  Finally clear the structure itself   */
  free(picture);

  return;
}

#define BUFFSIZE    8192

/*JS*********************************************************************
*   Do a dummy fseek from current position by reading and discarding the data.
*************************************************************************/

static int dummyFseek(FILE *fp,long offset)

/************************************************************************/
{
  char *buff;

#ifdef CONFIG_USE_ALLOCA
  if((buff = (char *)alloca(BUFFSIZE)) == NULL) return(-1);
#else
  if((buff = (char *)malloc(BUFFSIZE)) == NULL) return(-1);
#endif

  while(offset > 0) {
    long len = MIN(offset,BUFFSIZE);

    if(fread(buff,1,len,fp) != len) goto error;
    offset -= len;
  }

#ifndef CONFIG_USE_ALLOCA
  free(buff);
#endif

  return(0);
error:
#ifndef CONFIG_USE_ALLOCA
  free(buff);
#endif

  return(-1);
}
