/*JS*********************************************************************
*
*    Program : FPLOT
*    Language: ANSI-C
*    Author  : Joerg Schoen
*    Purpose : Routines for plotting data to a file.
*    Part    : General definitions and open routine.
*
*      Format of an fplot-file:
*       - MAGIC-value: The long integer MAGIC_PLOT (see below).
*       - An arbitrary number of hunks.
*
*      Format of a hunk:
*       - Hunk-Type:
*         A long integer. To recognize invalid fplot-files, the valid range
*         of values is the interval [HUNK_MIN , HUNK_MAX] (see below).
*         Routines that read fplot-files should *skip* unknown hunks and
*         *stop* on invalid ones.
*       - Hunk-Length:
*         A long integer, giving the number of long integers that follow.
*         The complete size of a hunk is therefore
*           (2 + <Hunk-Length>) * sizeof(long)
*
*      Comments:
*       - The hunk length is always rounded to an integer multiple of
*         sizeof(long), padded with 0-Bytes if necessary. This rounding
*         avoids misaligned copies when reading or writing which is usually
*         expensive on high-performance systems.
*
*       - The definition of the magic value as a long integer instead of a
*         string ensures that it is different for architectures with different
*         basic types (for example big and little-endian machines). Thus
*         fplot-files from one architecture are not accidentally recognized
*         from another system.
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id: fplotopen.c,v 1.6 1997/07/21 21:20:11 joerg Stab joerg $";
#endif

/*********     INCLUDES                                         *********/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <jssubs.h>

#include <fplot.h>

/*********     DEFINES                                          *********/
/*  May seek in directories as well  */
#define USE_FOPENPATH

#ifndef MAGIC_PLOT
/*  START-DEFINITIONS  */
/*    Plot files start with this long word  "PLotData version 1"   */
#define MAGIC_PLOT  (('P'<<24) | ('L'<<16) | ('D'<<8) | '1')

/*    Header in Outputfile: Precedes every hunk     */
typedef struct {
  unsigned long OH_Type;   /*  Type of the following hunk (see below). */
  /*  Length of this hunk in sizeof(long) without header  */
  unsigned long OH_Length;
} OutputHeader;

#define HUNK_MIN       2000UL /* For error recognizing	   */

#define HUNK_TEXT      2000UL /* Title  */
#define HUNK_AXES      2001UL /* Text for axes		   */

/*  Data hunks. ('S' means single precision values)  */
#define HUNK_VECTOR    2010UL /* For one-dimensional data  */
#define HUNK_SVECTOR   2011UL /* For one-dimensional data  */
#define HUNK_VECTORXY  2012UL /* For one-dimensional data with axe values */
#define HUNK_SVECTORXY 2013UL /* For one-dimensional data with axe values */

#define HUNK_TENSOR    2020UL /* For multi-dimensional data   */
#define HUNK_STENSOR   2021UL /* For multi-dimensional data   */
#define HUNK_TENSORXY  2022UL /* For multi-dimensional data with axe values */
#define HUNK_STENSORXY 2023UL /* For multi-dimensional data with axe values */

#define HUNK_AXEDATA   2030UL /* For special axe data hunk  */
#define HUNK_SAXEDATA  2031UL /* For special axe data hunk  */

#define HUNK_END       2222UL  /* End of current output	   */

#define HUNK_MAX       9999UL  /* For error recognizing	   */

/*   For routine fplotOpen:	*/
#define FPLOT_READ    0
#define FPLOT_WRITE   1
#define FPLOT_APPEND  2

/* ****  File type handled by fplot routines  *** */
typedef struct {
  FILE         *FF_File;
  unsigned long FF_Flags;
} FPFile;
#define FPLOT_FLAG_NOFSEEK      (1<<0)

/* ***	Macros for rounding lengths to longword boundary   *** */
#define HUNKROUND	  sizeof(long)
#define HUNKLENGTH(len)   (((len) + (HUNKROUND - 1)) / HUNKROUND)
#define HUNKPADDING(len)  ((HUNKROUND - ((len) & (HUNKROUND-1))) & \
			   (HUNKROUND-1))
/*  Defines for special type field (must be lower than 0x100)  */
#define FPLOT_TYPE_SPECTRUM      1  /*  Data are treated as spectral data    */
#define FPLOT_TYPE_XYCURVE       2  /*  Data is a general curve in x-y level */

#define FPLOT_TYPE_DUMPDATA     0xff /*  Dump data, don't plot  */

#define FPLOT_TYPE_MASK         0xff /*  Mask for type field bits  */

#define FPLOT_TYPE_COMPLEXDATA  (1<<8) /*  Flag for complex data  */
/*  END-DEFINITIONS  */
#endif

/* ERROR-DEFINITIONS from fplotOpen label _ERR_FPLTOPEN ord 3
   Wrong magic value
*/

#define Prototype extern
/*********     PROTOTYPES                                       *********/
Prototype FPFile        *fplotOpen(const char *name,int mode);
Prototype FPFile        *fplotOpen2(const char *name,int mode);
Prototype int            fplotClose(FPFile *ffp);

Prototype const char    *FPlotOpenPath,*FPlotOpenName;

/*********     GLOBAL VARIABLES 				*********/
/*  Path for files opened by fplotOpen  */
const char *FPlotOpenPath;
const char *FPlotOpenName;

/*JS*********************************************************************
*   Opens a plot file for reading and writing. If name is NULL, opens
*    stdin/stdout.
*************************************************************************/

FPFile *fplotOpen(const char *name,int mode)

/************************************************************************/
{
  FPFile *ffp;
  FILE *fp;
  long magic;

  switch(mode) {
  case FPLOT_READ:
#ifdef USE_FOPENPATH
    if(name) {
      const char *save1,*save2;

      save1 = FOpenPath;  save2 = FOpenName;

      FOpenPath = FPlotOpenPath;
      fp = fopenPath(name,"rb");
      FPlotOpenName = FOpenName;

      FOpenPath = save1;  FOpenName = save2;
    } else
      fp = stdin;
#else
    fp = name ? fopen(name,"rb") : stdin;
#endif

    break;
  case FPLOT_APPEND:
    /*  Try to open  */
#ifdef USE_FOPENPATH
    if(name) {
      const char *save1,*save2;

      save1 = FOpenPath;  save2 = FOpenName;

      FOpenPath = FPlotOpenPath;
      fp = fopenPath(name,"rb+");
      FPlotOpenName = FOpenName;

      FOpenPath = save1;  FOpenName = save2;
    } else
      fp = stdout;
#else
    fp = name ? fopen(name,"rb+") : stdout;
#endif
    if(fp) break;

    /*  File does not exist, so this is really write, not append  */
    mode = FPLOT_WRITE;
  case FPLOT_WRITE:
    fp = name ? fopen(name,"wb") : stdout;
    break;
  default:
    return(NULL);
  }

  if(fp == NULL) return(NULL);

  if(mode == FPLOT_READ || (mode == FPLOT_APPEND && name)) {
    /*   First read in Magic value     */
    if(fread(&magic,sizeof(magic),1,fp) == 1) {
      /*   Check magic value      */
      if(magic != MAGIC_PLOT) {
	JSErrNo = _ERR_FPLTOPEN + 0;
	goto error;
      }

      /*   Go to end of file    */
      if(mode == FPLOT_APPEND && fseek(fp,0L,SEEK_END)) goto error;
    } else if(!feof(fp))
      goto error;
  } else if(name || mode == FPLOT_WRITE) { /*    PLOT_WRITE    */
    /*   Set up Header for output file (Just one longword)  */
    magic = MAGIC_PLOT;

    /*    Write out Header   */
    if(fwrite(&magic,sizeof(magic),1,fp) != 1) goto error;
  }

  if((ffp = (FPFile *)malloc(sizeof(*ffp))) == NULL) goto error;

  ffp->FF_File = fp;
  ffp->FF_Flags = name ? 0 : FPLOT_FLAG_NOFSEEK;

  return(ffp);
error:
  fclose(fp);
  return(NULL);
}

/*JS*********************************************************************
*   Same as fplotOpen, but recognizes "-" as synomym for stdin/stdout.
*************************************************************************/

FPFile *fplotOpen2(const char *name,int mode)

/************************************************************************/
{
  return(fplotOpen(strcmp(name,"-") == 0 ? (const char *)NULL : name,mode));
}

/*JS*********************************************************************
*
*************************************************************************/

int fplotClose(FPFile *ffp)

/************************************************************************/
{
  FILE *fp;

  fp = ffp->FF_File;
  free(ffp);

  return(fclose(fp));
}
