/*JS*********************************************************************
*
*    Program : FPLOTCOPY
*    Language: ANSI-C
*    Author  : Joerg Schoen
*    Purpose : Routines for copying a picture.
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id: fplotcopy.c,v 1.8 1996/10/21 15:56:18 joerg Stab joerg $";
#endif

/*********     INCLUDES 					*********/
#include <jsalloca.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <math.h>

#include <jsconfig.h>
#include <jssubs.h>

#include <fplot.h>

/*********     DEFINES						*********/
#ifdef CONFIG_NO_STRDUP
# define strdup  mystrdup  /*  strdup is not POSIX conformant  */
#endif

/* ERROR-DEFINITIONS from fplotCopy label _ERR_FPLTCOPY ord 1
   No range for axe given
   0 dimensional data generated
*/

#define Prototype extern
/*********     PROTOTYPES					*********/
Prototype FPPicture     *fplotCopy(FPPicture *pic,unsigned long part,
				   double *bounds,int nBounds,int mode);

static long              findIndexF(float *data,long n,double x);
static long              findIndexD(double *data,long n,double x);

/*JS*********************************************************************
*   Copys a part of the picture and creates a new one. The bits of 'part'
*    mark what subpictures NOT to take (0 means all). The 'bounds' array
*    give the start (bounds[2*i]) and end (bounds[2*i+1]) interval in each
*    dimension for the copy; 'n' is the length of the array. Missing entries
*    means this dimension is to be fully included. If a end value is lower
*    than a start value, the start value marks where to take the cut --
*    the corresponding dimension is removed.
*************************************************************************/

FPPicture *fplotCopy(FPPicture *pic,unsigned long part,double *bounds,
		     int nBounds,int mode)

/************************************************************************/
{
  FPData *picData,*newData;
  FPPicture *newPic;
  int maxDim;
  long *lengths,*ind,*offs,*strides;

  /*  Get maximal dimension of subpicture  */
  maxDim = 0;
  for(picData = &pic->FP_Content ; picData ; picData = picData->FPD_Next)
    if(picData->FPD_NDim > maxDim) maxDim = picData->FPD_NDim;

#ifdef CONFIG_USE_ALLOCA
  if((lengths = (long *)alloca(4 * maxDim * sizeof(*lengths))) == NULL)
    goto error;
#else
  if((lengths = (long *)malloc(4 * maxDim * sizeof(*lengths))) == NULL)
    goto error;
#endif
  ind = &lengths[maxDim];
  offs = &ind[maxDim];
  strides = &offs[maxDim];

  /*  Get empty picture  */
  if((newPic = (FPPicture *)malloc(sizeof(*newPic))) == NULL) goto error;
  newPic->FP_Title = pic->FP_Title ? strdup(pic->FP_Title) : (char *)NULL;

  /*  Copy axes  */
  if((newPic->FP_NAxes = pic->FP_NAxes) > 0) {
    int i,j;

    /*  Count number of dimensions that are removed  */
    for(i = 0 ; i < nBounds ; i++)
      if(bounds[2*i] > bounds[2*i+1]) newPic->FP_NAxes--;

    if(newPic->FP_NAxes > 0) {
      if((newPic->FP_Axes = (char **)
	  malloc(newPic->FP_NAxes * sizeof(*(newPic->FP_Axes)))) == NULL)
	goto error;

      /*  Now copy axes  */
      for(j = i = 0 ; i < pic->FP_NAxes ; i++) {
	if(i < nBounds && bounds[2*i] > bounds[2*i+1]) continue;

	newPic->FP_Axes[j] = pic->FP_Axes[i] ? strdup(pic->FP_Axes[i]) :
	  (char *)NULL;
	j++;
      }
    } else
      newPic->FP_Axes = NULL;
  } else
    newPic->FP_Axes = NULL;

  newPic->FP_Content.FPD_Next = NULL;
  newPic->FP_Content.FPD_NDim = 0;
  newPic->FP_Content.FPD_Dims = NULL;
  newPic->FP_Content.FPD_Boundaries = NULL;
  newPic->FP_Content.FPD_AxeValues = NULL;
  newPic->FP_Content.FPD_Data = NULL;

  /*  Scan through, cut and copy  */
  newData = NULL;
  for(picData = &pic->FP_Content ; picData ; picData = picData->FPD_Next,
      part >>= 1) {
    long globOff,dim,nDim;
    int i,j,size,first;

    /*  Copy this part?  */
    if(part & 1) continue;

    /*  Get new structure  */
    if(newData) {
      /*   Allocate memory for new structure	 */
      if((newData->FPD_Next = (FPData *)malloc(sizeof(*newData))) == NULL)
	goto error;
      newData = newData->FPD_Next;
      newData->FPD_Next = NULL;
      newData->FPD_NDim = 0;
      newData->FPD_Dims = NULL;
      newData->FPD_Boundaries = NULL;
      newData->FPD_AxeValues = NULL;
      newData->FPD_Data = NULL;
    } else {
      newData = &newPic->FP_Content;
    }

    /*  Calculate global offset, new dimension and needed space  */
    globOff = nDim = 0;
    dim = 1;
    first = -1;

    for(strides[0] = 1, i = 0 ; i < picData->FPD_NDim ; i++) {
      /*  Set next stride value  */
      if(i > 0) strides[i] = strides[i - 1] * picData->FPD_Dims[i - 1];

      if(i < nBounds) {
	if(picData->FPD_AxeValues == NULL ||
	   picData->FPD_AxeValues[i] == NULL) {
	  double picMin,picMax;

	  /*  No axe data present, so get from given area  */
	  if((picData->FPD_Flags & FPLOT_FLAGS_SINGLE)) {
	    picMin = ((float *)picData->FPD_Boundaries)[2 * i];
	    picMax = ((float *)picData->FPD_Boundaries)[2 * i + 1];
	  } else {
	    picMin = ((double *)picData->FPD_Boundaries)[2 * i];
	    picMax = ((double *)picData->FPD_Boundaries)[2 * i + 1];
	  }
	  if(picMin == picMax) {
	    JSErrNo = _ERR_FPLTCOPY + 0;
	    goto error;
	  }

	  /*  Get offset and number of values  */
	  offs[i] = (long int)ceil((picData->FPD_Dims[i] - 1) *
				   (bounds[2*i] - picMin) / (picMax - picMin));

	  if(offs[i] < 0)
	    offs[i] = 0;
	  else if(offs[i] >= picData->FPD_Dims[i])
	    offs[i] = picData->FPD_Dims[i] - 1;

	  if(bounds[2*i] <= bounds[2*i+1]) {
	    lengths[i] = (long int)floor(((picData->FPD_Dims[i] - 1) *
					  (bounds[2*i+1] - picMin)) /
					 (picMax - picMin)) - offs[i] + 1;

	    if(lengths[i] > (picData->FPD_Dims[i] - offs[i]))
	      lengths[i] = picData->FPD_Dims[i] - offs[i];
	    else if(lengths[i] < 1)
	      lengths[i] = 1;
	  } else {
	    lengths[i] = 0; /*  This dimension is removed  */
	  }
	} else {
	  if((picData->FPD_Flags & FPLOT_FLAGS_SINGLE)) {
	    offs[i] = findIndexF((float *)(picData->FPD_AxeValues[i]),
				 picData->FPD_Dims[i],bounds[2*i]);
	  } else {
	    offs[i] = findIndexD((double *)(picData->FPD_AxeValues[i]),
				 picData->FPD_Dims[i],bounds[2*i]);
	  }

	  if(bounds[2*i] <= bounds[2*i+1]) {
	    long off2;

	    if((picData->FPD_Flags & FPLOT_FLAGS_SINGLE)) {
	      off2 =
		findIndexF(&(((float *)(picData->FPD_AxeValues[i]))[offs[i]]),
			   picData->FPD_Dims[i] - offs[i],bounds[2*i + 1]);

	      /*  Correct value  */
	      if((((float *)(picData->FPD_AxeValues[i]))[offs[i] + off2])
		 > bounds[2 * i + 1])
		off2--;
	    } else {
	      off2 =
		findIndexD(&(((double *)(picData->FPD_AxeValues[i]))[offs[i]]),
			   picData->FPD_Dims[i] - offs[i],bounds[2*i + 1]);

	      /*  Correct value  */
	      if((((double *)(picData->FPD_AxeValues[i]))[offs[i] + off2])
		 > bounds[2 * i + 1])
		off2--;
	    }

	    lengths[i] = 1 + off2;
	  } else {
	    lengths[i] = 0; /*  This dimension is removed  */
	  }
	}
      } else {
	/*  Include this dimension completely  */
	offs[i] = 0;
	lengths[i] = picData->FPD_Dims[i];
      }

#if defined(DEBUG)
      printf("%d: off %d len %d\n",i,offs[i],lengths[i]);
#endif

      if(lengths[i]) {
	nDim++;
	dim *= lengths[i];
	if(first == -1) first = i;
      }

      /*  Calculate global offset in data  */
      globOff += offs[i] * strides[i];
    }

    /*  All dimensions removed?  */
    if(nDim == 0) {
      JSErrNo = _ERR_FPLTCOPY + 1;
      goto error;
    }

    if(picData->FPD_Flags & FPLOT_FLAGS_SINGLE) size = sizeof(float);
    else size = sizeof(double);

    /*  Allocate needed memory  */
    if((newData->FPD_Dims = (unsigned long *)
	malloc(nDim *sizeof(*newData->FPD_Dims))) == NULL) goto error;
    if((newData->FPD_Boundaries = malloc(nDim * 2 * size)) == NULL) goto error;

    /*  Set up new structure */
    newData->FPD_Flags = picData->FPD_Flags;
    newData->FPD_Type = picData->FPD_Type;
    newData->FPD_NDim = nDim;
    for(j = 0, i = first ; i < picData->FPD_NDim ; i++)
      if(lengths[i]) newData->FPD_Dims[j++] = lengths[i];

    if((newData->FPD_Data = malloc(dim * size)) == NULL) goto error;

    if(picData->FPD_AxeValues) {
      /*  Set up axe values pointer  */
      if((newData->FPD_AxeValues = (void **)
	  malloc(nDim * sizeof(*newData->FPD_AxeValues))) == NULL) goto error;

      for(i = 0 ; i < nDim ; i++) newData->FPD_AxeValues[i] = NULL;
    }

    /*  Copy the axe data  */
    for(j = 0, i = first ; i < picData->FPD_NDim ; i++) {
      int k;

      if(lengths[i] == 0) continue;

      if(picData->FPD_Flags & FPLOT_FLAGS_SINGLE) {
	float *oPtr,*nPtr;

	if(picData->FPD_AxeValues && picData->FPD_AxeValues[i]) {
	  if((newData->FPD_AxeValues[j] = malloc(lengths[i] * sizeof(float)))
	     == NULL) goto error;

	  oPtr = (float *)(picData->FPD_AxeValues[i]) + offs[i];
	  nPtr = (float *)(newData->FPD_AxeValues[j]);

	  for(k = 0 ; k < lengths[i] ; k++) nPtr[k] = oPtr[k];
	}

	/*  Set up boundaries  */
	if(((float *)picData->FPD_Boundaries)[2 * i] ==
	   ((float *)picData->FPD_Boundaries)[2 * i + 1]) {
	  /*  Set to same value similar to original data  */
	  ((float *)newData->FPD_Boundaries)[2 * j] =
	    ((float *)newData->FPD_Boundaries)[2 * j + 1] =
	      ((float *)picData->FPD_Boundaries)[2 * i];
	} else if(picData->FPD_AxeValues && picData->FPD_AxeValues[i]) {
	  /*  Get actual boundaries similar to original data  */
	  getMinMaxS(nPtr,lengths[i],
		     &((float *)newData->FPD_Boundaries)[2 * j],
		     &((float *)newData->FPD_Boundaries)[2 * j + 1]);
	} else if(i < nBounds) {
	  float xa,dx;

	  /*  Calculate correct values  */
	  xa = ((float *)picData->FPD_Boundaries)[2 * i];
	  dx = (((float *)picData->FPD_Boundaries)[2 * i + 1] - xa) /
	    (picData->FPD_Dims[i] - 1);

	  ((float *)newData->FPD_Boundaries)[2 * j] = xa + dx *
	    (float)offs[i];
	  ((float *)newData->FPD_Boundaries)[2 * j + 1] = xa + dx *
	    (float)(offs[i] + lengths[i] - 1);
	} else {
	  /*  Use original data  */
	  ((float *)newData->FPD_Boundaries)[2 * j] =
	    ((float *)picData->FPD_Boundaries)[2 * i];
	  ((float *)newData->FPD_Boundaries)[2 * j + 1] =
	    ((float *)picData->FPD_Boundaries)[2 * i + 1];
	}
      } else {
	double *oPtr,*nPtr;

	if(picData->FPD_AxeValues && picData->FPD_AxeValues[i]) {
	  if((newData->FPD_AxeValues[j] = malloc(lengths[i] * sizeof(double)))
	     == NULL) goto error;

	  oPtr = (double *)(picData->FPD_AxeValues[i]) + offs[i];
	  nPtr = (double *)(newData->FPD_AxeValues[j]);

	  for(k = 0 ; k < lengths[i] ; k++) nPtr[k] = oPtr[k];
	}

	/*  Set up boundaries  */
	if(((double *)picData->FPD_Boundaries)[2 * i] ==
	   ((double *)picData->FPD_Boundaries)[2 * i + 1]) {
	  /*  Set to same value similar to original data  */
	  ((double *)newData->FPD_Boundaries)[2 * j] =
	    ((double *)newData->FPD_Boundaries)[2 * j + 1] =
	      ((double *)picData->FPD_Boundaries)[2 * i];
	} else if(picData->FPD_AxeValues && picData->FPD_AxeValues[i]) {
	  /*  Get actual boundaries similar to original data  */
	  getMinMax(nPtr,lengths[i],
		    &((double *)newData->FPD_Boundaries)[2 * j],
		    &((double *)newData->FPD_Boundaries)[2 * j + 1]);
	} else if(i < nBounds) {
	  double xa,dx;

	  /*  Calculate correct values  */
	  xa = ((double *)picData->FPD_Boundaries)[2 * i];
	  dx = (((double *)picData->FPD_Boundaries)[2 * i + 1] - xa) /
	    (picData->FPD_Dims[i] - 1);

	  ((double *)newData->FPD_Boundaries)[2 * j] = xa + dx *
	    (double)offs[i];
	  ((double *)newData->FPD_Boundaries)[2 * j + 1] = xa + dx *
	    (double)(offs[i] + lengths[i] - 1);
	} else {
	  /*  Use original data  */
	  ((double *)newData->FPD_Boundaries)[2 * j] =
	    ((double *)picData->FPD_Boundaries)[2 * i];
	  ((double *)newData->FPD_Boundaries)[2 * j + 1] =
	    ((double *)picData->FPD_Boundaries)[2 * i + 1];
	}
      }

      j++;
    }

    /*  Copy the data  */
    if(picData->FPD_Flags & FPLOT_FLAGS_SINGLE) {
      float *oPtr,*nPtr;

      oPtr = &((float *)picData->FPD_Data)[globOff];
      nPtr = (float *)newData->FPD_Data;

      for(i = first ; i < picData->FPD_NDim ; i++) ind[i] = offs[i] = 0;

      for(j = 0 ; ; ) {
	nPtr[j++] = oPtr[offs[first]];

	for(i = first ; i < picData->FPD_NDim ; i++) {
	  if(lengths[i] == 0) continue;
	  ind[i]++;

	  if(ind[i] < lengths[i]) {
	    offs[i] += strides[i];
	    break;
	  }
	}
	if(i == picData->FPD_NDim) break;
	while(i-- > 0) {
	  ind[i] = 0;
	  offs[i] = offs[i + 1];
	}
      }

      if(!(mode & FPLOT_MODE_NOGETMINMAX) &&
	 picData->FPD_FMin != picData->FPD_FMax) {
	float fMin,fMax;

	getMinMaxS(nPtr,dim,&fMin,&fMax);
	newData->FPD_FMin = fMin;
	newData->FPD_FMax = fMax;
      } else {
	newData->FPD_FMin = newData->FPD_FMax = 1.0;
      }
    } else {
      double *oPtr,*nPtr;

      oPtr = &((double *)picData->FPD_Data)[globOff];
      nPtr = (double *)newData->FPD_Data;

      for(i = first ; i < picData->FPD_NDim ; i++) ind[i] = offs[i] = 0;

      for(j = 0 ; ; ) {
	nPtr[j++] = oPtr[offs[first]];

	for(i = first ; i < picData->FPD_NDim ; i++) {
	  if(lengths[i] == 0) continue;
	  ind[i]++;

	  if(ind[i] < lengths[i]) {
	    offs[i] += strides[i];
	    break;
	  }
	}
	if(i == picData->FPD_NDim) break;
	while(i-- > 0) {
	  ind[i] = 0;
	  offs[i] = offs[i + 1];
	}
      }

      if(!(mode & FPLOT_MODE_NOGETMINMAX) &&
	 picData->FPD_FMin != picData->FPD_FMax) {
	getMinMax(nPtr,dim,&newData->FPD_FMin,&newData->FPD_FMax);
      } else {
	newData->FPD_FMin = newData->FPD_FMax = 1.0;
      }
    }
  }

#ifndef CONFIG_USE_ALLOCA
  free(lengths);
#endif

  /*  Any casts of data required?  */
  if(mode & FPLOT_MODE_CASTTOFLOAT) {
    if(fplotPicD2F(newPic) < 0)
      goto error;
  } else if(mode & FPLOT_MODE_CASTTODOUBLE) {
    if(fplotPicF2D(newPic) < 0)
      goto error;
  }

  if((mode & FPLOT_MODE_SORTDATA) && fplotSortData(newPic) < 0)
    goto error;

  if((mode & FPLOT_MODE_GENAXEDATA) && fplotGenAxeData(newPic) < 0)
    goto error;

  return(newPic);
error:
  return(NULL);
}

/*JS*********************************************************************
*   Locates the index of the element that is greater or equal to x.
*************************************************************************/

#define GENROUTINE(name,PRECISION) \
static long name(PRECISION *data,long n,double x) \
{ \
  long lower,upper; \
\
  /*  Locate position of x in data  */ \
  lower = 0;  upper = n - 1; \
  while(lower < upper) { \
    long m = (lower + upper) >> 1; \
\
    if(x > data[m]) lower = m + 1; \
    else            upper = m; \
  } \
\
  return(lower); \
}

/************************************************************************/
/*  Generate routines on the fly  */

GENROUTINE(findIndexF,float)

GENROUTINE(findIndexD,double)

#ifdef TEST
#include <stdlib.h>
main(int argc,char *argv[])
{
  double x[10] = { 1,2,3,4,5,6,7,8,9,10 };
  double f=strtod(argv[1],NULL);
  printf("%ld\n",findIndexD(x,10,f));
}
#endif
