/*JS*********************************************************************
*
*    Program : FPLOTCAST
*    Language: ANSI-C
*    Author  : Joerg Schoen
*    Purpose : Routines to cast data from 'fplotRead' to a certain
*              precision or to have axe data.
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id: fplotcast.c,v 1.2 1996/10/21 15:54:23 joerg Stab joerg $";
#endif

/*********     INCLUDES 					*********/
#include <jsalloca.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <math.h>

#include <jssubs.h>

#include <fplot.h>

/*********     DEFINES						*********/

#define Prototype extern
/*********     PROTOTYPES					*********/
Prototype int            fplotPicD2F(FPPicture *picture);
Prototype int            fplotPicF2D(FPPicture *picture);
Prototype int            fplotGenAxeData(FPPicture *picture);
Prototype int            fplotSortData(FPPicture *picture);
static void		 sortFAxeData(int *table,long n,float *fArray);
static void		 sortDAxeData(int *table,long n,double *dArray);

Prototype int            fplotScaleData(FPPicture *picture,
					const double *scales,int nScales);

/*JS*********************************************************************
*   Cast double precision picture to single precision.
*************************************************************************/

int fplotPicD2F(FPPicture *picture)

/************************************************************************/
{
  FPData *picData;

  /*  Scan through all subpictures  */
  for(picData = &picture->FP_Content ; picData ; picData = picData->FPD_Next) {
    float *data;
    unsigned long dim,len;
    int i,n;

    /*  Skip empty blocks and blocks that are already single precision  */
    if(picData->FPD_Data == NULL || (picData->FPD_Flags & FPLOT_FLAGS_SINGLE))
      continue;

    /*  Calculate dimension  */
    for(dim = 1, i = 0 ; i < picData->FPD_NDim ; i++)
      dim *= picData->FPD_Dims[i];

    /*  Cast boundary values  */
    data = (float *)picData->FPD_Boundaries;
    for(i = 0 ; i < (2 * picData->FPD_NDim) ; i++)
      data[i] = ((double *)data)[i];

    /*  Cast data  */
    data = (float *)picData->FPD_Data;
    for(i = 0 ; i < dim ; i++) data[i] = ((double *)data)[i];

    if((picData->FPD_Data = realloc(picData->FPD_Data,dim * sizeof(float)))
       == NULL) goto nomem;

    /*  Cast axe values  */
    if(picData->FPD_AxeValues)
      for(n = 0 ; n < picData->FPD_NDim ; n++) {
	if((data = (float *)picData->FPD_AxeValues[n]) == NULL) continue;
	len = picData->FPD_Dims[n];

	for(i = 0 ; i < len ; i++) data[i] = ((double *)data)[i];

	if((picData->FPD_AxeValues[n] = realloc(picData->FPD_AxeValues[n],len *
					       sizeof(float))) == NULL)
	  goto nomem;
      }

    /*  Set flag  */
    picData->FPD_Flags |= FPLOT_FLAGS_SINGLE;
  }

  return(0);
nomem:
  return(-1);
}

/*JS*********************************************************************
*   Cast single precision picture to double precision.
*************************************************************************/

int fplotPicF2D(FPPicture *picture)

/************************************************************************/
{
  FPData *picData;

  /*  Scan through all subpictures  */
  for(picData = &picture->FP_Content ; picData ; picData = picData->FPD_Next) {
    double *data;
    unsigned long dim,len;
    int i,n;

    /*  Skip empty blocks and blocks that are already double precision  */
    if(picData->FPD_Data == NULL || !(picData->FPD_Flags & FPLOT_FLAGS_SINGLE))
      continue;

    /*  Calculate dimension  */
    for(dim = 1, i = 0 ; i < picData->FPD_NDim ; i++)
      dim *= picData->FPD_Dims[i];

    /*  Cast boundary values  */
    if((picData->FPD_Boundaries =
	realloc(picData->FPD_Boundaries,2 * picData->FPD_NDim * sizeof(*data)))
       == NULL) goto nomem;
    data = (double *)picData->FPD_Boundaries;
    for(i = 2 * picData->FPD_NDim - 1 ; 0 <= i ; i--)
      data[i] = ((float *)data)[i];

    /*  Transfer data  */
    if((picData->FPD_Data = realloc(picData->FPD_Data,dim * sizeof(double))) ==
       NULL) goto nomem;
    data = (double *)picData->FPD_Data;
    for(i = dim - 1 ; 0 <= i ; i--) data[i] = ((float *)data)[i];

    if(picData->FPD_AxeValues)
      for(n = 0 ; n < picData->FPD_NDim ; n++) {
	if(picData->FPD_AxeValues[n] == NULL) continue;

	len = picData->FPD_Dims[n];
	if((picData->FPD_AxeValues[n] = realloc(picData->FPD_AxeValues[n],len *
						sizeof(double))) == NULL)
	  goto nomem;

	data = (double *)picData->FPD_AxeValues[n];

	for(i = len - 1 ; 0 <= i ; i--) data[i] = ((float *)data)[i];
      }

    /*  Delete flag  */
    picData->FPD_Flags &= ~FPLOT_FLAGS_SINGLE;
  }
  return(0);
nomem:
  return(-1);
}

/*JS*********************************************************************
*   Generate axe data if not present.
*************************************************************************/

int fplotGenAxeData(FPPicture *picture)

/************************************************************************/
{
  FPData *picData;

  /*  Scan through all subpictures  */
  for(picData = &picture->FP_Content ; picData ; picData = picData->FPD_Next) {
    int n;

    /*  Skip empty blocks  */
    if(picData->FPD_Data == NULL) continue;

    /*  Vector for axe data already allocated?  */
    if(picData->FPD_AxeValues == NULL) {
      if((picData->FPD_AxeValues = (void **)
	  malloc(picData->FPD_NDim * sizeof(*picData->FPD_AxeValues)))
	 == NULL) goto nomem;

      for(n = 0 ; n < picData->FPD_NDim ; n++)
	picData->FPD_AxeValues[n] = NULL;
    }

    for(n = 0 ; n < picData->FPD_NDim ; n++) {
      int i,len;

      if(picData->FPD_AxeValues[n]) continue;

      len = picData->FPD_Dims[n];

      if(picData->FPD_Flags & FPLOT_FLAGS_SINGLE) {
	float *data,xa,xe,dx;

	if((data = (float *)(picData->FPD_AxeValues[n] =
			     malloc(len * sizeof(float)))) == NULL)
	  goto nomem;

	xa = ((float *)picData->FPD_Boundaries)[2 * n];
	xe = ((float *)picData->FPD_Boundaries)[2 * n + 1];
	dx = (xe - xa) / (len - 1);

	for(i = 0 ; i < (len - 1) ; i++) data[i] = xa + i * dx;

	/*  Set last value directly to prevent rounding errors  */
	data[len - 1] = xe;
      } else {
	double *data,xa,xe,dx;

	if((data = (double *)(picData->FPD_AxeValues[n] =
			      malloc(len * sizeof(double)))) == NULL)
	  goto nomem;

	xa = ((double *)picData->FPD_Boundaries)[2 * n];
	xe = ((double *)picData->FPD_Boundaries)[2 * n + 1];
	dx = (xe - xa) / (len - 1);

	for(i = 0 ; i < (len - 1) ; i++) data[i] = xa + i * dx;

	/*  Set last value directly to prevent rounding errors  */
	data[len - 1] = xe;
      }
    }
  }

  return(0);
nomem:
  return(-1);
}

/*JS*********************************************************************
*   Sort all data values in ascending order.
*************************************************************************/

int fplotSortData(FPPicture *picture)

/************************************************************************/
{
  FPData *picData;

  /*  Scan through all subpictures  */
  for(picData = &picture->FP_Content ; picData ;
      picData = picData->FPD_Next) {
    double *work;
    int i,preDim,currDim,remDim,n;

    /*  Skip empty blocks  */
    if(picData->FPD_Data == NULL) continue;

    /*  Preset first and last dimension  */
    preDim = 1;
    for(remDim = 1, i = 1 ; i < picData->FPD_NDim ; i++)
      remDim *= picData->FPD_Dims[i];

    /*  Get maximal needed work space  */
    if((work = (double *)malloc(((picData->FPD_Dims[0] * remDim) /
				 picData->FPD_Dims[picData->FPD_NDim - 1]) *
				sizeof(*work))) == NULL) goto nomem;

    for(n = 0 ; ; ) {
      currDim = picData->FPD_Dims[n];

      /*  Axe data present?  */
      if(picData->FPD_AxeValues && picData->FPD_AxeValues[n]) {
	int *table;

	/*  Generate index table  */
	if((table = (int *)
#ifdef CONFIG_USE_ALLOCA
	    alloca(currDim * sizeof(*table))
#else
	    malloc(currDim * sizeof(*table))
#endif
	    ) == NULL) goto nomem;

	/*  Preset table  */
	for(i = 0 ; i < currDim ; i++) table[i] = i;

	/*  Generate permutation and rearrange axe values and data  */
	if(picData->FPD_Flags & FPLOT_FLAGS_SINGLE) {
	  sortFAxeData(table,currDim,(float *)picData->FPD_AxeValues[n]);

	  permute((float *)picData->FPD_AxeValues[n],
		  currDim,table,sizeof(float),work);

	  for(i = 0 ; i < remDim ; i++)
	    permute(&((float *)picData->FPD_Data)[preDim * currDim * i],
		    currDim,table,preDim * sizeof(float),work);
	} else {
	  sortDAxeData(table,currDim,(double *)picData->FPD_AxeValues[n]);

	  permute((double *)picData->FPD_AxeValues[n],
		  currDim,table,sizeof(double),work);

	  for(i = 0 ; i < remDim ; i++)
	    permute(&((double *)picData->FPD_Data)[preDim * currDim * i],
		    currDim,table,preDim * sizeof(double),work);
	}

#ifndef CONFIG_USE_ALLOCA
	free(table);
#endif
      }

      /*  Rearrange boundary values if necessary  */
      if(picData->FPD_Flags & FPLOT_FLAGS_SINGLE) {
	if(((float *)picData->FPD_Boundaries)[2 * n] >
	   ((float *)picData->FPD_Boundaries)[2 * n + 1]) {
	  float temp;

	  /*  Swap boundary values  */
	  temp = ((float *)picData->FPD_Boundaries)[2 * n];
	  ((float *)picData->FPD_Boundaries)[2 * n] =
	    ((float *)picData->FPD_Boundaries)[2 * n + 1];
	  ((float *)picData->FPD_Boundaries)[2 * n + 1] = temp;

	  /*  If data arrangement not yet done, mirror them now  */
	  if(picData->FPD_AxeValues == NULL ||
	     picData->FPD_AxeValues[n] == NULL) {
	    for(i = 0 ; i < remDim ; i++) {
	      int offset,j,j2;

	      offset = preDim * currDim * i;

	      for(j = 0 ; j < (currDim / 2) ; j++) {
		j2 = currDim - j - 1;

		/*  Swap j and j2  */
		memcpy(work,
		       &((float *)picData->FPD_Data)[offset + preDim * j],
		       preDim * sizeof(float));
		memcpy(&((float *)picData->FPD_Data)[offset + preDim * j],
		       &((float *)picData->FPD_Data)[offset + preDim * j2],
		       preDim * sizeof(float));
		memcpy(&((float *)picData->FPD_Data)[offset + preDim * j2],
		       work,preDim * sizeof(float));
	      }
	    }
	  }
	}
      } else {
	if(((double *)picData->FPD_Boundaries)[2 * n] >
	   ((double *)picData->FPD_Boundaries)[2 * n + 1]) {
	  double temp;

	  /*  Swap boundary values  */
	  temp = ((double *)picData->FPD_Boundaries)[2 * n];
	  ((double *)picData->FPD_Boundaries)[2 * n] =
	    ((double *)picData->FPD_Boundaries)[2 * n + 1];
	  ((double *)picData->FPD_Boundaries)[2 * n + 1] = temp;

	  /*  If data arrangement not yet done, mirror them now  */
	  if(picData->FPD_AxeValues == NULL ||
	     picData->FPD_AxeValues[n] == NULL) {
	    for(i = 0 ; i < remDim ; i++) {
	      int offset,j,j2;

	      offset = preDim * currDim * i;

	      for(j = 0 ; j < (currDim / 2) ; j++) {
		j2 = currDim - j - 1;

		/*  Swap j and j2  */
		memcpy(work,
		       &((double *)picData->FPD_Data)[offset + preDim * j],
		       preDim * sizeof(double));
		memcpy(&((double *)picData->FPD_Data)[offset + preDim * j],
		       &((double *)picData->FPD_Data)[offset + preDim * j2],
		       preDim * sizeof(double));
		memcpy(&((double *)picData->FPD_Data)[offset + preDim * j2],
		       work,preDim * sizeof(double));
	      }
	    }
	  }
	}
      }

      preDim *= picData->FPD_Dims[n];
      n++;
      if(n >= picData->FPD_NDim) break;
      remDim /= picData->FPD_Dims[n];
    }

    free(work);
  }

  return(0);
nomem:
  return(-1);
}

/*JS*********************************************************************
*   Sort single precision axe data in ascending order.
*************************************************************************/

/*  Generated automatically with "gensort.h"  */
#define GENSORT_NAME		     sortFAxeData
#define GENSORT_ARGS		     ,fArray
#define GENSORT_ARGSPROTO	     ,float fArray[]
#define GENSORT_TYPE		     int
#define GENSORT_KEYTYPE 	     float
#define GENSORT_GETKEY(a)            fArray[a]
#define GENSORT_COMPAREKEYS(k1,k2)   k1 < k2

static	/*  Generated as static routine    */
#include <gensort.h>

/*JS*********************************************************************
*   Sort double precision axe data in ascending order.
*************************************************************************/

/*  Generated automatically with "gensort.h"  */
#define GENSORT_NAME		     sortDAxeData
#define GENSORT_ARGS		     ,dArray
#define GENSORT_ARGSPROTO	     ,double dArray[]
#define GENSORT_TYPE		     int
#define GENSORT_KEYTYPE 	     double
#define GENSORT_GETKEY(a)            dArray[a]
#define GENSORT_COMPAREKEYS(k1,k2)   k1 < k2

static	/*  Generated as static routine    */
#include <gensort.h>

/*JS*********************************************************************
*   Rescale data: The pairs (scales[2*i],scales[2*i+1]) determine what to
*    do, i=0,..,nDim-1 for the axes, i=nDim for the data:
*    If both values are equal, simply multiply data by it.
*    If both values are different and none of them is HUGE_VAL, they are
*    the new interval the data should cover.
*    If the first value is HUGE_VAL, the data is rescaled so that the
*    upper limit will be scales[2*i+1], if the second is HUGE_VAL, so
*    that scales[2*i] will be the lower limit.
*************************************************************************/

int fplotScaleData(FPPicture *picture,const double *scales,int nScales)

/************************************************************************/
{
  FPData *picData;

  /*  Rescale all data  */
  for(picData = &picture->FP_Content ; picData ; picData = picData->FPD_Next) {
    int i,nDim;

    nDim = picData->FPD_NDim;

    for(i = 0 ; i < nScales ; i++) {
      double lo,hi,m,b;

      /* **  Get limits of data  ** */
      if(i < nDim) {
	if(picData->FPD_Flags & FPLOT_FLAGS_SINGLE) {
	  lo = ((float *)picData->FPD_Boundaries)[2*i];
	  hi = ((float *)picData->FPD_Boundaries)[2*i+1];
	} else {
	  lo = ((double *)picData->FPD_Boundaries)[2*i];
	  hi = ((double *)picData->FPD_Boundaries)[2*i+1];
	}
      } else {
	lo = picData->FPD_FMin;
	hi = picData->FPD_FMax;
      }

      if(scales[2*i] == HUGE_VAL) {
	/* **  Scale data so that upper limit equals scales[2*i+1]  ** */
	m = scales[2*i+1] / hi;
	b = 0.0;
      }	else if(scales[2*i+1] == HUGE_VAL) {
	/* **  Scale data so that lower limit equals scales[2*i]  ** */
	m = scales[2*i] / lo;
	b = 0.0;
      }	else if(scales[2*i] == scales[2*i+1]) {
	/* **  Only scale data with value  ** */
	if((m = scales[2*i]) == 1.0) /*  can be skipped  */
	  continue;
	b = 0.0;
      } else {
	/* **  Set data to new interval scales[2*i],scales[2*i+1]  ** */
	m = (scales[2*i+1] - scales[2*i]) / (hi - lo);
	b = (hi * scales[2*i] - lo * scales[2*i+1]) / (hi - lo);
      }

#ifdef DEBUG
      printf("%d: lo %g hi %g sc [%g,%g] -> m %g b %g\n",i,lo,hi,
	     scales[2*i],scales[2*i+1],m,b);
#endif

      if(i < nDim) {
	/* **  Axe values  ** */
	if(picData->FPD_Flags & FPLOT_FLAGS_SINGLE) {
	  if(picData->FPD_AxeValues && picData->FPD_AxeValues[i]) {
	    float *ptr = (float *)(picData->FPD_AxeValues[i]);
	    int j,len = picData->FPD_Dims[i];

	    if(b == 0.0)
	      for(j = 0 ; j < len ; j++) ptr[j] = m * ptr[j];
	    else
	      for(j = 0 ; j < len ; j++) ptr[j] = m * ptr[j] + b;
	  }

	  ((float *)picData->FPD_Boundaries)[2*i] =
	    m * ((float *)picData->FPD_Boundaries)[2*i] + b;
	  ((float *)picData->FPD_Boundaries)[2*i+1] =
	    m * ((float *)picData->FPD_Boundaries)[2*i+1] + b;
	} else {
	  if(picData->FPD_AxeValues && picData->FPD_AxeValues[i]) {
	    double *ptr = (double *)(picData->FPD_AxeValues[i]);
	    int j,len = picData->FPD_Dims[i];

	    if(b == 0.0)
	      for(j = 0 ; j < len ; j++) ptr[j] = m * ptr[j];
	    else
	      for(j = 0 ; j < len ; j++) ptr[j] = m * ptr[j] + b;
	  }

	  ((double *)picData->FPD_Boundaries)[2*i] =
	    m * ((double *)picData->FPD_Boundaries)[2*i] + b;
	  ((double *)picData->FPD_Boundaries)[2*i+1] =
	    m * ((double *)picData->FPD_Boundaries)[2*i+1] + b;
	}
      } else if(i == nDim) {
	/* **  Data values  ** */

	if(picData->FPD_Data) {
	  long dim;

	  /*  Calculate dimension of data  */
	  for(dim = 1, i = 0 ; i < nDim ; i++) dim *= picData->FPD_Dims[i];

	  if(picData->FPD_Flags & FPLOT_FLAGS_SINGLE) {
	    float *ptr = (float *)picData->FPD_Data;

	    if(b == 0.0)
	      for(i = 0 ; i < dim ; i++) ptr[i] = m * ptr[i];
	    else
	      for(i = 0 ; i < dim ; i++) ptr[i] = m * ptr[i] + b;
	  } else {
	    double *ptr = (double *)picData->FPD_Data;

	    if(b == 0.0)
	      for(i = 0 ; i < dim ; i++) ptr[i] = m * ptr[i];
	    else
	      for(i = 0 ; i < dim ; i++) ptr[i] = m * ptr[i] + b;
	  }
	}

	picData->FPD_FMin = m * picData->FPD_FMin + b;
	picData->FPD_FMax = m * picData->FPD_FMax + b;
      }
    }
  }

  return(0);
}
