/*JS*********************************************************************
*
*    Program : FPLOT
*    Language: ANSI-C
*    Author  : Joerg Schoen
*    Purpose : Routines for plotting data to a file.
*    Part    : Plot axes labels.
*
*      Format of axes hunk:
*        hunk header:
*                ulong  hunk_type   (== HUNK_AXES)
*                ulong  hunk_length (in "sizeof(long)" without hunk header).
*        hunk content:
*           n1 * bytes  text1   Description of first axe with final '\0'.
*           n2 * bytes  text2   Description of second axe with final '\0'.
*                ...
*                bytes  padd    Some padding '\0' to get multiples of
*                                HUNKROUND as total length.
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id: fplotaxes.c,v 1.5 1997/07/21 21:17:52 joerg Stab joerg $";
#endif

/*********     INCLUDES 					*********/
#include <jsalloca.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

#include <fplot.h>

/*********     DEFINES						*********/

#define Prototype extern
/*********     PROTOTYPES					*********/
Prototype int		 fplotAxes(FPFile *fp,int nArgs,const char **args);
Prototype int		 fplotAxesV(FPFile *fp,const char *xaxis, ...);
Prototype int		 fplotAxesS(FPFile *fp,const char *axes);

/*JS*********************************************************************
*   Plots text for the axes.
*************************************************************************/

int fplotAxes(FPFile *fp,int nArgs,const char **args)

/************************************************************************/
{
  OutputHeader oheader;
  long len;
  int i;

  for(len = 0, i = 0 ; i < nArgs ; i++)
    len += args[i] ? (strlen(args[i]) + 1) : 1;

  /*   Set up Header correctly	  */
  oheader.OH_Type   = HUNK_AXES;
  oheader.OH_Length = HUNKLENGTH(len);

  /*	Write out Header...   */
  if(fwrite(&oheader,sizeof(oheader),1,fp->FF_File) != 1) goto error;

  /*	...and text    */
  for(i = 0 ; i < nArgs ; i++) {
    if(args[i]) {
      if(fwrite(args[i],strlen(args[i]) + 1,1,fp->FF_File) != 1) goto error;
    } else
      if(putc('\0',fp->FF_File) == EOF) goto error;
  }

  /*  Write padding bytes  */
  i = HUNKROUND * oheader.OH_Length - len;
  if(i > 0 && fwrite("\0\0\0",i,1,fp->FF_File) != 1) goto error;

  return(0);
error:
  return(-1);
}

/*JS*********************************************************************
*   Plots text for the axes. List of axes must be terminated with NULL.
*************************************************************************/

int fplotAxesV(FPFile *fp,const char *xaxis, ...)

/************************************************************************/
{
  va_list ap;
  const char **args,*next;
  int argI,argM,i;

  /*  Collect all arguments in vector  */
  va_start(ap,xaxis);
  args = NULL;
  argI = argM = 0;
  for(next = xaxis ; next ; next = va_arg(ap,const char *)) {
    if(argI >= argM) {
      argM += 5;
      if((args = (const char **)realloc(args,argM * sizeof(*args))) == NULL)
	return(-1);
    }
    args[argI++] = next;
  }
  va_end(ap);

  i = fplotAxes(fp,argI,args);
  free(args);

  return(i);
}

/*JS*********************************************************************
*   Plots text for the axes. Axes are separated with ';', which may be
*    escaped.
*************************************************************************/

int fplotAxesS(FPFile *fp,const char *axes)

/************************************************************************/
{
  const char *src,**args;
  char *dest;
  int argI,i;

  /*  Count number of axes and length  */
  for(src = axes, argI = 1 ; *src ; src++) {
    if(*src == '\\' && src[1] == ';')
      src++;
    else if(*src == ';')
      argI++;
  }

  if((args = (const char **)
#ifdef CONFIG_USE_ALLOCA
      alloca(argI * sizeof(*args) + (src - axes + 1))
#else
      malloc(argI * sizeof(*args) + (src - axes + 1))
#endif
      ) == NULL) return(-1);
  dest = (char *)&args[argI];

  /*  Count number of axes and length  */
  args[0] = dest;
  i = 1;
  for(src = axes ; *src ; src++) {
    if(*src == '\\' && src[1] == ';')
      src++;
    else if(*src == ';') {
      *dest++ = '\0';
      args[i++] = dest;
      continue;
    }
    *dest++ = *src;
  }
  *dest = '\0';

  /*for(i = 0 ; i < argI ; i++) printf("args[%d] = \"%s\"\n",i,args[i]);*/
  i = fplotAxes(fp,argI,args);

#ifndef CONFIG_USE_ALLOCA
  free(args);
#endif

  return(i);
}
