/*JS*********************************************************************
*
*    Program : FPLOT.H
*    Language: ANSI-C
*    Author  : Joerg Schoen
*    Purpose : Include file for fplot library
*
*************************************************************************/
#ifndef __FPLOT_H__ /*   File must not be included more than once!   */
#define __FPLOT_H__

/*********     INCLUDES 					*********/
#include <stdio.h>   /*	 For "FILE" definition    */

/* ****    Some useful macros (be aware of side effects because   **** */
/* ****     they treat arguments more than once).		  **** */
#ifndef MAX
# define MAX(a,b)  ((a) > (b) ? (a) :  (b))
#endif
#ifndef MIN
# define MIN(a,b)  ((b) > (a) ? (a) :  (b))
#endif
#ifndef ABS
# define ABS(x)    ((x) >= 0  ? (x) : -(x))
#endif

/* ***********	 Extracted definitions of library code  ********* */

/* ----------- FILE: fplotopen.c          ------------ */
/*    Plot files start with this long word  "PLotData version 1"   */
#define MAGIC_PLOT  (('P'<<24) | ('L'<<16) | ('D'<<8) | '1')

/*    Header in Outputfile: Precedes every hunk     */
typedef struct {
  unsigned long OH_Type;   /*  Type of the following hunk (see below). */
  /*  Length of this hunk in sizeof(long) without header  */
  unsigned long OH_Length;
} OutputHeader;

#define HUNK_MIN       2000UL /* For error recognizing	   */

#define HUNK_TEXT      2000UL /* Title  */
#define HUNK_AXES      2001UL /* Text for axes		   */

/*  Data hunks. ('S' means single precision values)  */
#define HUNK_VECTOR    2010UL /* For one-dimensional data  */
#define HUNK_SVECTOR   2011UL /* For one-dimensional data  */
#define HUNK_VECTORXY  2012UL /* For one-dimensional data with axe values */
#define HUNK_SVECTORXY 2013UL /* For one-dimensional data with axe values */

#define HUNK_TENSOR    2020UL /* For multi-dimensional data   */
#define HUNK_STENSOR   2021UL /* For multi-dimensional data   */
#define HUNK_TENSORXY  2022UL /* For multi-dimensional data with axe values */
#define HUNK_STENSORXY 2023UL /* For multi-dimensional data with axe values */

#define HUNK_AXEDATA   2030UL /* For special axe data hunk  */
#define HUNK_SAXEDATA  2031UL /* For special axe data hunk  */

#define HUNK_END       2222UL  /* End of current output	   */

#define HUNK_MAX       9999UL  /* For error recognizing	   */

/*   For routine fplotOpen:	*/
#define FPLOT_READ    0
#define FPLOT_WRITE   1
#define FPLOT_APPEND  2

/* ****  File type handled by fplot routines  *** */
typedef struct {
  FILE         *FF_File;
  unsigned long FF_Flags;
} FPFile;
#define FPLOT_FLAG_NOFSEEK      (1<<0)

/* ***	Macros for rounding lengths to longword boundary   *** */
#define HUNKROUND	  sizeof(long)
#define HUNKLENGTH(len)   (((len) + (HUNKROUND - 1)) / HUNKROUND)
#define HUNKPADDING(len)  ((HUNKROUND - ((len) & (HUNKROUND-1))) & \
			   (HUNKROUND-1))
/*  Defines for special type field (must be lower than 0x100)  */
#define FPLOT_TYPE_SPECTRUM      1  /*  Data are treated as spectral data    */
#define FPLOT_TYPE_XYCURVE       2  /*  Data is a general curve in x-y level */

#define FPLOT_TYPE_DUMPDATA     0xff /*  Dump data, don't plot  */

#define FPLOT_TYPE_MASK         0xff /*  Mask for type field bits  */

#define FPLOT_TYPE_COMPLEXDATA  (1<<8) /*  Flag for complex data  */

/* ----------- FILE: fplotread.c          ------------ */
/*  Defines for routine 'fplotRead'  */
#define FPLOT_MODE_READ 	 (1<<0)
#define FPLOT_MODE_NOGETMINMAX	 (1<<1)
#define FPLOT_MODE_CASTTOFLOAT	 (1<<2)
#define FPLOT_MODE_CASTTODOUBLE  (1<<3)
#define FPLOT_MODE_GENAXEDATA    (1<<4)
#define FPLOT_MODE_SORTDATA      (1<<5)

#define FPLOT_MODE_USER          (1<<8)

/*  Set from routine 'fplotRead' in FPD_Flags field  */
#define FPLOT_FLAGS_MASK    0xff
#define FPLOT_FLAGS_SINGLE  (1<<8)  /*  Values are single precision  */

/*  Structure to store the contents of one hunk (double precision)  */
typedef struct FPData FPData;
struct FPData {
  FPData        *FPD_Next;
  unsigned long  FPD_Flags;
  int            FPD_Type,FPD_NDim;
  unsigned long *FPD_Dims;
  void          *FPD_Boundaries;
  double         FPD_FMin,FPD_FMax;
  void         **FPD_AxeValues;
  void          *FPD_Data;
};

/*  Structure to store the contents of one sequence    */
/*   of hunks (stopped with a END hunk).               */
typedef struct {
  int	   FP_User;
  char	  *FP_Title;
  FPData   FP_Content;
  int	   FP_NAxes;
  char	 **FP_Axes;
} FPPicture;

/*  Some useful abbreviations	 */
#define FP_Next    FP_Content.FPD_Next
#define FP_Flags   FP_Content.FPD_Flags

/* ----------- FILE: fplottrans.c         ------------ */
#define FPLOT_TRANS_NOEND    FPLOT_MODE_USER
#define FPLOT_TRANS_APPEND   (FPLOT_MODE_USER << 1)

/* *******   Prototypes of library functions  ***** */
/*  For C++ compatibility   */
# ifdef __cplusplus
#  define Prototype extern "C"
# else
#  define Prototype extern
# endif

/* ***** File: fplot.c               ***** */
Prototype int            fplotStart(FPFile *ffp);
Prototype int            fplotEnd(FPFile *ffp);

/* ***** File: fplotanalyze.c        ***** */
Prototype int            fplotAnalyze(char *names[],int nFiles,int readMode[],
				      int nPics,
				      int (*doAnalyze)(int ord,
						       FPPicture *pics[],
						       int nFiles,int nPics));
Prototype int            fplotAnalyze2(char *names[],int nFiles,int readMode[],
				       int nPics,
				       int (*doAnalyze)(int ord,
							FPPicture *pics[],
							int nFiles,int nPics));

/* ***** File: fplotaxes.c           ***** */
Prototype int		 fplotAxes(FPFile *fp,int nArgs,const char **args);
Prototype int		 fplotAxesV(FPFile *fp,const char *xaxis, ...);
Prototype int		 fplotAxesS(FPFile *fp,const char *axes);

/* ***** File: fplotcast.c           ***** */
Prototype int            fplotPicD2F(FPPicture *picture);
Prototype int            fplotPicF2D(FPPicture *picture);
Prototype int            fplotGenAxeData(FPPicture *picture);
Prototype int            fplotSortData(FPPicture *picture);
Prototype int            fplotScaleData(FPPicture *picture,
					const double *scales,int nScales);

/* ***** File: fplotcopy.c           ***** */
Prototype FPPicture     *fplotCopy(FPPicture *pic,unsigned long part,
				   double *bounds,int nBounds,int mode);

/* ***** File: fplotopen.c           ***** */
Prototype FPFile        *fplotOpen(const char *name,int mode);
Prototype FPFile        *fplotOpen2(const char *name,int mode);
Prototype int            fplotClose(FPFile *ffp);
Prototype const char    *FPlotOpenPath,*FPlotOpenName;

/* ***** File: fplotread.c           ***** */
Prototype FPPicture	*fplotRead(FPFile *ffp,int mode);
Prototype void		 fplotFreepic(FPPicture *picture);

/* ***** File: fplotstensor.c        ***** */
Prototype int		 fplotSTensor(FPFile *fp,int type,int nDim,
				      const float **axeData,
				      const float *data,
				      unsigned long *dims,long *strides,
				      const float *boundaries,
				      float fMin,float fMax);
Prototype int		 fplotSTensorV(FPFile *fp,int type,int nDim, ...);

/* ***** File: fplotsubs.c           ***** */
Prototype void           getMinMax(const double *vector,int n,double *pMin,
				   double *pMax);
Prototype void           getMinMaxS(const float *vector,int n,float *pMin,
				    float *pMax);

/* ***** File: fplottensor.c         ***** */
Prototype int		 fplotTensor(FPFile *fp,int type,int nDim,
				     const double **axeData,
				     const double *data,
				     unsigned long *dims,long *strides,
				     const double *boundaries,
				     double fMin,double fMax);
Prototype int		 fplotTensorV(FPFile *fp,int type,int nDim, ...);

/* ***** File: fplottext.c           ***** */
Prototype int            fplotText(FPFile *fp,const char *format, ...);

/* ***** File: fplottrans.c          ***** */
Prototype int		 fplotTransform(char *inName,char *outName,
					int readMode,int nPics,
					int (*doTrans)(int ord,
						       FPPicture *pics[],
						       int npics));
Prototype int		 fplotTransform2(char *inName,char *outName,
					 int readMode,int nPics,
					 int (*doTrans)(int ord,
							FPPicture *pics[],
							int npics));

/* ***** File: fplotvector.c         ***** */
Prototype int		 fplotVector(FPFile *ffp,int type,const double *xvector,
				      const double *yvector,unsigned long n,
				      long stride,
				      double xMin,double xMax,
				      double fMin,double fMax);

# undef Prototype   /* I like to be clean  */

#endif /*  "#ifdef __FPLOT_H__"  */
