/*JS*********************************************************************
*
*    Program : FOPENPATH
*    Language: ANSI-C with POSIX directory scanning functions
*    Author  : Joerg Schoen
*    Purpose : Seek in path and directories before fopen.
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id: fopenpath.c,v 1.10 1997/06/19 21:02:08 joerg Stab joerg $";
#endif

/*********     INCLUDES 					*********/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <jsconfig.h>

#include <jssubs.h>

#ifndef CONFIG_NO_POSIX
# include <unistd.h>
# include <sys/types.h>
# include <sys/stat.h>
# include <dirent.h>
#endif

#include <errno.h>

/*********     DEFINES						*********/
/*  START-DEFINITIONS */
#include <stdio.h>   /*	 For "FILE" definition    */
/*  END-DEFINITIONS */

#define Prototype extern
/*********     PROTOTYPES					*********/
Prototype FILE		*fopenPath(const char *name,const char *mode);
Prototype const char	*FOpenPath;

Prototype FILE		*fopenScan(const char *name,const char *mode);
Prototype const char	*FOpenName;

static FILE		*subFopenScan(char *current,char *rem,
				      const char *mode);

/*********     GLOBAL VARIABLES 				*********/
/*  Input for fopenPath: the path  */
const char *FOpenPath = NULL;

/*  Output of fopenPath and fopenScan: the 'real' file name  */
const char *FOpenName = NULL;

static const char PDelimitString[] = {
  CONFIG_DIR_DELIMITER,  CONFIG_DIR_DELIMITER, '\0'
};

/*JS*********************************************************************
*   Tries to find the file in the path given in the global variable
*    'FOpenPath'. This has to be a ':' separated list of directories. If
*    'FOpenPath' is not set, this call is equivalent to a normal fopen
*    call. The current directory is not searched for the file automatically,
*    so it must be put explicitly in the path. An empty entry in the path
*    is equivalent to the current directory. If the file was found anywhere
*    in the path, the global variable 'FOpenName' contains a pointer to a
*    static field containing the file name. This field may be equal to 'name'.
*    "//" sequences in the path lets the routine search recursively
*    through the directory tree: between two '/' an arbitrary path is
*    tried.
*
*    Example:
*      FOpenPath = ":/home//input//"     and   name = "test"
*
*    Possible results may be:
*      "test" or
*      "/home/input/test" or "/home/SPECIAL/input/GENERAL/test" or
*      "/home/AA/BB/input/XX/YY/test".
*************************************************************************/

FILE *fopenPath(const char *name,const char *mode)

/************************************************************************/
{
  const char *string;
static char nameBuffer[JS_FILENAME_MAX];
  char *string2;
  FILE *fp;
  int saveErrno;

  /*  If no path is set or path name contains '/', just normal fopen call  */
  if(FOpenPath == NULL || strchr(name,CONFIG_DIR_DELIMITER)) {
    /*	Use "fopenScan" only with path!  */
    if((fp = fopen(name,mode)))
      FOpenName = name;
    return(fp);
  }

  /*  Scan through path to find file  */
  saveErrno = errno;
  for(string = FOpenPath ; *string ; ) {
    /*	Copy next path name  */
    for(string2 = nameBuffer ; *string ; ) {
      if(*string == CONFIG_LIST_DELIMITER) {
	/*  Skip delimiter and break  */
	string++;
	break;
      }
      *string2++ = *string++;
    }

    /*	Insert '/' if necessary  */
    if(string2 > nameBuffer && string2[-1] != CONFIG_DIR_DELIMITER)
      *string2++ = CONFIG_DIR_DELIMITER;

    /*	And append name  */
    strcpy(string2,name);

    /*	Try to open file  */
    if((fp = fopenScan(nameBuffer,mode))) {
      errno = saveErrno;  /*  Reset error code from unsuccessful opens	*/

      return(fp);
    }
  }

  /*  File not found in path, so return NULL  */
  return(NULL);
}

#ifndef CONFIG_NO_POSIX
/* ***	Static buffer for seekroutines	*** */
static char NameBuffer[JS_FILENAME_MAX + 1];

/*JS*********************************************************************
*   Sets up for call of subOpen, which searchs through directory structure.
*************************************************************************/

FILE *fopenScan(const char *name,const char *mode)

/************************************************************************/
{
  FILE *fp;
  char *current;

  /*  Preset found name  */
  FOpenName = NULL;

  /*  Look if path contains special delimiter  */
  if((current = strstr(name,PDelimitString)) == NULL) {
    /*	Just call fopen  */
    if((fp = fopen(name,mode)))
      FOpenName = name;
  } else {
    int offset;
    char *rem;

    /*	Set up for call of subOpen  */
    offset = (current - name) + 1;

    /*	Generate first part of filename  */
    strncpy(NameBuffer,name,offset);
    NameBuffer[offset] = '\0';

    /*	Skip "//" and trailing '/'  */
    for(rem = current + 2 ; *rem == CONFIG_DIR_DELIMITER ; ) rem++;

    /*	 Now try to get file anywhere in directory structure  */
    if((fp = subFopenScan(&NameBuffer[offset],rem,mode)))
      FOpenName = NameBuffer;
  }

  return(fp);
}

/*JS*********************************************************************
*   Assumes, that 'NameBuffer' already contains the first fixed part of
*    the file name and 'current' points within 'NameBuffer' to the start
*    of the next part. 'current' is the position of a "//" string in the
*    name to generate. 'rem' points to the remaining part of the file name,
*    it is searched for "//" strings, too. 'mode' simply is the mode for
*    the fopen call. This routine tries first to insert nothing between
*    "//" and then searches the directory structure.
*************************************************************************/

static FILE *subFopenScan(char *current,char *rem,const char *mode)

/************************************************************************/
{
  struct stat stBuf;
  char *next;

  if((next = strstr(rem,PDelimitString)) == NULL) {
    FILE *fp;

    /*	Append remaining part and try direct open  */
    strcpy(current,rem);

    if((fp = fopen(NameBuffer,mode)))
      return(fp);
  } else {
    /*	Now first scan current directory  */
    /*	Append intermediate part between two successive "//"  */
    strncpy(current,rem,next - rem);
    current[next - rem] = '\0';

    /*	Check if directory  */
    if(stat(NameBuffer,&stBuf) >= 0 && S_ISDIR(stBuf.st_mode)) {
      char *curr2,*rem2;
      FILE *fp;

      /*  Call recursive  */
      curr2 = &current[next - rem];
      strcpy(curr2,PDelimitString + 1);
      for(rem2 = next + 2 ; *rem2 == CONFIG_DIR_DELIMITER ; ) rem2++;

      if( (fp = subFopenScan(curr2 + 1,rem2,mode)) )
	return(fp);
    }
  }

  /* ***  Now scan current directory and call recursive  *** */
  {
    DIR *dp;
    int remLength;

    /*	Get remaining space for filename length overflow check	*/
    remLength = &NameBuffer[JS_FILENAME_MAX] - &current[strlen(rem)];

    *current = '\0';
    if( (dp = opendir(NameBuffer)) ) {
      struct dirent *dirp;

      while( (dirp = readdir(dp)) ) {
	FILE *fp;
	int len;

	/*  Reject "." and ".." entries or all  */
	/*   directories starting with "."      */
	if(dirp->d_name[0] == '.'
#ifndef CONFIG_IGNORE_POINTDIRS
	  && (dirp->d_name[1] == '.' || dirp->d_name[1] == '\0')
#endif
	  ) continue;

	strcpy(current,dirp->d_name);

	if(stat(NameBuffer,&stBuf) < 0 || !S_ISDIR(stBuf.st_mode))
	  continue;

	len = strlen(dirp->d_name);

	/*  Check file name length overflow  */
	if(len >= remLength) break;

	strcpy(current + len,PDelimitString + 1);

	/*  Try recursive  */
	if((fp = subFopenScan(current + len + 1,rem,mode))) {
	  (void)closedir(dp);
	  return(fp);
	}
      }

      (void) closedir(dp);
    }
  }

  return(NULL);
}
#else
/* ***	No POSIX available, so normal fopen call  *** */
FILE *fopenScan(const char *name,const char *mode)
{
  FILE *fp;

  /*  Preset found name  */
  FOpenName = NULL;

  if((fp = fopen(name,mode))) FOpenName = name;

  return(fp);
}
#endif

#ifdef TEST
int main(int argc,char *argv[])
{
  if(argc != 3) {
    printf("Usage: testfopenpath <path> <file>\n");
    return(20);
  }

  /*  Set path	*/
  FOpenPath = argv[1];

  if(fopenPath(argv[2],"r"))
    printf("%s: %s\n",argv[2],FOpenName);
  else
    printf("ERROR: File \"%s\" not found!\n",argv[2]);

  return(0);
}
#endif
