/*JS*********************************************************************
*
*    Program : FLOCKS
*    Language: ANSI-C with UNIX fcntl style locking functions
*    Author  : Joerg Schoen
*    Purpose : Routines to lock regions of files.
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id: flocks.c,v 1.5 1998/01/22 16:29:05 joerg Stab joerg $";
#endif

/*********     INCLUDES                                         *********/
#include <stdio.h>
#include <string.h>

#include <jsconfig.h>

#ifndef CONFIG_NO_POSIX
# include <sys/types.h>
# include <unistd.h>
# include <fcntl.h>
#endif

/*********     DEFINES                                          *********/
/*  START-DEFINITIONS */
#include <stdio.h>   /*	 For "FILE" definition    */

/*  Definitions for routine lockRegion and fopenLock  */
/*  Don't change these values!  */
#define LOCK_READ     0
#define LOCK_READW    1
#define LOCK_WRITE    2
#define LOCK_WRITEW   3
#define LOCK_UNLOCK   4

/*  Macros for usage of routine lockRegion2  */
#define lockRegion(fp,type,offset,whence,len)  \
	lockRegion2(fileno(fp),type,offset,whence,len)

#define lockRead(fp,off,whence,len)   lockRegion(fp,LOCK_READ,off,whence,len)
#define lockReadW(fp,off,whence,len)  lockRegion(fp,LOCK_READW,off,whence,len)

#define lockWrite(fp,off,whence,len)  lockRegion(fp,LOCK_WRITE,off,whence,len)
#define lockWriteW(fp,off,whence,len) lockRegion(fp,LOCK_WRITEW,off,whence,len)

#define unLock(fp,off,whence,len)     lockRegion(fp,LOCK_UNLOCK,off,whence,len)
/*  END-DEFINITIONS */

#define Prototype extern
/*********     PROTOTYPES                                       *********/
Prototype int            lockRegion2(int fd,int type,long offset,int whence,
				     long len);
Prototype FILE          *fopenLock(const char *name,const char *mode,int type);

/*********     GLOBAL / STATIC VARIABLES                        *********/
#ifndef CONFIG_NO_POSIX
static int LockTypes[] = {
  F_RDLCK, F_RDLCK, F_WRLCK, F_WRLCK, F_UNLCK
};

/*JS*********************************************************************
*   Auxiliary routine to lock a region in a file.
*************************************************************************/

int lockRegion2(int fd,int type,long offset,int whence,long len)

/************************************************************************/
{
  struct flock lock;

  /*  A little bit type checking  */
  if(type < 0 || type >= 5) return(-1);

  lock.l_type   = LockTypes[type];
  lock.l_start  = offset;
  lock.l_whence = whence;
  lock.l_len    = len;

  return(fcntl(fd,(type & 1) ? F_SETLKW : F_SETLK,&lock));
}

/*JS*********************************************************************
*   Same as fopen, but locks the file too. Simulates MANDATORY locking,
*    so it prevents the file from beeing truncated if another process has
*    opened it and has a lock on it.
*************************************************************************/

FILE *fopenLock(const char *name,const char *mode,int type)

/************************************************************************/
{
  FILE *fp;
  struct flock lock;
  int oFlag;
  int fd;
  char mode2[5];

  fd = -1; /*  Preset for error case  */

  /*  Preset oFlag for open call  */
  if(strchr(mode,'+') == NULL) {
    /*  Only one kind of access mode  */
    switch(mode[0]) {
    case 'r': oFlag = O_RDONLY;                  break;
      /*  In case of 'w' do not truncate now  */
    case 'w': oFlag = O_WRONLY|O_CREAT;          break;
    case 'a': oFlag = O_WRONLY|O_APPEND|O_CREAT; break;
    default: goto error;
    }
  } else {
    /*  Read and write access  */
    switch(mode[0]) {
    case 'r': oFlag = O_RDWR;                  break;
      /*  In case of 'w' do not truncate now  */
    case 'w': oFlag = O_RDWR|O_CREAT;          break;
    case 'a': oFlag = O_RDWR|O_APPEND|O_CREAT; break;
    default: goto error;
    }
  }

  /*  A little bit type checking  */
  if(type < 0 || type >= 5) goto error;
  lock.l_type   = LockTypes[type];
  lock.l_start  = 0;
  lock.l_whence = SEEK_SET;
  lock.l_len    = 0;

  /*  Open file descriptor and try to lock  */
  if((fd = open(name,oFlag,0644)) < 0 ||
     fcntl(fd,(type & 1) ? F_SETLKW : F_SETLK,&lock))
    goto error;

  /*  If open and lock suceeded, truncate file in case of write!  */
  if(mode[0] == 'w' && ftruncate(fd,0)) goto error;

  /*  Now copy mode to mode2, but skip any 'b' characters  */
  /*   which are disallowed in our unix version.           */
  {
    const char *string;
    char *string2;

    string = mode;
    string2 = mode2;

    do {
      if(*string != 'b') *string2++ = *string;
    } while(*string++);
  }

  /*  Transfer to standard library file pointer  */
  if((fp = fdopen(fd,mode2)) == NULL) goto error;

  return(fp);
error:
  if(fd >= 0) close(fd);
  return(NULL);
}

#else

/* ***  No POSIX functions available  *** */
int lockRegion(FILE *fp,int type,long offset,int whence,long len)
{  return(0); }
FILE *fopenLock(const char *name,const char *mode,int type)
{  return(fopen(name,mode)); }
#endif

#ifdef TEST
int main(int argc,char **argv)
{
  FILE *fp;
  int lucky;

  if(argc == 1) lucky = LOCK_READ;
  else lucky = LOCK_WRITE;

  if((fp = fopenLock("testfile","r+",lucky)) == NULL) {
    printf("ERROR: Cannot open and lock file.\n");

    if((fp = fopen("testfile","r+")) == NULL) {
      printf("ERROR: Cannot open file.\n");
      exit(20);
    }
  }

  printf("Successfully opened...\n\
 Now waiting 200 secs\n");

  sleep(200);

  printf("Good bye\n");

  fclose(fp);
  return(0);
}
#endif
