/*JS*********************************************************************
*
*    Program : DMP2ASC
*    Language: ANSI-C
*    Author  : Joerg Schoen
*    Purpose : Transfer unformatted data to ascii.
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id: dmp2asc.c,v 1.2 1997/03/06 11:22:09 joerg Stab joerg $";
#endif

/************************************************************************/
/*********     INCLUDES 					*********/
/************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <errno.h>

#include <jsconfig.h>

#ifndef CONFIG_NO_POSIX
#include <utime.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>
#endif

#include <jssubs.h>

#include <fplot.h>

#include "jtsubs.h"

/*********     DEFINES						*********/
#define OPTSTRING   "hqascCn"

#define FPLOT_EXTENSION    ".out"
#define DUMP_EXTENSION	   ".dmp"
#define ASCII_EXTENSION    ".asc"

#define FORMAT_DATE_TIME      "%d-%m-%y %H:%M:%S"
#define FORMAT_DATE_TIME2     "%d-%d-%d %d:%d:%d" /*  For sscanf  */
#define FORMAT_DATE_TIME_MAX  20

#define DFMT   "%25.17lg"
#define DFMT2  "%.17lg"
#define SFMT   "%17.9g"
#define SFMT2  "%.9g"

#define MODE_APPEND   (1<<0)
#define MODE_SKIP     (1<<1)
#define MODE_QUIET    (1<<2)
#define MODE_SETDATE  (1<<3)
#define MODE_STDOUT   (1<<4)
#define MODE_STDOUT2  (1<<5)
#define MODE_USENLINE (1<<6)

#define MARK_TEXT       '0'
#define MARK_AXE(n)     ('A' + (n))

#define Prototype extern
/*********     PROTOTYPES					*********/
Prototype int		 fplot2Ascii(FPFile *fp,FILE *fpOut,int mode);
Prototype int		 dmp2Ascii(char *name,FILE *fpOut);
Prototype int		 text2Ascii(char *text,char marker,FILE *fpOut);
Prototype int            prNameLine(FILE *fpOut,char *name);

Prototype int		 readAscii(FILE *fpIn,int mode);
Prototype int		 ascii2fplot(FILE *fpIn,char *string,FPFile *fpOut,
				     int mode);

Prototype int            ascii2Text(char marker,FILE *fpIn,char **pText);
Prototype int            setFileDate(char *name,time_t tim);
static int               dummyFseek(FILE *fp,long offset);

/*********     GLOBAL VARIABLES 				*********/
char NameBuffer[JS_FILENAME_MAX];
time_t FileTime;

static const char Padds[HUNKROUND];

int LineCount;

/*  For ASCII files  */
static char *Markers[] = {
  "FILE",
  "DUMP_SPEC",
  "DUMP_JATE",
  "HUNK_",
  NULL
};

/*JS*********************************************************************
*   MAIN
*************************************************************************/

int main(int argc,char *argv[])

/************************************************************************/
{
  int i,mode;
  char *optArg;
  FILE *fpOut;

  mode = 0;

  while((i = jsGetOpt(&argc,argv,OPTSTRING,&optArg)) != EOF) {
    switch(i) {
    case 'h':
    default:
      goto help;
    case 'a':
      mode |= MODE_APPEND;
      break;
    case 'q':
      mode |= MODE_QUIET;
      break;
    case 's':
      mode |= MODE_SETDATE;
      break;
    case 'c':
      mode |= MODE_STDOUT;
      break;
    case 'C':
      mode |= MODE_STDOUT2;
      break;
    case 'n':
      mode |= MODE_USENLINE;
      break;
    }
  }

  if(mode & MODE_STDOUT2) {
    FPFile *fpIn;

    /*  Read fplot file from stdin  */
    if((fpIn = fplotOpen(NULL,FPLOT_READ)) == NULL) goto error;

    while(!feof(fpIn->FF_File))
      if(fplot2Ascii(fpIn,stdout,mode)) goto error;

    fplotClose(fpIn);
    goto ende;
  }

  if(argc == 1) goto help;

  for(i = 1 ; i < argc ; i++) {
    char *string;

    /*  No name and date given  */
    NameBuffer[0] = '\0';
    FileTime = -1;

    if(strcmp(argv[i],"-") == 0) {
      /*  Read ASCII from stdin  */
      LineCount = 0;
      while(!feof(stdin)) {
	if(readAscii(stdin,mode)) goto error;
      }
      continue;
    }

    if((string = strrchr(argv[i],'.')) == NULL ||
       strchr(string,CONFIG_DIR_DELIMITER)) {
      fprintf(stderr,"ERROR: \"%s\" has no extension -- Skipped.\n",
	      argv[i]);
      continue;
    }

#ifndef CONFIG_NO_POSIX
    if(mode & MODE_SETDATE) {
      struct stat stat_buf;

      if(stat(argv[i],&stat_buf)) goto error;

      FileTime = stat_buf.st_mtime;
    }
#endif

    if(strcmp(string,FPLOT_EXTENSION) == 0) {
      FPFile *fpIn;

      if(!(mode & MODE_STDOUT)) {

	/*  Transfer unformatted data to ascii  */
	sprintf(NameBuffer,"%.*s%s",string - argv[i],argv[i],ASCII_EXTENSION);

#if 0 /*  no longer used, don't warn if ascii file will be overwritten  */
	{
	  int saveErrno = errno;

	  /*  Check if file already exists and warn user  */
#ifndef CONFIG_NO_POSIX
	  if(!(mode & MODE_APPEND) && access(NameBuffer,F_OK) == 0) {
#else
	  if(!(mode & MODE_APPEND) && (fpOut = fopen(NameBuffer,"r"))) {
	    fclose(fpOut); fpOut = NULL;
#endif
	    fprintf(stderr,"\
ERROR: File \"%s\" already exists, remove before next try -- Skipped.\n",
		    NameBuffer);
	    continue;
	  }
	  errno = saveErrno;
	}
#endif

	if((fpOut = fopen(NameBuffer,(mode & MODE_APPEND) ? "a" : "w"))
	   == NULL) goto error;

	if(!(mode & MODE_QUIET))
	  printf("\"%s\" -> \"%s\"\n",argv[i],NameBuffer);
      } else
	fpOut = stdout;

      if((fpIn = fplotOpen(argv[i],FPLOT_READ)) == NULL) goto error;

      /*  Generate name line  */
      if(prNameLine(fpOut,argv[i])) goto error;

      while(!feof(fpIn->FF_File))
	if(fplot2Ascii(fpIn,fpOut,mode)) goto error;

      fplotClose(fpIn);
      if(!(mode & MODE_STDOUT)) {
	fclose(fpOut);

	if((mode & MODE_SETDATE) && FileTime >= 0 &&
	   setFileDate(NameBuffer,FileTime))
	  goto error;
      }
    } else if(strcmp(string,DUMP_EXTENSION) == 0) {
      if(!(mode & MODE_STDOUT)) {
	int saveErrno;

	sprintf(NameBuffer,"%.*s%s",string - argv[i],argv[i],ASCII_EXTENSION);

	saveErrno = errno;
	/*  Check if file already exists and warn user  */
#ifndef CONFIG_NO_POSIX
	if(!(mode & MODE_APPEND) && access(NameBuffer,F_OK) == 0) {
#else
	if(!(mode & MODE_APPEND) && (fpOut = fopen(NameBuffer,"r"))) {
	  fclose(fpOut); fpOut = NULL;
#endif
	  fprintf(stderr,"\
ERROR: File \"%s\" already exists, remove before next try -- Skipped.\n",
		  NameBuffer);
	  continue;
	}
	errno = saveErrno;

	if((fpOut = fopen(NameBuffer,(mode & MODE_APPEND) ? "a" : "w")) == NULL)
	  goto error;

	if(!(mode & MODE_QUIET))
	  printf("\"%s\" -> \"%s\"\n",argv[i],NameBuffer);
      } else
	fpOut = stdout;

      /*  Generate name line  */
      if(prNameLine(fpOut,argv[i])) goto error;

      if(dmp2Ascii(argv[i],fpOut)) goto error;

      if(!(mode & MODE_STDOUT)) {
	fclose(fpOut);

	if((mode & MODE_SETDATE) && FileTime >= 0 &&
	   setFileDate(NameBuffer,FileTime))
	  goto error;
      }
    } else if(strcmp(string,ASCII_EXTENSION) == 0) {
      FILE *fpIn;

      if((fpIn = fopen(argv[i],"r")) == NULL) {
	fprintf(stderr,"ERROR: Cannot open \"%s\" -- Break.\n",argv[i]);
	break;
      }

      sprintf(NameBuffer,"%.*s",string - argv[i],argv[i]);

      if(!(mode & MODE_QUIET)) printf("Reading \"%s\"...\n",argv[i]);

      LineCount = 0;
      while(!feof(fpIn)) {
	if(readAscii(fpIn,mode)) goto error;
      }

      fclose(fpIn);
    } else {
      fprintf(stderr,"ERROR: \"%s\" has unknown extension -- Skipped\n",
	      argv[i]);
      continue;
    }
  }

ende:
  return(0);
error:
  jsperror("dmp2asc");
  return(30);
help:
  printf("\
Usage: dmp2asc <file1> ...			       Joerg Schoen 1994-97\n\
  Reads unformatted 'fplot' and dump files and transfers them to pure ASCII\n\
  and vice versa. If the file ends in \"%s\", it is assumed to be a 'fplot'\n\
  file, for \"%s\" a jate or spec dump file and for \"%s\" an ASCII file. If\n\
  the file name is \"-\", standard input is used as an ASCII file.\n\
  Options:\n\
    -q  Be quiet during operation.\n\
    -a	Append to output file \".out\" or \".asc\".\n\
    -c  Write ASCII file to stdout.\n\
    -C  Read stdin as an 'fplot' file and write ASCII to stdout.\n\
    -n  Use file name and date from ASCII representation rather than generating\n\
        it automatically from input file.\n\
    -s  Set modification time of generated file to that of original.\n",
	 FPLOT_EXTENSION,DUMP_EXTENSION,ASCII_EXTENSION);
  return(5);
}

/*JS*********************************************************************
*   FPLOT2ASCII
*************************************************************************/

int fplot2Ascii(FPFile *fpIn,FILE *fpOut,int mode)

/************************************************************************/
{
  OutputHeader oheader;
  char	*text;
  int i,j,len,countHunks,fPos;

  /*	Read hunks until end hunk      */
  for(fPos = ftell(fpIn->FF_File), countHunks = 0 ; ; countHunks++) {
    int flag;
    int nDim,type;

    /*	Read Header    */
    if(fread(&oheader,sizeof(oheader),1,fpIn->FF_File) != 1) {
      /*  If user forgot end hunk, don't worry  */
      if(feof(fpIn->FF_File)) break;

      goto error;
    }

    len = oheader.OH_Length * HUNKROUND;

    /*  Set up end of current hunk  */
    if(!(fpIn->FF_Flags & FPLOT_FLAG_NOFSEEK))
      fPos += sizeof(oheader) + len;

    /*	Reading or seeking?  */
    if((mode & MODE_SKIP)) {
      /*  Just fseek this hunk	  */
      if(len &&
	 ((fpIn->FF_Flags & FPLOT_FLAG_NOFSEEK) ?
	  dummyFseek(fpIn->FF_File,len) : fseek(fpIn->FF_File,len,SEEK_CUR)))
	goto error;

      /*    Test if Hunk end found   */
      if(oheader.OH_Type == HUNK_END) break;

      /*   Do not read contents    */
      continue;
    }

    /*  Now read Hunks  */
    nDim = 0;
    flag = 0;
#define FLAG_AXEVALUES	 1
#define FLAG_SINGLEPREC  2

    switch(oheader.OH_Type) {
    case HUNK_END:
      /*   Stop reading hunks	 */
      fprintf(fpOut,"HUNK_END.\n\n");
      goto endfound;
    case HUNK_AXES:
    case HUNK_TEXT:
      if(len == 0) {
	fprintf(stderr,"WARNING: Hunk contains no data -- Skipped.\n");
	break;
      }

      /*  Get memory for text	*/
      if((text = (char *)malloc(len + 1)) == NULL) goto error;

      /*   Read in contents	*/
      if(fread(text,1,len,fpIn->FF_File) != len) goto error;
      text[len] = '\0';

      if(oheader.OH_Type == HUNK_TEXT) {
	/*  Print out text  */
	fprintf(fpOut,"HUNK_TEXT:\n");
	text2Ascii(text,MARK_TEXT,fpOut);
      } else {
	char *string,marker;

	/*  Print out axe information  */
	fprintf(fpOut,"HUNK_AXES:\n");
	for(marker = 0, string = text ; string < text + len ;
	    string += strlen(string) + 1, marker++) {
	  text2Ascii(string,MARK_AXE(marker),fpOut);
	}
      }

      free(text);
      break;
    case HUNK_VECTORXY:
    case HUNK_SVECTORXY:
      flag |= FLAG_AXEVALUES;
    case HUNK_VECTOR:
    case HUNK_SVECTOR:
      if(oheader.OH_Type == HUNK_SVECTORXY || oheader.OH_Type == HUNK_SVECTOR)
	flag |= FLAG_SINGLEPREC;

      /*  Only type field to be read  */
      if(fread(&type,sizeof(type),1,fpIn->FF_File) != 1) goto error;
      nDim = 1;

      /*  Print header	*/
      switch(oheader.OH_Type) {
      case HUNK_VECTOR:    fprintf(fpOut,"HUNK_VECTOR: (%d) ",type);    break;
      case HUNK_VECTORXY:  fprintf(fpOut,"HUNK_VECTORXY: (%d) ",type);  break;
      case HUNK_SVECTOR:   fprintf(fpOut,"HUNK_SVECTOR: (%d) ",type);   break;
      case HUNK_SVECTORXY: fprintf(fpOut,"HUNK_SVECTORXY: (%d) ",type); break;
      }
      break;
    case HUNK_TENSORXY:
    case HUNK_STENSORXY:
      flag |= FLAG_AXEVALUES;
    case HUNK_TENSOR:
    case HUNK_STENSOR:
      if(oheader.OH_Type == HUNK_STENSORXY || oheader.OH_Type == HUNK_STENSOR)
	flag |= FLAG_SINGLEPREC;

      /*  Read type and nDim field  */
      if(fread(&type,sizeof(type),1,fpIn->FF_File) != 1 ||
	 fread(&nDim,sizeof(nDim),1,fpIn->FF_File) != 1)
	goto error;

      /*  Print header	*/
      switch(oheader.OH_Type) {
      case HUNK_TENSOR:    fprintf(fpOut,"HUNK_TENSOR: %d (%d) ",nDim,type);
	break;
      case HUNK_TENSORXY:  fprintf(fpOut,"HUNK_TENSORXY: %d (%d) ",nDim,type);
	break;
      case HUNK_STENSOR:   fprintf(fpOut,"HUNK_STENSOR: %d (%d) ",nDim,type);
	break;
      case HUNK_STENSORXY: fprintf(fpOut,"HUNK_STENSORXY: %d (%d) ",nDim,type);
	break;
      }
      break;
    case HUNK_SAXEDATA:
      flag |= FLAG_SINGLEPREC;
    case HUNK_AXEDATA:
      {
	int index,n;

	/*  Read index and number of this axe data  */
	if(fread(&index,sizeof(index),1,fpIn->FF_File) != 1 ||
	   fread(&n,sizeof(n),1,fpIn->FF_File) != 1) goto error;

	switch(oheader.OH_Type) {
	case HUNK_SAXEDATA: fprintf(fpOut,"HUNK_SAXEDATA: %d %d\n",index,n);
	  break;
	case HUNK_AXEDATA:  fprintf(fpOut,"HUNK_AXEDATA: %d %d\n",index,n);
	  break;
	}

	/*  Read data  */
	if(flag & FLAG_SINGLEPREC) {
	  float x;

	  for(j = 0 ; j < n ; j++)
	    if(fread(&x,sizeof(x),1,fpIn->FF_File) != 1 ||
	       fprintf(fpOut,SFMT"\n",x) < 0)
	      goto error;
	} else {
	  double x;

	  for(j = 0 ; j < n ; j++)
	    if(fread(&x,sizeof(x),1,fpIn->FF_File) != 1 ||
	       fprintf(fpOut,DFMT"\n",x) < 0)
	      goto error;
	}
      }

      break;
    default:
      if(oheader.OH_Type < HUNK_MIN || oheader.OH_Type > HUNK_MAX) {
	fprintf(stderr,"ERROR: Not a Hunk ('%#lx') -- Break.\n",
		oheader.OH_Type);
	goto error;
      }

      fprintf(stderr,"ERROR: Unknown Hunk Type '%#lx' -- Skipped.\n",
	      oheader.OH_Type);
      break;
    }

    if(nDim > 0) {
      unsigned long *dims,dim;

      if((dims = (unsigned long *)malloc(nDim * sizeof(*dims))) == NULL)
	 goto error;

      /*  Read and print dimensions  */
      if(fread(dims,sizeof(*dims),nDim,fpIn->FF_File) != nDim) goto error;

      for(i = 0 ; i < nDim ; i++)
	if(fprintf(fpOut,i == 0 ? "%ld" : " x %ld",dims[i]) < 0)
	  goto error;

      /*  Skip alignment value  */
      if((nDim & 1) && nDim != 1) {
	unsigned long dummy;

	if(fread(&dummy,sizeof(dummy),1,fpIn->FF_File) != 1)
	  goto error;
      }

      /*  Read and print boundary values  */
      if(flag & FLAG_SINGLEPREC) {
	for(i = 0 ; i <= nDim ; i++) {
	  float b[2];
	  char *t;

	  if(i == 0) t = " ["SFMT2","SFMT2"]";
	  else if(i < nDim) t = " x ["SFMT2","SFMT2"]";
	  else t = " - ["SFMT2","SFMT2"]\n";

	  if(fread(b,sizeof(b),1,fpIn->FF_File) != 1 ||
	     fprintf(fpOut,t,b[0],b[1]) < 0)
	    goto error;
	}
      } else {
	for(i = 0 ; i <= nDim ; i++) {
	  double b[2];
	  char *t;

	  if(i == 0) t = " ["DFMT2","DFMT2"]";
	  else if(i < nDim) t = " x ["DFMT2","DFMT2"]";
	  else t = " - ["DFMT2","DFMT2"]\n";

	  if(fread(b,sizeof(b),1,fpIn->FF_File) != 1 ||
	     fprintf(fpOut,t,b[0],b[1]) < 0)
	    goto error;
	}
      }

      /*  Print data  */
      for(dim = 1, i = 0 ; i < nDim ; i++) dim *= dims[i];
      if(flag & FLAG_SINGLEPREC) {
	float x;

	for(j = 0 ; j < dim ; j++)
	  if(fread(&x,sizeof(x),1,fpIn->FF_File) != 1 ||
	     fprintf(fpOut,SFMT"\n",x) < 0)
	    goto error;
      } else {
	double x;

	for(j = 0 ; j < dim ; j++)
	  if(fread(&x,sizeof(x),1,fpIn->FF_File) != 1 ||
	     fprintf(fpOut,DFMT"\n",x) < 0)
	    goto error;
      }

      /*  Print axe values if present  */
      if(flag & FLAG_AXEVALUES) {
	for(i = 0 ; i < nDim ; i++) {
	  if(fprintf(fpOut,"\n") < 0) goto error;

	  if(flag & FLAG_SINGLEPREC) {
	    float x;

	    for(j = 0 ; j < dims[i] ; j++)
	      if(fread(&x,sizeof(x),1,fpIn->FF_File) != 1 ||
		 fprintf(fpOut,SFMT"\n",x) < 0)
		goto error;
	  } else {
	    double x;

	    for(j = 0 ; j < dims[i] ; j++)
	      if(fread(&x,sizeof(x),1,fpIn->FF_File) != 1 ||
		 fprintf(fpOut,DFMT"\n",x) < 0)
		goto error;
	  }
	}
      }

      free(dims);
    }

    /*  Check if hunk length compatible with read data  */
    if(!(fpIn->FF_Flags & FPLOT_FLAG_NOFSEEK)) {
      long fPos2;

      if((fPos2 = ftell(fpIn->FF_File)) < 0) goto error;

      if(fPos2 > fPos) {
	goto error;
      } else if(fPos2 < fPos) {
	fprintf(stderr,"\
WARNING fplotRead: Hunk contains %ld garbage bytes -- Skipped.\n",
		fPos - fPos2);
	if(fseek(fpIn->FF_File,fPos,SEEK_SET)) goto error;
      }
    }
  }

endfound:
  return(0);
error:
  return(-1);
}
#undef FLAG_AXEVALUES
#undef FLAG_SINGLEPREC

/*JS*********************************************************************
*   TEXT2ASCII
*************************************************************************/

int text2Ascii(char *text,char marker,FILE *fpOut)

/************************************************************************/
{
  int flag;

  for(flag = 1 ; *text ; text++) {
    if(flag) {
      /*  Print start mark  */
      if(fputc(marker,fpOut) == EOF) goto error;
      flag = 0;
    }
    if(fputc(*text,fpOut) == EOF) goto error;

    /*	Check if line end  */
    if(*text == '\n') flag = 2;
  }

  if(flag == 0 && fputc('\n',fpOut) == EOF) goto error;

  /*  Mark lines ending with '\n'  */
  if(flag == 2 && (fputc(marker,fpOut) == EOF || fputc('\n',fpOut) == EOF))
    goto error;

  return(0);
error:
  return(-1);
}

/*JS*********************************************************************
*   PRNAMELINE Generate a line that contains the file's name and date for
*    ASCII output.
*************************************************************************/

int prNameLine(FILE *fpOut,char *name)

/************************************************************************/
{
#ifndef CONFIG_NO_POSIX
  struct stat stat_buf;
  struct tm *fTime;
  char buffer[FORMAT_DATE_TIME_MAX];

  if(stat(name,&stat_buf)) goto error;
#endif

  fprintf(fpOut,"FILE \"");

  /*  Print file's name and treat '"' correctly  */
  while(*name) {
    if(*name == '\"' && fputc('\\',fpOut) == EOF) goto error;

    if(fputc(*name,fpOut) == EOF) goto error;
    name++;
  }

#ifndef CONFIG_NO_POSIX
  /*  Print file's date  */
  fTime = localtime(&stat_buf.st_mtime);

  strftime(buffer,FORMAT_DATE_TIME_MAX,FORMAT_DATE_TIME,fTime);

  fprintf(fpOut,"\" %s\n",buffer);
#else
  fprintf(fpOut,"\"\n");
#endif

  return(0);
error:
  return(-1);
}

/*JS*********************************************************************
*   DMP2ASCII
*************************************************************************/

int dmp2Ascii(char *name,FILE *fpOut)

/************************************************************************/
{
  SpecDumpFile *specDump;
  JateDumpFile *jateDump;
  int i,j;

  if((specDump = SPReadDump(name))) {
    /* ***  SPEC DUMP  *** */
    /*	Print header  */
    fprintf(fpOut,"DUMP_SPEC: NMax %d, NrStates %d, Dim %d\n",
	    specDump->SDF_NMax,specDump->SDF_NrStates,specDump->SDF_Dim);

    /*	Print vector  */
    for(i = 0 ; i < specDump->SDF_Dim ; i++)
      fprintf(fpOut,DFMT2", "DFMT2"\n",RE(specDump->SDF_Vector[i]),
	      IM(specDump->SDF_Vector[i]));

    SPFreeDump(specDump);
  } else if((jateDump = JTReadDump(name,0,0,-1))) {
    /* ***  JATE DUMP  *** */
    fprintf(fpOut,"\
DUMP_JATE: Mode %d, NMax %d, VMax %d, Dim %d, NSave %d\n\
  JTildp3h %d, Omega "DFMT2", OmegaQuant "DFMT2", E_E "DFMT2", E_A "DFMT2"\n\
  K "DFMT2", G "DFMT2", L "DFMT2", F "DFMT2", DeltaDiag "DFMT2"\n",
	    jateDump->JDF_Mode,jateDump->JDF_NMax,jateDump->JDF_VMax,
	    jateDump->JDF_Dim,jateDump->JDF_NSave,jateDump->JDF_JTildp3h,
	    jateDump->JDF_Omega,jateDump->JDF_OmegaQuant,
	    jateDump->JDF_E_E,jateDump->JDF_E_A,
	    jateDump->JDF_K,jateDump->JDF_G,jateDump->JDF_L,jateDump->JDF_F,
	    jateDump->JDF_DeltaDiag);

    /*	Print quantum numbers  */
    for(i = 0 ; i < jateDump->JDF_Dim ; i++)
      fprintf(fpOut,"%3d %3d %c\n",jateDump->JDF_QNum[i].QN_N,
	      jateDump->JDF_QNum[i].QN_L,
	      ElecChar[1 + jateDump->JDF_QNum[i].QN_Elec]);

    /*	Print eigenvalues  */
    for(i = 0 ; i < jateDump->JDF_NSave ; i++)
      fprintf(fpOut,DFMT2"\n",jateDump->JDF_EigenValue[i]);

    for(j = 0 ; j < jateDump->JDF_NSave ; j++) {
      for(i = 0 ; i < jateDump->JDF_Dim ; i++)
	fprintf(fpOut,DFMT2"\n",
		jateDump->JDF_EigenVector[jateDump->JDF_Dim * j + i]);
    }
  } else {
    fprintf(stderr,"ERROR: Unknown dump format -- Break.\n");
    return(1);
  }

  return(0);
}

/*JS*********************************************************************
*   READASCII
*************************************************************************/

int readAscii(FILE *fpIn,int mode)

/************************************************************************/
{
  char lineBuffer[MAXLINE],*string,*string2;
  FILE *fp1;
  FPFile *ffp;
  int i,len,offset;

  fp1 = NULL;
  ffp = NULL;

  /*  Get offset in file name where to put extension  */
  if((offset = strlen(NameBuffer)) == 0)
    offset = -1; /*  No file name given  */

  for( ; fgets(lineBuffer,MAXLINE,fpIn) ; ) {
    LineCount++;

    /*	Skip spaces  */
    for(string = lineBuffer ; *string && isspace(*(unsigned char*)string) ; )
      string++;

    if(*string == '\0') continue; /*  empty line  */

    /*	Identify leading word  */
    for(i = 0 ; Markers[i] ; i++)
      if((string2 = (char *)strcompare(string,Markers[i]))) break;

    if(Markers[i] == NULL) {
      fprintf(stderr,"ERROR: Unknown line -- Skipped.\n\
%d >%s",LineCount,string);
      continue;
    }

    string = string2;

    switch(i) {
    case 0: /*  "file" line  */
      /*  If NameBuffer already set, ignore this line  */
      if(fp1) fclose(fp1);
      if(ffp) fplotClose(ffp);
      if(fp1 || ffp) {
	/*  Set file's time if required  */
	if(FileTime >= 0 && NameBuffer[0] &&
	   setFileDate(NameBuffer,FileTime)) goto error;

	NameBuffer[0] = '\0';
      }
      fp1 = NULL;
      ffp = NULL;

      if((mode & MODE_USENLINE) || NameBuffer[0] == '\0') {
	int day,month,year,hour,minute,second;

	while(isspace(*(unsigned char *)string)) string++;

	if(*string != '\"') goto syntError;
	string++;

	/*  Read name  */
	for(string2 = NameBuffer ; *string && *string != '\"' ; ) {
	  if(*string == '\\' && string[1]) *string2++ = *string++;

	  *string2++ = *string++;
	}
	*string2 = '\0';

	if(*string != '\"') goto syntError;
	string++;
	while(isspace(*(unsigned char *)string)) string++;

	/*  file's date present?  */
	if(*string) {
	  if(sscanf(string,FORMAT_DATE_TIME2,&day,&month,&year,
		    &hour,&minute,&second) != 6)
	    goto syntError;
	  FileTime = mktime2(year,month,day,hour,minute,second);
	} else
	  FileTime = -1;

	offset = -1;
      }
      break;
    case 1: /*	DUMP_SPEC  */
      if(NameBuffer[0] == '\0') goto noName;

      {
	SpecDumpHeader specHeader;
	dcmplx value;
	int vecDim,saveErrno;

	if(sscanf(string,": NMax %d, NrStates %d, Dim %d",
	   &specHeader.DH_NMax,&specHeader.DH_NrStates,&vecDim) != 3) {
	  fprintf(stderr,"ERROR l. %d: Cannot read header line -- Break.\n",
		  LineCount);
	  goto error;
	}

	specHeader.DH_Magic  = SPECDUMPMAGIC;
	len = sizeof(specHeader) + vecDim * sizeof(value);
	specHeader.DH_Length = HUNKLENGTH(len);

	/*  Generate dump name	*/
	if(offset >= 0) strcpy(NameBuffer + offset,DUMP_EXTENSION);

	if(!(mode & MODE_QUIET))
	  printf("Generating spec dump file \"%s\".\n",NameBuffer);

	if(fp1) goto syntError;

	saveErrno = errno;
#ifndef CONFIG_NO_POSIX
	if(access(NameBuffer,F_OK) == 0) {
#else
	if((fp1 = fopen(NameBuffer,"r"))) {
	  fclose(fp1); fp1 = NULL;
#endif
	  fprintf(stderr,"\
ERROR: File \"%s\" already exists, remove before next try -- Skipped.\n",
		  NameBuffer);
	  continue;
	}
	errno = saveErrno;

	if((fp1 = fopen(NameBuffer,"wb")) == NULL) {
	  fprintf(stderr,"ERROR: Could not open dump file \"%s\" -- Break.\n",
		  NameBuffer);
	  goto error;
	}

	len = HUNKPADDING(len);
	if(fwrite(&specHeader,sizeof(specHeader),1,fp1) != 1) goto error;

	/*  Read and write values  */
	for(i = 0 ; i < vecDim ; i++) {
	  if(fscanf(fpIn,"%lg,%lg ",&RE(value),&IM(value)) != 2) {
	    fprintf(stderr,"ERROR: Could not read complex values -- Break.\n");
	    goto error;
	  }
	  LineCount++;
	  if(fwrite(&value,sizeof(value),1,fp1) != 1) goto error;
	}

	if(len && fwrite(Padds,1,len,fp1) != len) goto error;

	fclose(fp1);
	fp1 = NULL;

	/*  Now set file's time if required  */
	if(FileTime >= 0) {
	  if(setFileDate(NameBuffer,FileTime)) goto error;
	  FileTime = -1;
	}
	NameBuffer[0] = '\0';
      }
      break;
    case 2: /*	DUMP_JATE  */
      if(NameBuffer[0] == '\0') goto noName;

      {
	JateDumpHeader jateHeader;
	int dim,j,saveErrno;

	if(sscanf(string,": Mode %d, NMax %d, VMax %d, Dim %d, NSave %d",
		  &jateHeader.DH_Mode,&jateHeader.DH_NMax,&jateHeader.DH_VMax,
		  &jateHeader.DH_Dim,&jateHeader.DH_NSave) != 5 ||
	   fscanf(fpIn,"\
 JTildp3h %d, Omega %lg, OmegaQuant %lg, E_E %lg, E_A %lg\n\
  K %lg, G %lg, L %lg, F %lg, DeltaDiag %lg\n",&jateHeader.DH_JTildp3h,
		  &jateHeader.DH_Omega,&jateHeader.DH_OmegaQuant,
		  &jateHeader.DH_E_E,&jateHeader.DH_E_A,
		  &jateHeader.DH_K,&jateHeader.DH_G,
		  &jateHeader.DH_L,&jateHeader.DH_F,&jateHeader.DH_DeltaDiag)
	   != 10) {
	  fprintf(stderr,"ERROR: Cannot read header line -- Break.\n");
	  goto error;
	}
	LineCount += 2;

	dim = jateHeader.DH_Dim;

	jateHeader.DH_Magic  = JATEDUMPMAGIC;
	jateHeader.DH_Dummy = 0;
	for(i = 0 ; i < JATEDUMP_UNUSED ; i++) jateHeader.DH_Unused[i] = 0.0;

	/*  Generate dump name	*/
	if(offset >= 0) strcpy(NameBuffer + offset,DUMP_EXTENSION);

	if(!(mode & MODE_QUIET))
	  printf("Generating jate dump file \"%s\".\n",NameBuffer);

	if(fp1) goto syntError;

	saveErrno = errno;
#ifndef CONFIG_NO_POSIX
	if(access(NameBuffer,F_OK) == 0) {
#else
	if((fp1 = fopen(NameBuffer,"r"))) {
	  fclose(fp1); fp1 = NULL;
#endif
	  fprintf(stderr,"\
ERROR: File \"%s\" already exists, remove before next try -- Skipped.\n",
		  NameBuffer);
	  continue;
	}
	errno = saveErrno;

	if((fp1 = fopen(NameBuffer,"wb")) == NULL) {
	  fprintf(stderr,"ERROR: Could not open dump file \"%s\" -- Break.\n",
		  NameBuffer);
	  goto error;
	}

	if(fwrite(&jateHeader,sizeof(jateHeader),1,fp1) != 1) goto error;

	/*  Read and write values  */
	for(i = 0 ; i < dim ; i++) {
	  QNumber qNum;
	  char c;

	  if(fscanf(fpIn,"%hd %hd %c\n",&qNum.QN_N,&qNum.QN_L,&c) != 3)
	    goto readerror;
	  LineCount++;

	  qNum.QN_Elec = strchr(ElecChar,c) - ElecChar - 1;
	  if(fwrite(&qNum,sizeof(qNum),1,fp1) != 1) goto error;
	}

	/*  Eigenvalues  */
	for(i = 0 ; i < jateHeader.DH_NSave ; i++) {
	  double value;

	  if(fscanf(fpIn,"%lg",&value) != 1) {
	    fprintf(stderr,"ERROR: Could not read double value %d -- Break.\n",
		    i);
	    goto error;
	  }
	  LineCount++;

	  if(fwrite(&value,sizeof(value),1,fp1) != 1) goto error;
	}

	/*  Eigenvectors  */
	for(j = 0 ; j < jateHeader.DH_NSave ; j++) {
	  for(i = 0 ; i < dim ; i++) {
	    double value;

	    if(fscanf(fpIn,"%lg",&value) != 1) {
	      fprintf(stderr,"ERROR: Could not read double value (%d,%d) -- Break.\n",j,i);
	      goto error;
	    }
	    LineCount++;
	    if(fwrite(&value,sizeof(value),1,fp1) != 1) goto error;
	  }
	}

	fclose(fp1);
	fp1 = NULL;

	/*  Now set file's time if required  */
	if(FileTime >= 0) {
	  if(setFileDate(NameBuffer,FileTime)) goto error;
	  FileTime = -1;
	}
	NameBuffer[0] = '\0';
      }
      break;
    case 3: /*	FPLOT out files  */
      if(ffp == NULL) {
	if(NameBuffer[0] == '\0') goto noName;

	if(offset >= 0) strcpy(NameBuffer + offset,FPLOT_EXTENSION);

	if(!(mode & MODE_QUIET))
	  printf("%s output file \"%s\".\n",(mode & MODE_APPEND) ?
		 "Appending to" : "Generating",NameBuffer);

	if((ffp = fplotOpen(NameBuffer,(mode & MODE_APPEND) ? FPLOT_APPEND :
			    FPLOT_WRITE)) == NULL) {
	  fprintf(stderr,"ERROR: Could not open output file \"%s\" -- Break.\n",
		  NameBuffer);
	  goto error;
	}

	if(fplotStart(ffp)) goto error;
      }

      /*  Transfer this hunk  */
      if(ascii2fplot(fpIn,string,ffp,mode)) goto error;
      break;
    }
  }

  if(fp1) fclose(fp1);
  if(ffp) fplotClose(ffp);
  if(fp1 || ffp) {
    /*  Now set file's time if required  */
    if(FileTime >= 0 && setFileDate(NameBuffer,FileTime)) goto error;
  }

  return(0);
readerror:
  fprintf(stderr,"ERROR in reading -- Break.\n");
error:
  return(-1);
syntError:
  fprintf(stderr,"ERROR l. %d: Syntax error -- Break.\n",LineCount);
  return(-1);
noName:
  fprintf(stderr,"ERROR l. %d: No file name given -- Break.\n",LineCount);
  return(-1);
}

/*JS*********************************************************************
*   ASCII2FPLOT
*************************************************************************/

int ascii2fplot(FILE *fpIn,char *string,FPFile *fpOut,int mode)

/************************************************************************/
{
  int i,type,nDim,flag;
#define FLAG_AXEVALUES	 1
#define FLAG_SINGLEPREC  2
  char *string2;
static char *HunkTypes[] = {
  "TEXT",      "AXES",
  "VECTORXY",  "SVECTORXY", "VECTOR",  "SVECTOR",
  "TENSORXY",  "STENSORXY", "TENSOR",  "STENSOR",
  "AXEDATA",   "SAXEDATA",
  "END",
  NULL
};

  /*  Recognize hunk type  */
  for(i = 0 ; HunkTypes[i] ; i++)
    if((string2 = (char *)strcompare(string,HunkTypes[i]))) break;

  if(HunkTypes[i] == NULL) {
    fprintf(stderr,"ERROR: Unknown hunk type encountered -- Break.\n\
%d >%s",LineCount,string);
    goto error;
  }

  /*  Skip colon too  */
  string = string2 + 1;
  flag = 0;
  nDim = 0;
  switch(i) {
  case  0: /*  HUNK_TEXT	*/
    {
      char *text;

      if(ascii2Text(MARK_TEXT,fpIn,&text) || text == NULL ||
	 fplotText(fpOut,"%s",text))
	goto readerror;

      free(text);
    }
    break;
  case  1: /*  HUNK_AXES	*/
    {
      char **axes;
      int axeMax,axeInd;

      axes = NULL;
      for(axeInd = axeMax = 0 ; ; axeInd++) {
	char *text;

	if(ascii2Text(MARK_AXE(axeInd),fpIn,&text)) goto readerror;

	if(text == NULL) break;

	if(axeInd >= axeMax) {
	  axeMax += 5;
	  if((axes = (char **)realloc(axes,axeMax * sizeof(*axes)))
	     == NULL) goto error;
	}
	axes[axeInd] = text;
      }

      if(fplotAxes(fpOut,axeInd,(const char **)axes)) goto error;

      for(i = 0 ; i < axeInd ; i++) free(axes[i]);
    }
    break;
  case  2: /*  HUNK_VECTORXY  */
  case  3: /*  HUNK_SVECTORXY  */
      flag |= FLAG_AXEVALUES;
  case  4: /*  HUNK_VECTOR  */
  case  5: /*  HUNK_SVECTOR  */
    if(i == 3 || i == 5) flag |= FLAG_SINGLEPREC;

    /*  Read defining values  */
    if(sscanf(string," (%d)%n",&type,&i) != 1) goto readerror;
    string += i;
    nDim = 1;
    break;
  case  6: /*  HUNK_TENSORXY  */
  case  7: /*  HUNK_STENSORXY  */
      flag |= FLAG_AXEVALUES;
  case  8: /*  HUNK_TENSOR  */
  case  9: /*  HUNK_STENSOR  */
    if(i == 7 || i == 9) flag |= FLAG_SINGLEPREC;

    if(sscanf(string," %d (%d)%n",&nDim,&type,&i) != 2) goto readerror;
    string += i;
    if(nDim <= 0) goto readerror;

    break;
  case 11: /*  HUNK_SAXEDATA  */
    flag |= FLAG_SINGLEPREC;
  case 10: /*  HUNK_AXEDATA  */
    {
      int index,n,len;
      OutputHeader oheader;

      if(sscanf(string,"%d %d",&index,&n) != 2) goto readerror;

      if(flag & FLAG_SINGLEPREC) {
	oheader.OH_Type = HUNK_SAXEDATA;
	len = sizeof(index) + sizeof(n) + n * sizeof(float);
      } else {
	oheader.OH_Type = HUNK_AXEDATA;
	len = sizeof(index) + sizeof(n) + n * sizeof(double);
      }
      oheader.OH_Length = HUNKLENGTH(len);

      /*  Write header  */
      if(fwrite(&oheader,sizeof(oheader),1,fpOut->FF_File) != 1 ||
	 fwrite(&index,sizeof(index),1,fpOut->FF_File) != 1 ||
	 fwrite(&n,sizeof(n),1,fpOut->FF_File) != 1) goto error;

      /*  Read data  */
      if(flag & FLAG_SINGLEPREC) {
	float x;

	for(i = 0 ; i < n ; i++) {
	  if(fscanf(fpIn,"%g",&x) != 1) goto readerror;
	  LineCount++;
	  if(fwrite(&x,sizeof(x),1,fpOut->FF_File) != 1) goto error;
	}
      } else {
	double x;

	for(i = 0 ; i < n ; i++) {
	  if(fscanf(fpIn,"%lg",&x) != 1) goto readerror;
	  LineCount++;
	  if(fwrite(&x,sizeof(x),1,fpOut->FF_File) != 1) goto error;
	}
      }

      /*   Write out padding bytes	  */
      if((len = oheader.OH_Length * HUNKROUND - len) > 0 &&
	 fwrite(Padds,1,len,fpOut->FF_File) != len) goto error;
    }
    break;
  case 12: /*  HUNK_END  */
    if(fplotEnd(fpOut) || fplotStart(fpOut)) goto error;
    break;
  }

  if(nDim > 0) {
    unsigned long *dims,dim,axeDim;
    int siz,len;
    void **axeData,*boundaries;
    char *p;

    if((dims = (unsigned long *)malloc(nDim * sizeof(*dims))) == NULL)
      goto error;

    /*  Read dimensions  */
    for(i = 0 ; i < nDim ; i++) {
      if(sscanf(string,i == 0 ? " %ld%n" : " x %ld%n",&dims[i],&len) != 1)
	goto readerror;
      string += len;
    }

    for(dim = 1, axeDim = 0, i = 0 ; i < nDim ; i++) {
      dim *= dims[i];
      axeDim += dims[i];
    }

    siz = (flag & FLAG_SINGLEPREC) ? sizeof(float) : sizeof(double);

    if(flag & FLAG_AXEVALUES) {
      i = (dim + axeDim + 2 * (nDim + 1)) * siz + nDim * sizeof(*axeData);
    } else {
      i = (dim + 2 * (nDim + 1))* siz;
    }
    if((p = (char *)malloc(i)) == NULL) goto error;

    if(flag & FLAG_AXEVALUES) {
      boundaries = &p[(dim + axeDim) * siz];

      axeData = (void *)&p[(dim + axeDim + 2 * (nDim + 1)) * siz];
      for(axeData[0] = (void *)&p[dim * siz], i = 1 ; i < nDim ; i++)
	axeData[i] = &((char *)(axeData[i - 1]))[dims[i - 1] * siz];
    } else {
      boundaries = &p[dim * siz];
      axeData = NULL;
    }

    /*  Read boundary values  */
    if(flag & FLAG_SINGLEPREC) {
      char *t;

      for(i = 0 ; i <= nDim ; i++) {
	if(i == 0) t = " [%g,%g]%n";
	else if(i < nDim) t = " x [%g,%g]%n";
	else t = " - [%g,%g]\n%n";

	if(sscanf(string,t,&((float *)boundaries)[2 * i],
		  &((float *)boundaries)[2 * i + 1],&len) != 2)
	  goto readerror;
	string += len;
      }
    } else {
      char *t;

      for(i = 0 ; i <= nDim ; i++) {
	if(i == 0) t = " [%lg,%lg]%n";
	else if(i < nDim) t = " x [%lg,%lg]%n";
	else t = " - [%lg,%lg]\n%n";

	if(sscanf(string,t,&((double *)boundaries)[2 * i],
		  &((double *)boundaries)[2 * i + 1],&len) != 2)
	  goto readerror;
	string += len;
      }
    }

    if(*string)
      fprintf(stderr,"WARNING: Garbage in header line ignored:\n>%s",string);

    /*  Read data  */
    if(flag & FLAG_SINGLEPREC) {
      float *ptr = (float *)p;

      for(i = 0 ; i < dim ; i++) {
	if(fscanf(fpIn,"%g",&ptr[i]) != 1) goto readerror;
	LineCount++;
      }
    } else {
      double *ptr = (double *)p;

      for(i = 0 ; i < dim ; i++) {
	if(fscanf(fpIn,"%lg",&ptr[i]) != 1) goto readerror;
	LineCount++;
      }
    }

    /*  Read axe values if present  */
    if(axeData) {
      for(i = 0 ; i < nDim ; i++) {
	int j;

	if(flag & FLAG_SINGLEPREC) {
	  float *ptr = axeData[i];

	  for(j = 0 ; j < dims[i] ; j++) {
	    if(fscanf(fpIn,"%g",&ptr[j]) != 1) goto readerror;
	    LineCount++;
	  }
	} else {
	  double *ptr = axeData[i];

	  for(j = 0 ; j < dims[i] ; j++) {
	    if(fscanf(fpIn,"%lg",&ptr[j]) != 1) goto readerror;
	    LineCount++;
	  }
	}
      }
    }

    /*  Write data  */
    if(flag & FLAG_SINGLEPREC) {
      if(fplotSTensor(fpOut,type,nDim,(const float **)axeData,(float *)p,
		      dims,NULL,(const float *)boundaries,
		      ((float *)boundaries)[2 * nDim],
		      ((float *)boundaries)[2 * nDim + 1]))
	goto error;
    } else {
      if(fplotTensor(fpOut,type,nDim,(const double **)axeData,(double *)p,
		     dims,NULL,(const double *)boundaries,
		     ((double *)boundaries)[2 * nDim],
		     ((double *)boundaries)[2 * nDim + 1]))
	goto error;
    }

    free(p);
    free(dims);
  }

  return(0);
readerror:
  fprintf(stderr,"ERROR l. %d (%ld): Corrupted data -- Break.\n",LineCount,
	  ftell(fpIn));
error:
  return(-1);
}
#undef FLAG_AXEVALUES
#undef FLAG_SINGLEPREC

/*JS*********************************************************************
*   ASCII2TEXT Reads a couple of lines starting with marker and stores
*    allocated text space in variable pText points to. Does not save the
*    last '\n' of the text!
*************************************************************************/

int ascii2Text(char marker,FILE *fpIn,char **pText)

/************************************************************************/
{
  char *text;
  int current,maxSize,c,flag;

  text = NULL;
  maxSize = 0;

  for(flag = TRUE, current= 0 ; ; ) {
    if((c = getc(fpIn)) == EOF) break;

    /*  Check for marker  */
    if(flag && c != marker) {
      ungetc(c,fpIn);
      break;
    }

    /*	Increase text buffer  */
    if(current >= maxSize) {
      maxSize = MAX(current + MAXLINE,2 * maxSize);
      if((text = (char *)realloc(text,maxSize)) == NULL) return(-1);
    }

    if(flag) {
      /*  Save EOL from last line  */
      if(current) text[current++] = '\n';

      flag = FALSE;
    } else if(c == '\n') {
      flag = TRUE;
      LineCount++;
    } else
      text[current++] = c;
  }

  if(text) text[current] = '\0';

  *pText = text;

  return(0);
}

/*JS*********************************************************************
*   SETFILEDATE
*************************************************************************/

int setFileDate(char *name,time_t tim)

/************************************************************************/
{
#ifndef CONFIG_NO_POSIX
  struct utimbuf timbuf;

  timbuf.actime = timbuf.modtime = tim;
  if(utime(name,&timbuf)) return(-1);
#endif

  return(0);
}

#define BUFFSIZE    8192
/*JS*********************************************************************
*   DUMMYFSEEK Do a dummy fseek from current position by reading and
*    discarding the data.
*************************************************************************/

static int dummyFseek(FILE *fp,long offset)

/************************************************************************/
{
  char *buff;

#ifdef CONFIG_USE_ALLOCA
  if((buff = (char *)alloca(BUFFSIZE)) == NULL) return(-1);
#else
  if((buff = (char *)malloc(BUFFSIZE)) == NULL) return(-1);
#endif

  while(offset > 0) {
    long len = MIN(offset,BUFFSIZE);

    if(fread(buff,1,len,fp) != len) goto error;
    offset -= len;
  }

#ifndef CONFIG_USE_ALLOCA
  free(buff);
#endif

  return(0);
error:
#ifndef CONFIG_USE_ALLOCA
  free(buff);
#endif

  return(-1);
}
