/*JS*********************************************************************
*
*    Program : COMPEXPR
*    Language: ANSI-C
*    Author  : Joerg Schoen
*    Purpose : Compile arithmetric expressions for fast interpreting.
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id: compexpr.c,v 1.13 1997/08/02 21:41:56 joerg Stab joerg $";
#endif
/*********     INCLUDES                                         *********/
#include <jsconfig.h>

#include <jsalloca.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <math.h>

#include <jssubs.h>

/*********     DEFINES                                          *********/
/*#define DEBUG*/

/*  Use function "intpow" if exponent is an integer value  */
/*#define CONFIG_USE_INTPOW*/

#ifndef _typedef_Expression
/*  START-DEFINITIONS */
typedef struct {
  char          *EX_Name;  /*  When this expression is a function, the name */
  int            EX_NVars; /*  Number of variables  */
  int            EX_MaxStack; /*  Maximal stack depth needed  */
  double        *EX_Consts;   /*  Table of contant values  */
  unsigned long *EX_IpnCode; /* Code table for interpreting this expression */
} Expression;
#define _typedef_Expression
/*  END-DEFINITIONS */
#endif

/*  Internal states  */
#define STATE_OP   1
#define STATE_VAR  2

/*  Types of IPN codes  */
#define TYPE_MASK    0xff000000UL
#define VALUE_MASK   0x00ffffffUL

#define TYPE_OP      0x00000000UL
#define TYPE_CONST   0x01000000UL
#define TYPE_VAR     0x02000000UL
#define TYPE_FCALL   0x03000000UL
#define TYPE_UCALL   0x04000000UL

/*  Implemented operators  */
#define OPERATORS ",+-*/^"

#define OP_COMMA    4UL
#define OP_PLUS     8UL
#define OP_MINUS   12UL
#define OP_TIMES   16UL
#define OP_DIVIDE  20UL

#define OP_NEGATE  23UL  /*  Unary minus  */

#define OP_POWER   24UL
#define OP_MAX     24UL

/*  Allowed characters for identifier  */
#define isident(c)  ((c) == '_' || isalpha(c))

#ifndef DEBUG
# define push(c)   (stack[stackIndex++] = (c))
# define pop()     stack[--stackIndex]
# define lookup()  stack[stackIndex - 1]
#else
# define push(c)   (printf("push %d\n",stackIndex),stack[stackIndex++] = (c))
# define pop()     (printf("pop %d\n",stackIndex - 1),stack[--stackIndex])
# define lookup()  (printf("look %d\n",stackIndex - 1),stack[stackIndex - 1])
#endif


/* ERROR-DEFINITIONS from CompExpression label _ERR_COMPEXPR ord 6
   Wrong definition
   Wrong argument list
   Wrong function definition
   No arguments to function call
   Unknown identifier
   No identifier
   Comma operator only allowed in function call
   Incompatible number of arguments in function call
   Invalid expression
*/

#define Prototype extern
#define Local static
/*********     PROTOTYPES                                       *********/
Prototype Expression    *CompDefinition(const char *input,
					const Expression *fcts[],
					const char **pEnd);
Prototype Expression    *CompExpression(const char *input,const char *vars[],
                                        const Expression *fcts[],
					const char **pEnd);
Local int                getString(const char *input);

Prototype void           ShowExpression(const Expression *expr,
					const char *vars[],
                                        const Expression *fcts[]);

Prototype double         EvalExpression(const Expression *expr,
					const double vars[],double *stack,
					const Expression *fcts[]);


Prototype int            ExpressionMode;

/*********     STATIC / GLOBAL VARIABLES                        *********/
static const char *CExprOperators = OPERATORS;

int ExpressionMode = TRUE; /*  Set if expression will be simplified  */

/* ***  Internal function definitions  *** */
/*  Maximum function  */
static double internalMax(double a[],int n)
{ if(a[0] > a[1]) return(a[0]); else return(a[1]); }

/*  Minimum function  */
static double internalMin(double a[],int n)
{ if(a[0] < a[1]) return(a[0]); else return(a[1]); }

/*  Arctan with two arguments  */
static double atan2wrap(double a[],int n)
{ return(atan2(a[0],a[1])); }

/*  Theta function  */
static double theta(double x)
{ return(x >= 0.0 ? 1.0 : 0.0); }

/*  Constant PI  */
static double func_pi(double  a[],int n)
{ return(PI); }

#define E1(name)            { #name, 1, { name  } }
#define E2(name,name2)      { #name, 1, { name2 } }
#define E3(name,n,name2)    { #name, n, { (double (*)(double))name2 } }
static struct InternalFunctions {
  char    *IF_Name;
  int      IF_NVars;
  union {
    double (*if_function)(double);
    double (*if_functionN)(double [],int n);
#define IF_Function   IF_un.if_function
#define IF_FunctionN  IF_un.if_functionN
  } IF_un;
} InternalFunctions[] = {
  /*  Functions from ANSI math library  */
  E2(abs,fabs),
  E1(exp),  E1(log),  E1(sqrt),
  E1(sin),  E1(cos),  E1(tan),
  E1(asin), E1(acos), E1(atan),
  E1(sinh), E1(cosh), E1(tanh),
  E3(atan2,2,atan2wrap),
  /*  Some useful internal functions  */
  E3(max,2,internalMax),
  E3(min,2,internalMin),
  E1(theta),
  E1(errfkt),
  E3(pi,0,func_pi),
  { NULL, 0, { NULL } }
};
#undef E1
#undef E2
#undef E3

/*JS*********************************************************************
*   Additionally to the functionality of the 'CompExpression' routine this
*    routine allows 'input' to contain a function definition in the form
*    "f(x,y)= ...". On return this may be recognized when checking the
*    'expr->EX_Name' field.
*************************************************************************/

Expression *CompDefinition(const char *input,const Expression *fcts[],
			   const char **pEnd)

/************************************************************************/
{
  Expression *expr;
  char *name,*string,**vars;
  int len,nvars,i;

  name = NULL;
  vars = NULL;
  nvars = 0;

  /*  Check if this is going to be a definition  */
  if(strchr(input,'=')) {
    while(isspace(*(unsigned char *)input)) input++;

    /*  Get definition name  */
    if((len = getString(input)) <= 0) {
      JSErrNo = _ERR_COMPEXPR + 0;
      goto error;
    }

    if((name = (char *)malloc(len + 1)) == NULL) goto error;
    strncpy(name,input,len);
    name[len] = '\0';

    /*  Check next character  */
    for(input += len ; isspace(*(unsigned char *)input) ; ) input++;

    if(*input == '(') {
      input++;

      /*  Definition of function  */
      while(isspace(*(unsigned char *)input)) input++;

      /*  Count arguments  */
      for(nvars = 1, string = (char *)input ; *string && *string != ')' ;
          string++)
        if(*string == ',') nvars++;

      if((vars = (char **)malloc((nvars + 1) *
				 sizeof(*vars) + (string - input))) == NULL)
        goto error;

      string = (char *) &vars[nvars + 1];

      for(i = 0 ; i < nvars && (len = getString(input)) > 0 ; i++) {
        /*  Set up next variable  */
        vars[i] = string;
        strncpy(string,input,len);
        string[len] = '\0';
        string += len + 1;
        input += len;

        while(isspace(*(unsigned char *)input)) input++;

        /*  Check for correct delimiter  */
        if((i == (nvars - 1)) ? *input != ')' : *input != ',') {
	  JSErrNo = _ERR_COMPEXPR + 1;
          goto error;
        }
        input++;

        while(isspace(*(unsigned char *)input)) input++;
      }

      vars[nvars] = NULL;
    }

    if(*input != '=') {
      JSErrNo = _ERR_COMPEXPR + 2;
      goto error;
    }
    input++;

    /*  Now compile expression  */
    if((expr = CompExpression(input,(const char **)vars,fcts,pEnd))) {
      /*  Set up expression correctly  */
      expr->EX_Name = name;
      expr->EX_NVars = nvars;
    }

    /*  Free variable list  */
    if(vars) free(vars);
  } else {
    /*  Simply call compExpression  */
    expr = CompExpression(input,NULL,fcts,pEnd);
  }

  return(expr);
error:
  if(name) free(name);
  if(vars) free(vars);
  return(NULL);
}

/*JS*********************************************************************
*   The null terminated arithmetric expression at 'input' is compiled for
*    fast evaluation and returned. 'vars' may be null or contains a null
*    terminated list of recognized variables that may be used within the
*    expression. Similar 'fcts' may be null or contains a null terminated
*    list of functions for usage in the expression.
*************************************************************************/

Expression *CompExpression(const char *input,const char *vars[],
			   const Expression *fcts[],const char **pEnd)

/************************************************************************/
{
  Expression *expr;
  double *constTable,x;
  long ipnIndex,ipnMax,stackMax,stackIndex,constMax,constIndex;
  unsigned long *stack,*ipnTable;
  long i,len,flag,bracketCount;
  char c;

  /*  Preset variables for error case  */
  stack = NULL;
  stackIndex = 0;

  /*  Initialize table of constants  */
  constMax = constIndex = 0;
  constTable = NULL;

  /*  Initialize IPN table  */
  ipnIndex = ipnMax = 0;
  ipnTable = NULL;

  /*  Allocate expression and set up  */
  if((expr = (Expression *)malloc(sizeof(*expr))) == NULL) goto error;
  expr->EX_Name = NULL;
  expr->EX_NVars = 0;

  if(vars)
    while(vars[expr->EX_NVars])
      (expr->EX_NVars)++;

  /*  Initialize stack  */
  stackMax = 10;
  if((stack = (unsigned long *)malloc(stackMax * sizeof(*stack))) == NULL)
    goto error;

  /*  Push start mark on stack  */
  push(0);

  /* *************** Mainlooping *********************** */
  for(bracketCount = 0, flag = STATE_VAR ; flag ; ) {
    /*  Check if enough memory in output queue  */
    /*  At most we have to copy whole stack content on  */
    /*   output queue, so allocate for worst case       */
    if((ipnIndex + stackIndex) >= ipnMax) {
      ipnMax = MAX(ipnMax + 10,ipnIndex + stackIndex + 2);
      if((ipnTable = (unsigned long *)realloc(ipnTable,ipnMax *
					      sizeof(*ipnTable))) == NULL)
	goto error;
    }

    /*  Check if enough memory for constant allocation  */
    if(constIndex >= constMax) {
      constMax += 20; /*    Increase table  */
      if((constTable = (double *)realloc(constTable,constMax *
					 sizeof(*constTable))) == NULL)
	goto error;
    }

    /*  Check if enough memory on stack  */
    if(stackIndex >= stackMax) {
      stackMax += 10;
      if((stack = (unsigned long *)realloc(stack,stackMax * sizeof(*stack)))
	 == NULL) goto error;
    }

    /*  Skip spaces and get next input character  */
    while(isspace(*(unsigned char *)input)) input++;
    c = *input;

#ifdef DEBUG
    printf("INPUT %c (%d): s=%d (<%d), i=%d (<%d)\n",c,c,stackIndex,stackMax,
	   ipnIndex,ipnMax);
#endif

    if(flag == STATE_VAR) {
      /* ***  READ VARIABLE  *** */

      /*  New subexpression?  */
      if(c == '(') {
        /*  Start new expression by putting operator with low priority  */
        push(0);
	bracketCount++;

        /*  Don't change state and skip parenthesis  */
        input++;
        continue;
      }

      /*  Previous function call and no parenthesis -- that's wrong  */
      if((lookup() & TYPE_MASK) == TYPE_FCALL ||
	 (lookup() & TYPE_MASK) == TYPE_UCALL) {
	JSErrNo = _ERR_COMPEXPR + 3;
	goto error;
      }

      /*  Check for unary minus  */
      if(c == '-') {
        /*  Pop operators with higher priority from stack  */
        while(lookup() > OP_MAX) ipnTable[ipnIndex++] = pop();

        push(OP_NEGATE);
        input++;

        continue;
      }

      /*  Check if number  */
      if(isdigit((unsigned char)c) || c == '.') {
        /*  Get number  */
        x = strtod(input,(char **)&input);

        /*  Set correct type    */
        i = TYPE_CONST + constIndex;

        /*  Put constant in table  */
        constTable[constIndex++] = x;
      } else if((len = getString(input)) > 0) {
        if(vars)
          /*  Search for this variable  */
          for(i = 0 ; vars[i] ; i++) {
	    char *str;
            if((str = (char *)strcompare(input,vars[i])) &&
	       (str - input) == len) break;
	  }

        if(vars && vars[i]) {
          /*  Variable found  */
          i |= TYPE_VAR;
          input += len;
        } else {
          if(fcts)
            /*  Search for function name  */
            for(i = 0 ; fcts[i] ; i++) {
	      char *str;
              if((str = (char *)strcompare(input,fcts[i]->EX_Name)) &&
		 (str - input) == len) break;
	    }

          if(fcts && fcts[i]) { /*  User supplied function  */
            /*  Function with no arguments?  */
            if(fcts[i]->EX_NVars == 0) {
	      input += len;

	      /*  Evaluate constants here  */
	      constTable[constIndex] =
		EvalExpression(fcts[i],NULL,NULL,(const Expression **)fcts);

	      i = TYPE_CONST + constIndex++;
	      goto putonqueue;
	    }

	    i |= TYPE_UCALL;
          } else {
            /*  Search for internal function  */
            for(i = 0 ; InternalFunctions[i].IF_Name ; i++) {
	      char *str;
              if((str = (char *)strcompare(input,InternalFunctions[i].IF_Name))
		 && (str - input) == len) break;
	    }

            if(InternalFunctions[i].IF_Name == NULL) {
	      JSErrNo = _ERR_COMPEXPR + 4;
              goto error;
            }

	    /*  Internal function with no argument?  */
	    if(InternalFunctions[i].IF_NVars == 0) {
	      input += len;

	      /*  Evaluate constants here  */
	      constTable[constIndex] =
		(*InternalFunctions[i].IF_FunctionN)(NULL,0);

	      i = TYPE_CONST + constIndex++;
	      goto putonqueue;
	    }

            /*  Internal function  */
            i |= TYPE_FCALL;
          }

          /*  Pop operators with higher priority from stack  */
          while(lookup() > OP_MAX) ipnTable[ipnIndex++] = pop();

          /*  Push this function call on stack  */
          push(i);

          input += len;
          continue; /*  Function call is like pre operator  */
        }
      } else {
	JSErrNo = _ERR_COMPEXPR + 5;
        goto error;
      }

putonqueue:
      /*  Put expression in output queue  */
      ipnTable[ipnIndex++] = i;

      flag = STATE_OP;
    } else {
      char *string;

      /* ***  READ OPERATOR  *** */
      /*  Look in table for operator  */
      if(c != ')' && c != '\0' && (string = strchr(CExprOperators,c)) ==NULL) {
	/*  Maybe insert default operator '*'  */
	if(c == '(' || (c == '.' && isdigit(input[1])) || isdigit(c) ||
	   isident(c)) {
	  string = strchr(CExprOperators,c = '*');
	} else {
	  /*  Mark end of input  */
	  c = '\0';
	}
      } else if(c == ',' && bracketCount == 0) {
	/*  This comma belongs not to us  */
	c = '\0';
      } else if(c)
	/*  Skip input character (if not end of input)  */
	input++;

      /*  Check if end of expression  */
      if(c == ')' || c == '\0') {
        long s;

        /*  Don't change current state  */

        /*  Now pop until empty stack content  */
        while((s = pop()) && s != OP_COMMA) ipnTable[ipnIndex++] = s;

        if(s == OP_COMMA) {
          /*  Count number of arguments  */
          for(i = 2 ; (s = pop()) && s == OP_COMMA ; i++)
            /*  empty  */;

	  if(stackIndex > 0 && s == 0) s = pop();

	  /*  When bracket found, skip, otherwise check directly  */
          if(stackIndex <= 0 || ((s & TYPE_MASK) != TYPE_UCALL &&
				 (s & TYPE_MASK) != TYPE_FCALL)) {
	    JSErrNo = _ERR_COMPEXPR + 6;
	    goto error;
	  } else if(((s & TYPE_MASK) == TYPE_UCALL ?
		     fcts[s & VALUE_MASK]->EX_NVars :
		     InternalFunctions[s & VALUE_MASK].IF_NVars) != i) {
	    JSErrNo = _ERR_COMPEXPR + 7;
            goto error;
          }

          /*  Save function call on stack  */
          ipnTable[ipnIndex++] = s;
        } else if(stackIndex > 1 &&
		  ((s = lookup()) & TYPE_MASK) == TYPE_UCALL &&
                  fcts[s & VALUE_MASK]->EX_NVars != 1) {
	  JSErrNo = _ERR_COMPEXPR + 7;
          goto error;
        }

        /*  If end of input, break main looping  */
        if(c == '\0') {
	  flag = 0;
	} else if(stackIndex == 0) {
	  /*  Last ')' has no counterpart - do not read and break  */
	  input--;
	  flag = 0;
	} else
	  bracketCount--;
      } else {
	/*  Get index of operator  */
        i = (1 + (string - CExprOperators)) << 2;

        if(i == OP_COMMA) {
          /*  Special treatment of comma: collect expressions on stack  */
          /*  Pop operators with higher priority from stack  */
          while(lookup() > i) ipnTable[ipnIndex++] = pop();
	} else {
          /*  Pop operators with higher priority from stack  */
          while(lookup() >= i) ipnTable[ipnIndex++] = pop();
        }

        /*  And now push operator  */
        push(i);

        /*  Change state  */
        flag = STATE_VAR;
      }
    }
  }

  /*  Check if expression was correctly end. The final '\0'
   *   or an unknown operator popped our 0, so we expect
   *   stackIndex to be zero.
   */
  if(stackIndex != 0 || bracketCount != 0) {
    JSErrNo = _ERR_COMPEXPR + 8;
    goto error;
  }

  free(stack);
  stack = NULL;

  /* ***  Simplify constant sub expressions ? *** */
  if(ExpressionMode) {
    /*  Preset counter for scanning of expression  */
    ipnMax = ipnIndex; /*  save end of IPN queue  */
    constIndex = 0;    /*  constants will be counted again  */

    for(ipnIndex = i = 0 ; i < ipnMax ; i++) {
      unsigned long type;

      /* ***  Simplify unary operator  *** */
      if(ipnTable[i] == (TYPE_OP + OP_NEGATE) &&
	 (ipnTable[ipnIndex - 1] & TYPE_MASK) == TYPE_CONST) {
	int nr;

	/*  Unary minus  */
	nr = ipnTable[ipnIndex - 1] & VALUE_MASK;
	constTable[nr] = -constTable[nr];

	continue; /*  Redo looping  */
      }

      type = (ipnTable[i] & TYPE_MASK);

      /* ***  Simplify dyadic operators  *** */
      if(type == TYPE_OP &&
	 (ipnTable[ipnIndex - 2] & TYPE_MASK) == TYPE_CONST &&
	 (ipnTable[ipnIndex - 1] & TYPE_MASK) == TYPE_CONST) {
	switch((ipnTable[i] & VALUE_MASK)) {
	case OP_PLUS:
	  constTable[ipnTable[ipnIndex - 2] & VALUE_MASK] +=
	    constTable[ipnTable[ipnIndex - 1] & VALUE_MASK];
	  break;
	case OP_MINUS:
	  constTable[ipnTable[ipnIndex - 2] & VALUE_MASK] -=
	    constTable[ipnTable[ipnIndex - 1] & VALUE_MASK];
	  break;
	case OP_TIMES:
	  constTable[ipnTable[ipnIndex - 2] & VALUE_MASK] *=
	    constTable[ipnTable[ipnIndex - 1] & VALUE_MASK];
	  break;
	case OP_DIVIDE:
	  constTable[ipnTable[ipnIndex - 2] & VALUE_MASK] /=
	    constTable[ipnTable[ipnIndex - 1] & VALUE_MASK];
	  break;
	case OP_POWER:
	  {
#ifdef CONFIG_USE_INTPOW
	    long iVal;

	    if((double)(iVal = (long)floor(constTable[ipnTable[ipnIndex - 1] &
						     VALUE_MASK]))
	       == constTable[ipnTable[ipnIndex - 1] & VALUE_MASK])
	      /*  Use faster integer exponentiation  */
	      constTable[ipnTable[ipnIndex - 2] & VALUE_MASK] =
		intpow(constTable[ipnTable[ipnIndex - 2] & VALUE_MASK],iVal);
	    else
#endif
	      constTable[ipnTable[ipnIndex - 2] & VALUE_MASK] =
		pow(constTable[ipnTable[ipnIndex - 2] & VALUE_MASK],
		    constTable[ipnTable[ipnIndex - 1] & VALUE_MASK]);
	  }
	  break;
	}

	/*  We have deleted one entry in IPN queue and one constant  */
	ipnIndex--;
	constIndex--;

	continue; /*  Redo looping  */
      }

      /* ***  Simplify constant function call (User function)  *** */
      if(type == TYPE_UCALL) {
	/*  Check if all arguments constant  */
	for(flag = fcts[ipnTable[i] & VALUE_MASK]->EX_NVars ;
	    flag > 0 ; flag--)
	  if((ipnTable[ipnIndex - flag] & TYPE_MASK) != TYPE_CONST) break;

	if(flag == 0) {
	  ipnIndex   -= fcts[ipnTable[i] & VALUE_MASK]->EX_NVars - 1;
	  constIndex -= fcts[ipnTable[i] & VALUE_MASK]->EX_NVars - 1;

	  /*  Evaluate function call  */
	  constTable[constIndex - 1] =
	    EvalExpression(fcts[ipnTable[i] & VALUE_MASK],
			   &constTable[constIndex - 1],NULL,
			   (const Expression **)fcts);

	  continue; /*  Redo looping  */
	}
      }

      /* ***  Simplify constant function call (Intrinsic function)  *** */
      if(type == TYPE_FCALL) {
	/*  Check if all arguments constant  */
	for(flag = InternalFunctions[ipnTable[i] & VALUE_MASK].IF_NVars ;
	    flag > 0 ; flag--)
	  if((ipnTable[ipnIndex - flag] & TYPE_MASK) != TYPE_CONST) break;

	if(flag == 0) {
	  int nVars;

	  nVars = InternalFunctions[ipnTable[i] & VALUE_MASK].IF_NVars;

	  /*  Call function  */
	  if(nVars == 1) {
	    constTable[constIndex - 1] =
	      (*(InternalFunctions[ipnTable[i] & VALUE_MASK].IF_Function))
		(constTable[constIndex - 1]);
	  } else {
	    ipnIndex   -= nVars - 1;
	    constIndex -= nVars - 1;

	    constTable[constIndex - 1] =
	      (*(InternalFunctions[ipnTable[i] & VALUE_MASK].IF_FunctionN))
		(&constTable[constIndex - 1],nVars);
	  }
	  continue; /*  Redo looping  */
	}
      }

      /* ***  Copy constants (positions may change!)  *** */
      if(type == TYPE_CONST) {
	/*  Copy constant value and recalculate entry  */
	constTable[constIndex] = constTable[ipnTable[i] & VALUE_MASK];
	ipnTable[i] = (ipnTable[i] & ~VALUE_MASK) + constIndex;

	constIndex++;
      }

      /*  Default if not simplified: Copy next value from IPN queue  */
      ipnTable[ipnIndex++] = ipnTable[i];
    }
  } /* ***  END SIMPLIFY  *** */

  /* ***  Calculate maximum of needed stack  *** */
  stackMax = stackIndex = 0;
  for(i = 0 ; i < ipnIndex ; i++) {
    switch(ipnTable[i] & TYPE_MASK) {
    case TYPE_CONST:  case TYPE_VAR:
      stackIndex++;
      break;
    case TYPE_OP:
      /*  Not unary negation operator?  */
      if(ipnTable[i] != (TYPE_OP + OP_NEGATE)) stackIndex--;
      break;
    case TYPE_UCALL: /*  User function call  */
      /*  How much stack will function call take?  */
      len = stackIndex + fcts[ipnTable[i] & VALUE_MASK]->EX_MaxStack;
      if(len > stackMax) stackMax = len;

      stackIndex -= fcts[ipnTable[i] & VALUE_MASK]->EX_NVars - 1;
      break;
    }

    if(stackIndex > stackMax) stackMax = stackIndex;
  }

  /*  End IPN queue with empty expression  */
  ipnTable[ipnIndex] = 0;

  /*  Now minimize memory requirements  */
  ipnTable = (unsigned long *)realloc(ipnTable,(ipnIndex + 1) *
				      sizeof(*ipnTable));
  if(constIndex > 0) {
    constTable = (double *)realloc(constTable,constIndex *
				   sizeof(*constTable));
  } else {
    /*  The expression contains no constants  */
    free(constTable);
    constTable = NULL;
  }

  /*  Initialize structure  */
  expr->EX_MaxStack = stackMax;
  expr->EX_Consts  = constTable;
  expr->EX_IpnCode = ipnTable;

  /*  Return end of string  */
  if(pEnd) *pEnd = input;

  return(expr);
error:
#ifdef DEBUG
  fprintf(stderr,"Content of output queue (#%d):\n",ipnIndex);
  for(i = 0 ; i < ipnIndex ; i++)
    fprintf(stderr," q[%d] = %#x\n",i,ipnTable[i]);

  fprintf(stderr,"Content of stack (#%d):\n",stackIndex);
  for(i = 0 ; i < stackIndex ; i++)
    fprintf(stderr," s[%d] = %#x\n",i,stack[i]);
#endif

  if(expr) free(expr);
  if(ipnTable) free(ipnTable);
  if(stack) free(stack);
  if(constTable) free(constTable);

  return(NULL);
}

/*JS*********************************************************************
*   Returns the length of the identifier on position input.
*************************************************************************/

Local int getString(const char *input)

/************************************************************************/
{
  const char *string;

  for(string = input ; isident(*(unsigned char *)string) ; ) {
    do {
      string++;
    } while(isdigit(*(unsigned char *)string));
  }

  return(string - input);
}

/*JS*********************************************************************
*   The compiled expression 'expr' is shown (inverse polish notation). If
*    'vars' and 'fcts' are given (both may be null) the output is more
*    verbose.
*************************************************************************/

void ShowExpression(const Expression *expr,const char *vars[],
		    const Expression *fcts[])

/************************************************************************/
{
  int i,code;

  if(expr->EX_Name || expr->EX_NVars)
    printf("\"%s\"(#%d) [stack #%d] =",
           expr->EX_Name ? expr->EX_Name : "",
           expr->EX_NVars,expr->EX_MaxStack);
  else
    printf("[stack #%d] ",expr->EX_MaxStack);

  for(i = 0 ; (code = expr->EX_IpnCode[i]) ; i++) {
    switch(code & TYPE_MASK) {
    case TYPE_OP:
      if(code == OP_NEGATE)
        printf(" ~");
      else
        printf(" %c",CExprOperators[((code & VALUE_MASK) - 1) >> 2]);
      break;
    case TYPE_CONST:
      printf(" %g",expr->EX_Consts[code & VALUE_MASK]);
      break;
    case TYPE_VAR:
      if(vars)
        printf(" %s",vars[code & VALUE_MASK]);
      else
        printf(" v[%ld]",code & VALUE_MASK);
      break;
    case TYPE_FCALL:
      printf(" %s[#%d]",InternalFunctions[code & VALUE_MASK].IF_Name,
	     InternalFunctions[code & VALUE_MASK].IF_NVars);
      break;
    case TYPE_UCALL:
      if(fcts)
        printf(" %s[#%d]",fcts[code & VALUE_MASK]->EX_Name,
               fcts[code & VALUE_MASK]->EX_NVars);
      else
        printf(" f[%ld][#?]",code & VALUE_MASK);
      break;
    default:
      printf(" ?%d?",code);
      break;
    }
  }

  printf("\n");

  return;
}

/*JS*********************************************************************
*   The expression 'expr' is evaluated with the variables given by the
*    table 'vars' and the functions given by 'fcts'. The stack must at
*    least have the size 'expr->EX_MaxStack' but may be null, so that it
*    is allocated with malloc or alloca.
*************************************************************************/

double EvalExpression(const Expression *expr,const double vars[],
		      double *stack,const Expression *fcts[])

/************************************************************************/
{
#ifndef CONFIG_USE_ALLOCA
  double *saveStack;
#endif
  double x;
  unsigned long *table;

  /*  Allocate stack if not given  */
#ifdef CONFIG_USE_ALLOCA
  if(stack == NULL && (stack = (double *)alloca(expr->EX_MaxStack *
						sizeof(*stack)))
#else
  if(stack) saveStack = NULL;
  else if((saveStack = stack = (double *)malloc(expr->EX_MaxStack *
						sizeof(*stack)))
#endif
	  == NULL)
    return(HUGE_VAL);

  for(table = expr->EX_IpnCode ; *table ; table++) {
    long i;

    switch(*table & TYPE_MASK) {
    case TYPE_OP:
      stack--;

      switch((*table & VALUE_MASK)) {
      case OP_PLUS:   stack[-1] += stack[0]; break;
      case OP_MINUS:  stack[-1] -= stack[0]; break;
      case OP_TIMES:  stack[-1] *= stack[0]; break;
      case OP_DIVIDE: stack[-1] /= stack[0]; break;
      case OP_POWER:
#ifdef CONFIG_USE_INTPOW
	if((double)(i = (long)floor(stack[0])) == stack[0])
	  /*  Use faster integer exponentiation  */
	  stack[-1] = intpow(stack[-1],i);
	else
#endif
	  stack[-1] = pow(stack[-1],stack[0]);
	break;
      case OP_NEGATE: stack++; stack[-1] = -stack[-1]; break;
      }
      break;
    case TYPE_CONST:
      *stack++ = expr->EX_Consts[*table & VALUE_MASK];
      break;
    case TYPE_VAR:
      *stack++ = vars[*table & VALUE_MASK];
      break;
    case TYPE_FCALL:
      if((i = InternalFunctions[*table & VALUE_MASK].IF_NVars) == 1) {
	stack[-1] = (*InternalFunctions[*table & VALUE_MASK].IF_Function)
	  (stack[-1]);
      } else {
	stack -= i - 1;

	stack[-1] = (*InternalFunctions[*table & VALUE_MASK].IF_FunctionN)
	  (&stack[-1],i);
      }
      break;
    case TYPE_UCALL:
      /*  Drop variables from stack  */
      i = fcts[*table & VALUE_MASK]->EX_NVars - 1;
      stack -= i;

      /*  Call recursively  */
      stack[-1] = EvalExpression(fcts[*table & VALUE_MASK],&stack[-1],
				 &stack[i],fcts);
      break;
    }
  }

  /*  Get value  */
  x = *--stack;

#ifndef CONFIG_USE_ALLOCA
  if(saveStack) free(saveStack);
#endif

  return(x);
}
