/*JS*********************************************************************
*
*    Program : CFFTN
*    Language: ANSI-C
*    Author  : Joerg Schoen
*    Purpose : Multi dimensional complex Fast Fourier Tranformation.
*
*      This material is taken from "Numerical Recipes in C" and slightly
*       altered (The original code was a little bit cryptic). The order
*       of storage of the multidimensional array was reversed.
*       The former name of the routine was fourn.
*
*      The forward transform is defined in the following way:
*        H(j_1,...,j_n) =
*          Sum_{k_n = 0}^{N_n - 1} ... Sum_{k_1 = 0}^{N_1 - 1}
*            exp(2 pi i k_n j_n / N_n) ... exp(2 pi i k_1 j_1 / N_1)
*              h(k_1,...,k_n)
*      and replaces h(k_1,...,k_n) by H(j_1,...,j_n).
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id: cfftn.c,v 1.4 1996/04/18 08:46:22 joerg Stab joerg $";
#endif

/*********     INCLUDES 					*********/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <jssubs.h>

/*********     DEFINES						*********/
/*  START-DEFINITIONS */
#include "jscmplx.h" /*  For "dcmplx" definition  */
/*  END-DEFINITIONS */

/*#define DIFFORDERING*/
/*#define USE_EXPITABLE*/

/* ERROR-DEFINITIONS from cfftn label _ERR_CFFTN ord 22
   n is not a power of two
*/

#define Prototype extern
/*********     PROTOTYPES					*********/
Prototype int            cfftn(dcmplx data[],long nn[],long lDim,int nDim,
			       int isign);
Prototype int            cfftnInit(long nn[],int nDim);

/*********     STATIC VARIABLES                                 *********/
#ifdef USE_EXPITABLE
static dcmplx *ExpITab;
static int ExpITabMax;
#endif

/*JS*********************************************************************
*   Replaces data by its nDim-dimensional discrete Fourier transform, if
*    isign is 1. nn[0..nDim-1] is an integer array containing the lengths
*    of each dimension (number of complex values), which MUST all be powers
*    of 2. lDim is the leading dimension of the first dimension of the array.
*    If a dimension is negative, the array is not transformed in respect
*    to that dimension.
*    Data is a complex array of length the product of these lengths, in
*    which the data are stored as in a multidimensional complex array:
*    the LEFTMOST index of the array increases most rapidly as one
*    proceeds along data. For a two-dimensional array, this is equivalent
*    to storing the array by columns.
*    If sign is input as -1, data is replaced by its inverse transform
*    times the product of the lengths of all dimensions.
*************************************************************************/

int cfftn(dcmplx data[],long nn[],long lDim,int nDim,int isign)

/************************************************************************/
{
  int idim;
  unsigned long nTot,nPrev,n;

#ifdef USE_EXPITABLE
  /*  Initialize table  */
  if(ExpITab == NULL && cfftnInit(nn,nDim)) return(2);
#endif

  /*  Test if all dimensions are a power of two  */
  for(idim = 0 ; idim < nDim ; idim++) {
    n = ABS(nn[idim]);
    if(n & (n - 1)) {
      JSErrNo = _ERR_CFFT + 0;
      return(-1);
    }
  }

  if(lDim == 0) lDim = ABS(nn[0]);

  /*  Compute total number of complex values  */
  for(nTot = lDim, idim = 1 ; idim < nDim ; idim++) nTot *= ABS(nn[idim]);

  /*  Main loop over the dimensions  */
  for(nPrev = 1, idim = 0 ; idim < nDim ;
      nPrev *= (idim == 0 ? lDim : n), idim++) {
    unsigned long i1,i2,i2a,i2End,i2Step,j2,i3,j3,mask;
    unsigned long mmax,istep;

    n = ABS(nn[idim]);

    /*  Negative value for dimension -> don't transform  */
    if(nn[idim] < 0) continue;

    i2End = n * nPrev;
    i2Step = (idim == 0 ? lDim * nPrev : i2End);

    /*  This is the bit reversal section of the routine  */
    for(j2 = i2 = 0 ; i2 < i2End ; i2 += nPrev) {
      if(i2 < j2) {
#ifdef DIFFORDERING
	/*  Use different ordering: fastest loop has stride 1  */
        for(i1 = i2 ; i1 < nTot ; i1 += i2Step) {
	  for(i3 = i1 ; i3 < i1 + nPrev ; i3++) {
#else
	/*  Original loop ordering  */
        for(i1 = i2 ; i1 < i2 + nPrev ; i1++) {
	  for(i3 = i1 ; i3 < nTot ; i3 += i2Step) {
#endif
	    dcmplx temp;

	    j3 = j2 + i3 - i2;
	    /*  Swap elements i3 and j3  */
	    temp = data[i3];  data[i3] = data[j3];  data[j3] = temp;
	  }
	}
      }

      for(mask = i2End >> 1 ; mask >= nPrev && j2 >= mask ; mask >>= 1)
	/*  Delete highest bit  */
	j2 -= mask;

      j2 += mask; /*  Set next bit  */
    }

    /* ***  Danielson-Lanczos section of the routine  *** */
    for(mmax = nPrev ; mmax < i2End ; mmax = istep) {
      dcmplx w;
#ifdef USE_EXPITABLE
      int tabCount,tabStep;
#else
      dcmplx wpm1;
      double theta;
#endif

      istep = mmax << 1;

#ifndef USE_EXPITABLE
      /*  Initialize the trigonometric recurrence  */
      {
	double temp;

	theta = isign > 0 ? ((nPrev * PI) / mmax) : (-(nPrev * PI) / mmax);
	temp = sin(0.5 * theta);
	/*   Set wpm1 to e^{theta}-1  */
	SETC(wpm1,-2.0 * temp * temp,sin(theta));
      }

      /*  Here are the two+one nested inner loops  */
# ifndef __cplusplus
      SETC(w,1.0,0.0);
# else
      w = 1.0;
# endif
#else
      /*  Initialize table step  */
      tabStep = (ExpITabMax / (mmax / nPrev));
      tabCount = 0;
#endif
      for(i3 = 0 ; i3 < mmax ; i3 += nPrev) {
#ifdef USE_EXPITABLE
	w = ExpITab[tabCount++ * tabStep];
	if(isign <= 0) IM(w) = - IM(w);
#endif

#ifdef DIFFORDERING
	/*  Use different ordering: fastest loop has stride 1  */
	for(i1 = i3 ; i1 < nTot ; i1 += i2Step) {
	  for(i2a = i1 ; i2a < i1 + i2End ; i2a += istep) {
	    for(i2 = i2a ; i2 < i2a + nPrev ; i2++) {
#else
	/*  Original ordering. One loop had to be  */
        /*   split due to new variable 'lDim'      */
	for(i1 = i3 ; i1 < i3 + nPrev ; i1++) {
	  for(i2a = i1 ; i2a < i2End ; i2a += istep) {
	    for(i2 = i2a ; i2 < nTot ; i2 += i2Step) {
#endif
	      dcmplx temp;
	      unsigned int k2;

	      k2 = i2 + mmax;

	      /*  This is the Danielson- Lanczos formula:  */
#ifndef __cplusplus
	      SETCOCO(temp,w,data[k2]);
	      RE(data[k2]) = RE(data[i2]) - RE(temp);
	      IM(data[k2]) = IM(data[i2]) - IM(temp);
	      RE(data[i2]) += RE(temp);
	      IM(data[i2]) += IM(temp);
#else
	      temp = w * data[k2];
	      data[k2] = data[i2] - temp;
	      data[i2] += temp;
#endif
	    }
	  }
	}

#ifndef USE_EXPITABLE
	/*  Trigonometric recurrence  */
# ifndef __cplusplus
	{
	  double dtemp = RE(w);

	  RE(w) += RE(wpm1) * dtemp - IM(wpm1) * IM(w);
	  IM(w) += RE(wpm1) * IM(w) + IM(wpm1) * dtemp;
	}
# else
	w += wpm1 * w;
# endif
#endif
      }
    }
  }

  return(0);
}

#ifdef USE_EXPITABLE
/*JS*********************************************************************
*   Initializes recurrence table.
*************************************************************************/

int cfftnInit(long nn[],int nDim)

/************************************************************************/
{
  int i;
  double dphi;
  long max;

  /*  Get maximum needed dimension  */
  for(max = i = 0 ; i < nDim ; i++)
    if(nn[i] > max) max = nn[i];

  max >>= 1;
  ExpITabMax = max;

  /*  Get memory and initialize table  */
  if((ExpITab = malloc(max * sizeof(*ExpITab))) == NULL) return(-1);

  /*  Initialize tables  */
  dphi = PI / max;

  for(i = 0 ; i < max ; i++) ExpITab[i] = cexpi(dphi * i);

  return(0);
}
#endif
