/*JS*********************************************************************
*
*    Program : CFFT
*    Language: ANSI-C
*    Author  : Joerg Schoen
*    Purpose : Complex Fast Fourier Tranformation.
*
*      This material is taken from "Numerical Recipes in C" and slightly
*       altered (The original code was a little bit cryptic). The former
*       name of the routine was four1.
*
*      The forward transform is defined in the following way:
*        H(j) = Sum_{k = 0}^{N - 1} exp(2 pi i k j / N) h(k)
*      and replaces h(k) by H(j).
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id: cfft.c,v 1.3 1996/04/18 08:45:47 joerg Stab joerg $";
#endif

/*********     INCLUDES 					*********/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <jssubs.h>

/*********     DEFINES						*********/
/*  START-DEFINITIONS */
#include "jscmplx.h" /*  For "dcmplx" definition  */
/*  END-DEFINITIONS */

/* ERROR-DEFINITIONS from cfft label _ERR_CFFT ord 21
   n is not a power of two
*/

#define Prototype extern
/*********     PROTOTYPES					*********/
Prototype int            cfft(dcmplx data[],unsigned long n,int isign);

/*JS*********************************************************************
*   Double precision complex Fast Fourier transform for n a power of 2.
*    Replaces data[0..n-1] by its discrete Fourier transform, if isign is
*    input as 1; or replaces data[0..n-1] by n times its inverse discrete
*    Fourier transform, if isign is input as -1. n MUST be an integer
*    power of 2 (this is checked for!). If the routine fails (n is not an
*    integer power of 2, it returns non zero.
*************************************************************************/

int cfft(dcmplx data[],unsigned long n,int isign)

/************************************************************************/
{
  unsigned int i,j,mmax,istep;

  /*  Test if n is power of two  */
  if(n & (n-1)) {
    JSErrNo = _ERR_CFFT + 0;
    return(-1);
  }

  /* ***  Bit reversal of array  *** */
  for(i = j = 0 ; i < n ; i++) {
    unsigned int mask;

    if(i < j) {
      dcmplx temp;

      /*  Swap elements i and j  */
      temp = data[i];  data[i] = data[j]; data[j] = temp;
    }

    for(mask = n >> 1 ; mask >= 1 && j >= mask ; mask >>= 1)
      /*  Delete highest Bit  */
      j -= mask;

    j += mask; /*  Set next bit  */
  }

  /* ***  Danielson- Lanczos section of the routine  *** */
  /*  Outer loop executed log_2 n times  */
  for(mmax = 1 ; mmax < n ; mmax = istep) {
    dcmplx w,wpm1;
    double theta;

    istep = mmax << 1;

    /*  Initialize the trigonometric recurrence  */
    {
      double temp;

      theta = isign > 0 ? (PI / mmax) : -(PI / mmax);
      temp = sin(0.5 * theta);
      /*   Set wpm1 to e^{i theta}-1  */
      SETC(wpm1,-2.0 * temp * temp,sin(theta));
    }

    /*  Here are the two nested inner loops  */
#ifndef __cplusplus
    SETC(w,1.0,0.0);
#else
    w = 1.0;
#endif
    for(j = 0 ; j < mmax ; j++) {
      for(i = j ; i < n ; i += istep) {
	dcmplx temp;
	unsigned int k;

	k = mmax + i;

	/*  This is the Danielson- Lanczos formula:  */
#ifndef __cplusplus
	SETCOCO(temp,w,data[k]);
	RE(data[k]) = RE(data[i]) - RE(temp);
	IM(data[k]) = IM(data[i]) - IM(temp);
	RE(data[i]) += RE(temp);
	IM(data[i]) += IM(temp);
#else
	temp = w * data[k];
	data[k] = data[i] - temp;
	data[i] += temp;
#endif
      }

      /*  Trigonometric recurrence  */
#ifndef __cplusplus
      {
	double dtemp = RE(w);

	RE(w) += RE(wpm1) * dtemp - IM(wpm1) * IM(w);
	IM(w) += RE(wpm1) * IM(w) + IM(wpm1) * dtemp;
      }
#else
      w += wpm1 * w;
#endif
    }
  }

  return(0);
}
