/*JS*********************************************************************
*
*    Program : CALCDERIV
*    Language: ANSI-C
*    Author  : Joerg Schoen
*    Purpose : Calculate the derivative of a whole function.
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id: calcderiv.c,v 1.1 1996/07/31 19:52:52 joerg Stab joerg $";
#endif

/*********     INCLUDES                                         *********/
#include <stdlib.h>
#include <math.h>

#include <jssubs.h>

/*********     DEFINES                                          *********/

#define Prototype extern
/*********     PROTOTYPES                                       *********/
Prototype int            calcDeriv(double *data,int n,double x0,double x1);
static double            evalFunc(double x);
Prototype int            calcDerivFFT(double *data,int n,double x0,double x1);

/*********     GLOBAL / STATIC VARIABLES                        *********/
static double Xa[2],*Save,*Y2;
static int NDim;

/*JS*********************************************************************
*   Calculate the derivative of a tabulated function at equally spaced
*    points using Ridders' method
*************************************************************************/

int calcDeriv(double *data,int n,double x0,double x1)

/************************************************************************/
{
  int i;
  double h;

  Xa[0] = x0;
  Xa[1] = x1;
  NDim = n;

  if((Y2 = spline(Xa,data,-n,HUGE_VAL,HUGE_VAL,NULL)) == NULL ||
     (Save = (double *)malloc(n * sizeof(*Save))) == NULL)
    return(-1);

  /*  Copy data  */
  for(i = 0 ; i < n ; i++) Save[i] = data[i];

  h = (x1 - x0) / (n - 1);
  for(i = 0 ; i < n ; i++)
    data[i] = ridDeriv(x0 + i * h,h,evalFunc,NULL);

  free(Save);
  free(Y2);

  return(0);
}

/*JS*********************************************************************
*
*************************************************************************/

static double evalFunc(double x)

/************************************************************************/
{
  return(splint(x,Xa,Save,Y2,-NDim,-1));
}

/*JS*********************************************************************
*   Calculate the derivative of a tabulated function at equally spaced
*    points using FFT.
*************************************************************************/

int calcDerivFFT(double *data,int n,double x0,double x1)

/************************************************************************/
{
  double temp,*data2;
  int i,dim;

  /*  Find first power of two that covers n  */
  for(dim = 1 ; dim < (1<<25) ; dim <<= 1)
    if(dim >= n) break;

  if(dim != n) {
    double a;

    if((data2 = splineData(NULL,data,n,dim + 1,NULL)) == NULL)
      goto error;

    /*  Make 2-pi-periodic data continous  */
    a = (data2[0] - data2[dim]) / dim; /*  data2[dim] + a * dim == data2[0]  */
    for(i = 0 ; i < dim ; i++) data2[i] += i * a;
  } else
    data2 = data;

  /*  Forward FFT  */
  rfft(data2,dim,+1);

  /*  Multiply to get derivative  */
  temp = (2.0 * PI) * (dim - 1) / (x1 - x0);

  /*  Include normalization factor and save division by dim in loop  */
  temp *= 2.0 / ((double)dim * (double)dim);

  data2[0] = 0.0;
  /*  Treat upper value as mixing of c_{-n/2} and c_{+n/2}
   *   so that derivative vanishes.
   */
  data2[1] = 0.0;
  for(i = 2 ; i < dim ; i += 2) {
    double p,re,im;

    p = (i / 2) * temp;
    re = data2[i    ] * p;
    im = data2[i + 1] * p;

    data2[i    ] =  im;
    data2[i + 1] = -re;
  }

  /*  Backward FFT  */
  rfft(data2,dim,-1);

  if(dim != n) {
    double *data3;

    if((data3 = splineData(NULL,data2,dim,n,NULL)) == NULL)
      goto error;
    free(data2);

    for(i = 0 ; i < n ; i++) data[i] = data3[i];

    free(data3);
  }

  return(0);
error:
  if(data2) free(data2);
  return(-1);
}
