/*JS*********************************************************************
*
*    Program : ANALYZE
*    Language: ANSI-C
*    Author  : Joerg Schoen
*    Purpose : Analyze data in various ways.
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id$";
#endif

/*********     INCLUDES                                         *********/
#include "analyze.h"

/*********     DEFINES                                          *********/
/*  START-DEFINITIONS */
/* ***  For routine transform11  *** */
#define NPICS11   1

#define MODE11_POLYNOMINT   MODE_USER

/* ***  For routine transform12  *** */
#define NPICS12   1

#define MODE12_BACKTRAFO    MODE_USER

/* ***  For routine transform13  *** */
#define NPICS13   2
/*  END-DEFINITIONS  */

/*********     PROTOTYPES                                       *********/
Prototype struct AnaMode Descript11;
static int               do11(int argc,char *argv[]);
static int               transform11(int ord,FPPicture *pics[],int nPics);

/* ------------------------------------------------------------------ */
Prototype struct AnaMode Descript12;
static int               do12(int argc,char *argv[]);
static int               transform12(int ord,FPPicture *pics[],int nPics);

/* ------------------------------------------------------------------ */
Prototype struct AnaMode Descript13;
static int               do13(int argc,char *argv[]);
static int               transform13(int ord,FPPicture *pics[],int nPics);

/* ------------------------------------------------------------------ */
struct AnaMode Descript11 = {
  "spLINE",
  "Spline data in first dimension to equidistant grid.",
  "\
Usage: analyze spline <newdim> <infile> [<outfile>]\n\
  Read data from file <infile> and interpolate it with respect to its first\n\
  dimension to an equidistant grid with <newdim> values using splines. Write\n\
  to file <outfile>. Only the first picture in a sequence of <npics> is\n\
  processed. Default for <npics> is %d.\n\
  Options:\n\
    -p    Use polynomial interpolation instead of splines.\n",
  NPICS11, do11
};

/* ***  For routine transform11  *** */
static int Dim11;

/*JS*********************************************************************
*   DO11
*************************************************************************/

static int do11(int argc,char *argv[])

/************************************************************************/
{
  int dim,readMode;
  char *outfile;

  if(argc == 2) outfile = OUTPUTFILE;
  else if(argc == 3) outfile = argv[2];
  else return(-2);

  dim = atoi(argv[0]);

  Dim11 = dim;

  readMode = FPLOT_MODE_READ|FPLOT_MODE_CASTTODOUBLE;
  if(Mode & MODE_NOEND) readMode |= FPLOT_TRANS_NOEND;
  if(Mode & MODE_FPAPPEND) readMode |= FPLOT_TRANS_APPEND;

  if(fplotTransform(argv[1],outfile,readMode,NPics,transform11)) {
    fprintf(stderr,"ERROR in routine fplotTransform!\n");
    return(-1);
  }

  return(0);
}

/*JS*********************************************************************
*   TRANSFORM11
*************************************************************************/

static int transform11(int ord,FPPicture *pics[],int nPics)

/************************************************************************/
{
  double *x,*data,*newData;
  int i,n1,n;

  if(MaxRuns >= 0 && ord >= MaxRuns) return(1);

  /*  Skip empty data  */
  if(pics[0]->FP_Content.FPD_Data == NULL) return(0);

  if(Verbose)
    printf("  Interpolating \"%s\".\n",pics[0]->FP_Title ?
	   pics[0]->FP_Title : "<empty>");

  /*  X-Values, data and first dimension. x may be NULL  */
  x = pics[0]->FP_Content.FPD_AxeValues ?
    (double *)pics[0]->FP_Content.FPD_AxeValues[0] : NULL;
  data = (double *)pics[0]->FP_Content.FPD_Data;
  n1 = pics[0]->FP_Content.FPD_Dims[0];

  /*  Get number of loop iterations  */
  for(n = i = 1 ; i < pics[0]->FP_Content.FPD_NDim ; i++)
    n *= pics[0]->FP_Content.FPD_Dims[i];

  if((newData = (double *)malloc(n * Dim11 * sizeof(*newData))) == NULL)
    goto error;

  /*  Do the transformation  */
  for(i = 0 ; i < n ; i++)
    if(Mode & MODE11_POLYNOMINT) {
      if(polIntData(x,&data[i * n1],n1,Dim11,&newData[i * Dim11]) == NULL)
	goto error;
    } else {
      if(splineData(x,&data[i * n1],n1,Dim11,&newData[i * Dim11]) == NULL)
	goto error;
    }

  /* ***  Free old content and set to new ones  *** */
  /*  Since new data is equidistant, no axe values are needed  */
  if(pics[0]->FP_Content.FPD_AxeValues) {
    pics[0]->FP_Content.FPD_AxeValues[0] = NULL;
    if(x) free(x);
  }

  free(pics[0]->FP_Content.FPD_Data);
  pics[0]->FP_Content.FPD_Data = newData;

  pics[0]->FP_Content.FPD_Dims[0] = Dim11;

  return(0);
error:
  return(-1);
}

/* ------------------------------------------------------------------ */
struct AnaMode Descript12 = {
  "fFT",
  "Do a Fourier transform of complex data.",
  "\
Usage: analyze fft <infile> [<outfile>]\n\
  Read data from file <infile> and apply a fourier transformation, write\n\
  results to <outfile>. Data must be complex (see 'fplot' how to mark data).\n\
  Only the first picture in a sequence of <npics> is processed. Default for\n\
  <npics> is %d.\n\
  To do more sophisticated FFT's for one-dimensional data, use program\n\
  \"dofft\".\n\
  Options:\n\
    -b   Do back-FFT.\n",
  NPICS12, do12
};

/*JS*********************************************************************
*   DO12
*************************************************************************/

static int do12(int argc,char *argv[])

/************************************************************************/
{
  int readMode;
  char *outfile;

  if(argc == 1) outfile = OUTPUTFILE;
  else if(argc == 2) outfile = argv[1];
  else return(-2);

  readMode = FPLOT_MODE_READ|FPLOT_MODE_CASTTODOUBLE;
  if(Mode & MODE_NOEND) readMode |= FPLOT_TRANS_NOEND;
  if(Mode & MODE_FPAPPEND) readMode |= FPLOT_TRANS_APPEND;

  if(fplotTransform(argv[0],outfile,readMode,NPics,transform12)) {
    fprintf(stderr,"ERROR in routine fplotTransform!\n");
    return(-1);
  }

  return(0);
}

/*JS*********************************************************************
*   TRANSFORM12
*************************************************************************/

static int transform12(int ord,FPPicture *pics[],int nPics)

/************************************************************************/
{
  long *dims;
  int i;

  if(MaxRuns >= 0 && ord >= MaxRuns) return(1);

  /*  Skip empty data  */
  if(pics[0]->FP_Content.FPD_Data == NULL) return(0);

  if(Verbose)
    printf("  FFT for \"%s\".\n",pics[0]->FP_Title ?
	   pics[0]->FP_Title : "<empty>");

  /*  Check for complex data  */
  if(!(pics[0]->FP_Content.FPD_Type & FPLOT_TYPE_COMPLEXDATA)) {
    fprintf(stderr,"ERROR: Data is not complex!\n");
    goto error;
  }

  if((dims = (long *)malloc(pics[0]->FP_Content.FPD_NDim * sizeof(dims)))
     == NULL) goto error;

  /*  Lowest dimension must be halfed to account for complex data  */
  dims[0] = pics[0]->FP_Content.FPD_Dims[i] / 2;
  for(i = 1 ; i < pics[0]->FP_Content.FPD_NDim ; i++)
    dims[i] = pics[0]->FP_Content.FPD_Dims[i];

  if(lcfftn((dcmplx *)pics[0]->FP_Content.FPD_Data,dims,0,
	    pics[0]->FP_Content.FPD_NDim,(Mode & MODE12_BACKTRAFO) ? -1 : +1))
    goto error;

  free(dims);

  return(0);
error:
  return(-1);
}

/* ------------------------------------------------------------------ */
struct AnaMode Descript13 = {
  "cONCAT",
  "Concatenate one-dimensional data together.",
  "\
Usage: analyze concatenate <infile> [<outfile>]\n\
  Read <npics> pictures from file <infile> and concatenate the values\n\
  together. The data must be one-dimensional. Write the results to <outfile>.\n\
  Default for <npics> is %d.\n",
  NPICS13, do13
};

/*JS*********************************************************************
*   DO13
*************************************************************************/

static int do13(int argc,char *argv[])

/************************************************************************/
{
  int readMode;
  char *outfile;

  if(argc == 1) outfile = OUTPUTFILE;
  else if(argc == 2) outfile = argv[1];
  else return(-2);

  readMode = FPLOT_MODE_READ;
  if(Mode & MODE_NOEND) readMode |= FPLOT_TRANS_NOEND;
  if(Mode & MODE_FPAPPEND) readMode |= FPLOT_TRANS_APPEND;

  if(fplotTransform(argv[0],outfile,readMode,NPics,transform13)) {
    fprintf(stderr,"ERROR in routine fplotTransform!\n");
    return(-1);
  }

  return(0);
}

/*JS*********************************************************************
*   TRANSFORM13
*************************************************************************/

static int transform13(int ord,FPPicture *pics[],int nPics)

/************************************************************************/
{
  long dim,offset;
  void *data,*aData;
  int size,i,axeFlag;

  if(MaxRuns >= 0 && ord >= MaxRuns) return(1);

  /*  Check and get whole dimension  */
  dim = 0;
  size = 0;
  axeFlag = -1;
  for(i = 0 ; i < nPics ; i++) {
    int size2,axeFlag2;

    /*  Skip empty data  */
    if(pics[i]->FP_Content.FPD_Data == NULL) continue;

    if(pics[i]->FP_Content.FPD_NDim != 1) goto formatError;

    /*  Determine size and check for consistency  */
    size2 = (pics[i]->FP_Content.FPD_Flags & FPLOT_FLAGS_SINGLE) ?
      sizeof(float) : sizeof(double);

    if(size == 0) size = size2;
    else if(size2 != size) goto formatError;

    /*  Look for axe data and check for consistency  */
    axeFlag2 = (pics[i]->FP_Content.FPD_AxeValues &&
		pics[i]->FP_Content.FPD_AxeValues[0]) ? 1 : 0;

    if(axeFlag < 0) axeFlag = axeFlag2;
    else if(axeFlag2 != axeFlag) goto formatError;

    dim += pics[i]->FP_Content.FPD_Dims[0];
  }

  if(dim == 0) goto formatError;

  if(Verbose)
    printf("  \"%s\": Concatenating %d pictures.\n",pics[0]->FP_Title ?
	   pics[0]->FP_Title : "<empty>",nPics);

  if((data = malloc(dim * size)) == NULL)
    goto error;
  if(!axeFlag)
    aData = NULL;
  else if((aData = malloc(dim * size)) == NULL)
    goto error;

  offset = 0;
  for(i = 0 ; i < nPics ; i++) {
    long len;

    /*  Skip empty data  */
    if(pics[i]->FP_Content.FPD_Data == NULL) continue;

    len = pics[i]->FP_Content.FPD_Dims[0] * size;
    memcpy((char *)data + offset,pics[i]->FP_Content.FPD_Data,len);

    if(aData)
      memcpy((char *)aData + offset,pics[i]->FP_Content.FPD_AxeValues[0],len);

    offset += len;
  }

  /*  Free all other pictures  */
  for(i = 1 ; i < nPics ; i++) {
    fplotFreepic(pics[i]);
    pics[i] = NULL;
  }

  /*  Set up new data  */
  if(changeTitle(pics[0],"Concatenation of \"%s\" ...")) goto error;
  pics[0]->FP_Content.FPD_Dims[0] = dim;
  free(pics[0]->FP_Content.FPD_Data);
  pics[0]->FP_Content.FPD_Data = data;
  if(aData) {
    free(pics[0]->FP_Content.FPD_AxeValues[0]);
    pics[0]->FP_Content.FPD_AxeValues[0] = aData;
  }

  return(0);
formatError:
  fprintf(stderr,"ERROR: Format error on input picture %d -- Break.\n",
	  ord * nPics + i);
error:
  return(-1);
}
