/*JS*********************************************************************
*
*    Program : ANALYZE
*    Language: ANSI-C
*    Author  : Joerg Schoen
*    Purpose : Analyze data in various ways.
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id$";
#endif

/*********     INCLUDES                                         *********/
#include "analyze.h"

#include <ctype.h>

/*********     DEFINES                                          *********/
/*  START-DEFINITIONS */
/* ***  For routine transform8  *** */
#define NPICS8   1

#define MODE8_FFTMETHOD     MODE_USER

/* ***  For routine transform10  *** */
#define NPICS10   1

#define MODE10_GENAXEDATA   MODE_USER
/*  END-DEFINITIONS  */

/*********     PROTOTYPES                                       *********/
Prototype struct AnaMode Descript8;
static int               do8(int argc,char *argv[]);
static int               transform8(int ord,FPPicture *pics[],int nPics);

/* ------------------------------------------------------------------ */
Prototype struct AnaMode Descript9;
static int               do9(int argc,char *argv[]);

/* ------------------------------------------------------------------ */
Prototype struct AnaMode Descript10;
static int               do10(int argc,char *argv[]);
static int               transform10(int ord,FPPicture *pics[],int nPics);

/* ------------------------------------------------------------------ */
struct AnaMode Descript8 = {
  "dERIVATIVE",
  "Differentiate one dimensional data.",
  "\
Usage: analyze derivative <infile> [<outfile>]\n\
  Differentiate the data from file <infile> employing Ridder's method and\n\
  write the derivative to file <outfile>. Only the first picture in a\n\
  sequence of <npics> is processed. Default for <npics> is %d.\n\
  Options:\n\
    -f Use FFT method instead.\n",
  NPICS8, do8
};

/*JS*********************************************************************
*   DO8
*************************************************************************/

static int do8(int argc,char *argv[])

/************************************************************************/
{
  int readMode;
  char *outfile;

  if(argc == 1) outfile = OUTPUTFILE;
  else if(argc == 2) outfile = argv[1];
  else return(-2);

  readMode = FPLOT_MODE_READ|FPLOT_MODE_CASTTODOUBLE|FPLOT_MODE_NOGETMINMAX;
  if(Mode & MODE_NOEND) readMode |= FPLOT_TRANS_NOEND;
  if(Mode & MODE_FPAPPEND) readMode |= FPLOT_TRANS_APPEND;

  if(fplotTransform(argv[0],outfile,readMode,NPics,transform8)) {
    fprintf(stderr,"ERROR in routine fplotTransform!\n");
    return(-1);
  }

  return(0);
}

/*JS*********************************************************************
*   TRANSFORM8
*************************************************************************/

static int transform8(int ord,FPPicture *pics[],int nPics)

/************************************************************************/
{
  int (*fftRoutine)(double *,int,double,double);

  if((MaxRuns >= 0 && ord >= MaxRuns) || pics[0]->FP_Content.FPD_NDim != 1)
    return(1);

  if(Verbose) printf("  Differentiating \"%s\".\n",pics[0]->FP_Title ?
		     pics[0]->FP_Title : "<empty>");

  fftRoutine = (Mode & MODE8_FFTMETHOD) ? calcDerivFFT : calcDeriv;

  if((*fftRoutine)((double *) pics[0]->FP_Content.FPD_Data,
		   pics[0]->FP_Content.FPD_Dims[0],
		   ((double*)pics[0]->FP_Content.FPD_Boundaries)[0],
		   ((double*)pics[0]->FP_Content.FPD_Boundaries)[1]))
    goto error;

  if(changeTitle(pics[0],"Derivative of \"%s\"")) goto error;

  return(0);
error:
  return(-1);
}

/* ------------------------------------------------------------------ */
struct AnaMode Descript9 = {
  "swAPDIMENSION",
  "Swap multidimensional data in two dimensions.",
  "\
Usage: analyze swapdimension <infile> <outfile> [<dim1> [<dim2>]]\n\
  Read data from file <infile> and write to file <outfile>, swaping dimension\n\
  <dim1> and <dim2>, which default to 1 and 2, resp.\n",
  -1, do9
};

/*JS*********************************************************************
*   DO9
*************************************************************************/

static int do9(int argc,char *argv[])

/************************************************************************/
{
  FPFile *fpIn,*fpOut;
  FPPicture *pic;
  int dim1,dim2,ord;

  if(argc < 2 || argc > 4) return(-2);

  dim1 = (argc > 2) ? (atoi(argv[2]) - 1) : 0;
  dim2 = (argc > 3) ? (atoi(argv[3]) - 1) : 1;

  fpOut = NULL;

  /*  Open files and get memory  */
  if((fpIn = fplotOpen(argv[0],FPLOT_READ)) == NULL) goto error;
  if((fpOut = fplotOpen(argv[1],(Mode & MODE_FPAPPEND) ? FPLOT_APPEND :
			FPLOT_WRITE)) == NULL) goto error;

  for(ord = 1 ; !feof(fpIn->FF_File) ; ord++) {
    int lDim1,lDim2;
    FPData *picData;

    /*  Read next picture  */
    if((pic = fplotRead(fpIn,FPLOT_MODE_READ|FPLOT_MODE_NOGETMINMAX)) == NULL) {
      if(feof(fpIn->FF_File)) break;
      goto error;
    }

    if(Verbose)
      printf("%2d: %s\n",ord,pic->FP_Title ? pic->FP_Title : "<empty>");

    /*  Check dimensions to swap  */
    lDim1 = dim1;
    lDim2 = dim2;
    if(pic->FP_Axes && lDim1 >= 0 && lDim1 < pic->FP_NAxes &&
       lDim2 >= 0 && lDim2 < pic->FP_NAxes) { /*  Swap axe title  */
      char *sTmp;

      sTmp = pic->FP_Axes[lDim1];
      pic->FP_Axes[lDim1] = pic->FP_Axes[lDim2];
      pic->FP_Axes[lDim2] = sTmp;
    }

    /* ***  Write to file  *** */

    /*  Title...  */
    if(fplotStart((Mode & MODE_NOEND) ? NULL : fpOut) ||
       (pic->FP_Title && fplotText(fpOut,pic->FP_Title)))
      goto error;

    /*  data...  */
    for(picData = &pic->FP_Content ; picData ; picData = picData->FPD_Next) {
      long *strides;
      int i;

      if(picData->FPD_Data == NULL) continue;

      /*  Swap dimensions  */
      if(lDim1 >= 0 && lDim1 < picData->FPD_NDim &&
	 lDim2 >= 0 && lDim2 < picData->FPD_NDim) {
	if(Verbose)
	  printf("  Swapped dimension %d and %d\n",1 + lDim1,1 + lDim2);

	/*  Create fake stride vector   */
	if((strides = (long *)malloc(picData->FPD_NDim * sizeof(*strides)))
	   == NULL) goto error;
	strides[0] = 1;
	for(i = 1 ; i < picData->FPD_NDim ; i++)
	  strides[i] = strides[i - 1] * picData->FPD_Dims[i - 1];

	/*  Swap strides, dimension, axe data and boundaries  */
	{
	  long lTmp = strides[lDim1];
	  strides[lDim1] = strides[lDim2];
	  strides[lDim2] = lTmp;
	}
	{
	  unsigned long uTmp = picData->FPD_Dims[lDim1];
	  picData->FPD_Dims[lDim1] = picData->FPD_Dims[lDim2];
	  picData->FPD_Dims[lDim2] = uTmp;
	}
	if(picData->FPD_AxeValues) {
	  void *pTmp = picData->FPD_AxeValues[lDim1];
	  picData->FPD_AxeValues[lDim1] = picData->FPD_AxeValues[lDim2];
	  picData->FPD_AxeValues[lDim2] = pTmp;
	}
	if(picData->FPD_Flags & FPLOT_FLAGS_SINGLE) {
	  float dTmp;

	  dTmp = ((float *)picData->FPD_Boundaries)[2 * lDim1];
	  ((float *)picData->FPD_Boundaries)[2 * lDim1] =
	    ((float *)picData->FPD_Boundaries)[2 * lDim2];
	  ((float *)picData->FPD_Boundaries)[2 * lDim2] = dTmp;

	  dTmp = ((float *)picData->FPD_Boundaries)[2 * lDim1 + 1];
	  ((float *)picData->FPD_Boundaries)[2 * lDim1 + 1] =
	    ((float *)picData->FPD_Boundaries)[2 * lDim2 + 1];
	  ((float *)picData->FPD_Boundaries)[2 * lDim2 + 1] = dTmp;
	} else {
	  double dTmp;

	  dTmp = ((double *)picData->FPD_Boundaries)[2 * lDim1];
	  ((double *)picData->FPD_Boundaries)[2 * lDim1] =
	    ((double *)picData->FPD_Boundaries)[2 * lDim2];
	  ((double *)picData->FPD_Boundaries)[2 * lDim2] = dTmp;

	  dTmp = ((double *)picData->FPD_Boundaries)[2 * lDim1 + 1];
	  ((double *)picData->FPD_Boundaries)[2 * lDim1 + 1] =
	    ((double *)picData->FPD_Boundaries)[2 * lDim2 + 1];
	  ((double *)picData->FPD_Boundaries)[2 * lDim2 + 1] = dTmp;
	}
      } else
	strides = NULL;

      /*  Now write data  */
      if(picData->FPD_Flags & FPLOT_FLAGS_SINGLE) {
	if(fplotSTensor(fpOut,picData->FPD_Type,picData->FPD_NDim,
			(const float **)picData->FPD_AxeValues,
			(const float *)picData->FPD_Data,
			picData->FPD_Dims,strides,
			(const float *)picData->FPD_Boundaries,
			picData->FPD_FMin,picData->FPD_FMax))
	  goto error;
      } else {
	if(fplotTensor(fpOut,picData->FPD_Type,picData->FPD_NDim,
		       (const double **)picData->FPD_AxeValues,
		       (const double *)picData->FPD_Data,
		       picData->FPD_Dims,strides,
		       (const double *)picData->FPD_Boundaries,
		       picData->FPD_FMin,picData->FPD_FMax))
	  goto error;
      }

      if(strides) free(strides);
    }

    if((pic->FP_Axes && fplotAxes(fpOut,pic->FP_NAxes,
				  (const char **)pic->FP_Axes)) ||
       fplotEnd((Mode & MODE_NOEND) ? NULL : fpOut)) goto error;

    /*	Free memory  */
    fplotFreepic(pic);
  }

  fplotClose(fpIn);
  fplotClose(fpOut);

  return(0);
error:
  if(fpOut) fplotClose(fpOut);
  if(fpIn) fplotClose(fpIn);
  return(-1);
}

/* ------------------------------------------------------------------ */
struct AnaMode Descript10 = {
  "tRANSFORM",
  "Transform data with arbitrary expression.",
  "\
Usage: analyze transform <rank> <expr> <infile> [<outfile>]\n\
  <npics> data blocks are read in a sequence. If their rank is equal to\n\
  <rank> and they contain solely one data block, the data is transformed,\n\
  treating <expr> as the transformation expression, which consists of ';'\n\
  separated expressions for the axe values and the data itself. Expressions\n\
  of the form \"x1=...\" are treated as axe values transformations. The axe\n\
  values are denoted by \'x1\' .. and the data values of the <npics> pictures\n\
  by \'f1\' ... Only one output data block is written. Data is read from file\n\
  <infile> and written to <outfile>. If the input blocks do not meet the\n\
  above mentioned criteria, the data is discarded completely and no output is\n\
  produced for these blocks. The processing continues.\n\
  Options:\n\
    -g   Generate axe data. If axe transformations are not linear, useful to\n\
         assure that final data gets not misinterpreted as equidistant.\n",
  NPICS10, do10
};

/* ***  For routine transform10  *** */
static Expression **Expr10;
static int Rank10;

/*JS*********************************************************************
*   DO10
*************************************************************************/

static int do10(int argc,char *argv[])

/************************************************************************/
{
  int rank,i;
  const char *string;
  char *eString,*outfile;
  char **vars,*ptr;

  if(argc == 3) outfile = OUTPUTFILE;
  else if(argc == 4) outfile = argv[3];
  else return(-2);

  rank = atoi(argv[0]);
  eString = argv[1];

  if((Expr10 = (Expression **)malloc((rank + 1) * sizeof(*Expr10))) == NULL ||
     (vars = (char **)malloc((rank + NPics + 1) * sizeof(*vars) +
			     (3 + (int)log10((double)rank)) * rank +
			     (3 + (int)log10((double)NPics)) * NPics)) == NULL)
    goto error;

  /*  Preset variables  */
  Rank10 = rank;
  ptr = (char *)&vars[rank + NPics + 1];
  /*  First generate axe names, then data value names  */
  for(i = 0 ; i < rank ; i++) {
    vars[i] = ptr;
    ptr += 1 + sprintf(ptr,"x%d",1 + i);
  }
  for(i = 0 ; i < NPics ; i++) {
    vars[rank + i] = ptr;
    ptr += 1 + sprintf(ptr,"f%d",1 + i);
  }
  vars[rank + NPics] = NULL;

  /*  Preset default expressions  */
  for(i = 0 ; i <= rank ; i++) Expr10[i] = NULL;

  /*  Scan for expressions separated with ';'  */
  while(*eString) {
    while(*eString && isspace(*(unsigned char *)eString)) eString++;

    for(i = 0 ; i < rank ; i++) {
      /*  Recognize variable name  */
      if((string = strcompare(eString,vars[i]))) {
	/*  Check for '='  */
	while(*string && isspace(*(unsigned char *)string)) string++;
	if(*string == '=') break;
      }
    }

    if(i < rank) string++;
    else string = eString;

    if(Expr10[i] != NULL) {
      fprintf(stderr,"\
ERROR: Transformation for %s%s given twice -- Break.\n",
	      i < rank ? "axe " : "data",i < rank ? vars[i] : "");
      goto error;
    }

    /*  Compile next expression  */
    if((Expr10[i] = CompExpression(string,(const char **)vars,NULL,
				   (const char **)&eString))
       == NULL ||
       eString == string)
      goto error;

    while(*eString && isspace(*(unsigned char *)eString)) eString++;
    if(*eString == ';') eString++;
  }

  free(vars);

  i = FPLOT_MODE_READ|FPLOT_MODE_CASTTODOUBLE|FPLOT_MODE_NOGETMINMAX;
  if(Mode & MODE_NOEND) i |= FPLOT_TRANS_NOEND;
  if(Mode & MODE_FPAPPEND) i |= FPLOT_TRANS_APPEND;
  if(Mode & MODE10_GENAXEDATA) i |= FPLOT_MODE_GENAXEDATA;

  if(fplotTransform(argv[2],outfile,i,NPics,transform10)) {
    fprintf(stderr,"ERROR in routine fplotTransform!\n");
    goto error;
  }

  for(i = 0 ; i <= rank ; i++)
    if(Expr10[i]) FreeExpression(Expr10[i]);

  free(Expr10);

  return(0);
error:
  return(-1);
}

/*JS*********************************************************************
*   TRANSFORM10
*************************************************************************/

static int transform10(int ord,FPPicture *pics[],int nPics)

/************************************************************************/
{
  int i,rank;
  double *vals;

  rank = Rank10;

  if(MaxRuns >= 0 && ord >= MaxRuns) return(1);

  for(i = 0 ; i < nPics ; i++) {
    int j;

    /*  Skip empty data  */
    if(pics[i]->FP_Content.FPD_Data == NULL) continue;

    /*  Check dimension and that we have single data  */
    if(pics[i]->FP_Content.FPD_NDim != rank || pics[i]->FP_Content.FPD_Next)
      break;

    /*  Check if dimensions and boundaries match  */
    if(i) {
      for(j = 0 ; j < rank ; j++) {
	if(pics[i]->FP_Content.FPD_Dims[j] !=
	   pics[0]->FP_Content.FPD_Dims[j] ||
	   ((double*)(pics[i]->FP_Content.FPD_Boundaries))[2 * j] !=
	   ((double*)(pics[0]->FP_Content.FPD_Boundaries))[2 * j] ||
	   ((double*)(pics[i]->FP_Content.FPD_Boundaries))[2 * j + 1] !=
	   ((double*)(pics[0]->FP_Content.FPD_Boundaries))[2 * j + 1])
	  break;
      }
      if(j < rank) break;
    }
  }

  if(i < nPics) {
    /*  Discard this data completely  */
    for(i = 0 ; i < nPics ; i++) {
      fplotFreepic(pics[i]);
      pics[i] = NULL;
    }
    return(0);
  }

  if(Verbose)
    printf("%2d: %s\n",ord,pics[0]->FP_Title ? pics[0]->FP_Title : "<empty>");

  if((vals = (double *)malloc((rank + nPics) * sizeof(*vals))) == NULL)
    goto error;

  /*  Preset for axe transformation  */
  for(i = 0 ; i < (rank + NPics) ; i++)
    /*  Prevents usage of illegal values  */
    vals[i] = HUGE_VAL;

  /* ***  Any axe transformation required?  */
  for(i = 0 ; i < rank ; i++)
    if(Expr10[i]) {
      int j;

      for(j = 0 ; j < nPics ; j++) {
	if(pics[j]->FP_Content.FPD_AxeValues &&
	   pics[j]->FP_Content.FPD_AxeValues[i]) {
	  int n,k;

	  n = pics[j]->FP_Content.FPD_Dims[i];
	  for(k = 0 ; k < n ; k++) {
	    vals[i] = ((double *)(pics[0]->FP_Content.FPD_AxeValues[i]))[k];

	    ((double *)(pics[0]->FP_Content.FPD_AxeValues[i]))[k] =
	      EvalExpression(Expr10[i],vals,NULL,NULL);
	  }
	}

	if(((double *)(pics[j]->FP_Content.FPD_Boundaries))[2*i] !=
	   ((double *)(pics[j]->FP_Content.FPD_Boundaries))[2*i+1]) {
	  vals[i] = ((double *)(pics[j]->FP_Content.FPD_Boundaries))[2*i];
	  ((double *)(pics[j]->FP_Content.FPD_Boundaries))[2*i] =
	    EvalExpression(Expr10[i],vals,NULL,NULL);

	  vals[i] = ((double *)(pics[j]->FP_Content.FPD_Boundaries))[2*i+1];
	  ((double *)(pics[j]->FP_Content.FPD_Boundaries))[2*i+1] =
	    EvalExpression(Expr10[i],vals,NULL,NULL);
	}
      }
      vals[i] = HUGE_VAL;
    }

  /* ***  Data transformation required?  *** */
  if(Expr10[rank] && pics[0]->FP_Content.FPD_Data) {
    double **data;
    int *index;
    int arrayIndex;

    if((data = (double **)malloc(nPics * sizeof(*data))) == NULL) goto error;

    /*  Get all data pointers  */
    for(i = 0 ; i < nPics ; i++)
      data[i] = (double *)pics[i]->FP_Content.FPD_Data;

    if((index = (int *)malloc(rank * sizeof(*index))) == NULL)
	goto error;

    /*  Preset variables  */
    for(i = 0 ; i < rank ; i++) {
      index[i] = 0;
      if(pics[0]->FP_Content.FPD_AxeValues &&
	 pics[0]->FP_Content.FPD_AxeValues[i])
	vals[i] = ((double *)(pics[0]->FP_Content.FPD_AxeValues[i]))[0];
      else
	vals[i] = ((double *)(pics[0]->FP_Content.FPD_Boundaries))[2 * i];
    }

    arrayIndex = 0;
    do {
      /*  Generate values and evaluate  */
      for(i = 0 ; i < nPics ; i++)
	vals[rank + i] = data[i] ? data[i][arrayIndex] : 0.0;

#ifdef DEBUG
      printf("AXEVALS: ");
      for(i = 0 ; i < rank ; i++) printf("%g ",vals[i]);
      printf("FUNCVALS: ");
      for(i = 0 ; i < nPics ; i++) printf("%g ",vals[rank + i]);
      printf("\n");
#endif

      /*  Evaluate  */
      data[0][arrayIndex] = EvalExpression(Expr10[rank],vals,NULL,NULL);

      arrayIndex++;

      /*  Increment  */
      for(i = 0 ; i < rank ; i++) {
	index[i]++;

	if(index[i] < pics[0]->FP_Content.FPD_Dims[i]) {
	  if(pics[0]->FP_Content.FPD_AxeValues &&
	     pics[0]->FP_Content.FPD_AxeValues[i])
	    vals[i] = ((double *)(pics[0]->FP_Content.FPD_AxeValues[i]))
	      [index[i]];
	  else
	    vals[i] = ((double *)pics[0]->FP_Content.FPD_Boundaries)[2*i] +
	      (index[i] * (((double *)pics[0]->FP_Content.FPD_Boundaries)
			   [2*i+1] -
			   ((double *)pics[0]->FP_Content.FPD_Boundaries)
			   [2*i])) /
	      (pics[0]->FP_Content.FPD_Dims[i] - 1);
	  break;
	}

        index[i] = 0;
	if(pics[0]->FP_Content.FPD_AxeValues &&
	   pics[0]->FP_Content.FPD_AxeValues[i])
	  vals[i] = ((double *)(pics[0]->FP_Content.FPD_AxeValues[i]))[0];
	else
	  vals[i] = ((double *)pics[0]->FP_Content.FPD_Boundaries)[2*i];
      }
    } while(i < rank);

    free(index);
    free(data);
  }

  free(vals);

  /*  Discard all data but first one  */
  for(i = 1 ; i < nPics ; i++) {
    fplotFreepic(pics[i]);
    pics[i] = NULL;
  }

  return(0);
error:
  return(-1);
}
