/*JS*********************************************************************
*
*    Program : ANALYZE
*    Language: ANSI-C
*    Author  : Joerg Schoen
*    Purpose : Analyze data in various ways.
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id$";
#endif

/*********     INCLUDES                                         *********/
#include "analyze.h"

/*********     DEFINES                                          *********/
/*  START-DEFINITIONS */
/* ***  For routine analyze4   *** */
#define NPICS4   2

/* ***  For routine analyze5   *** */
#define NPICS5   1
#define EPSILON  (10.0 * DBL_EPSILON)

/* ***  For routine analyze6   *** */
#define NPICS6   1

/* ***  For routine analyze7   *** */
#define NPICS7   1

#define MODE7_VOLNELEM      MODE_USER
/*  END-DEFINITIONS  */

/*********     PROTOTYPES                                       *********/
Prototype struct AnaMode Descript4;
static int               do4(int argc,char *argv[]);
static int               analyze4(int ord,FPPicture *pics[],int nFiles,
				  int nPics);
/* ------------------------------------------------------------------ */
Prototype struct AnaMode Descript5;
static int               do5(int argc,char *argv[]);
static int               analyze5(int ord,FPPicture *pics[],int nFiles,
				  int nPics);

Prototype double         Epsilon5;

/* ------------------------------------------------------------------ */
Prototype struct AnaMode Descript6;
static int               do6(int argc,char *argv[]);
static int               analyze6(int ord,FPPicture *pics[],int nFiles,
				  int nPics);

/* ------------------------------------------------------------------ */
Prototype struct AnaMode Descript7;
static int               do7(int argc,char *argv[]);
static int               analyze7(int ord,FPPicture *pics[],int nFiles,
				  int nPics);

/* ------------------------------------------------------------------ */
struct AnaMode Descript4 = {
  "eXPECTRHO",
  "Expectation value of rho of a sequence of wave functions.",
  "\
Usage: analyze expectrho <infile> [<outfile>]\n\
  Calculate the expectation value of rho of the first two dimensional\n\
  pictures. Read from <infile> and if <outfile> is given, write to it.\n\
  Otherwise print the results to stdout. Default for <npics> is %d.\n",
  NPICS4, do4
};

/* ***  For routine analyze4  *** */
static double **Datas4;
static double *XData4;
static int Index4,MaxData4;

/*JS*********************************************************************
*   DO4
*************************************************************************/

static int do4(int argc,char *argv[])

/************************************************************************/
{
  char *outfile;
  FPFile *fp;
  int *readModes,i;

  if(argc == 1) outfile = NULL;
  else if(argc == 2) outfile = argv[1];
  else return(-2);

  /*  Get memory  */
  if(outfile) {
    if((Datas4 = (double **)malloc(NPics * sizeof(*Datas4))) == NULL)
      goto error;

    for(i = 0 ; i < NPics ; i++) Datas4[i] = NULL;
  }

  if((readModes = (int *)malloc(NPics * sizeof(*readModes))) == NULL)
    goto error;

  /*  How to read pictures  */
  for(i = 0 ; i < NPics ; i++)
    readModes[i] = FPLOT_MODE_READ|FPLOT_MODE_CASTTODOUBLE;

  if(fplotAnalyze(argv,1,readModes,NPics,analyze4)) {
    fprintf(stderr,"ERROR in routine fplotAnalyze!\n");
    goto error;
  }
  free(readModes);

  if(outfile) {
    if((fp = fplotOpen(outfile,(Mode & MODE_FPAPPEND) ? FPLOT_APPEND :
		       FPLOT_WRITE)) == NULL) goto plError;
    for(i = 0 ; i < NPics ; i++)
      if(Datas4[i] &&
	 (fplotStart((Mode & MODE_NOEND) ? NULL : fp) ||
	  fplotText(fp,"<rho> of \"%s\".%d ",argv[0],1 + i) ||
	  fplotVector(fp,0,XData4,Datas4[i],Index4,1,XData4[0],
		      XData4[Index4 - 1],1.0,1.0) ||
	  fplotEnd((Mode & MODE_NOEND) ? NULL : fp)))
	goto plError;

    if(fplotClose(fp)) goto plError;
  }

  if(Datas4) {
    for(i = 0 ; i < NPics ; i++) free(Datas4[i]);
    free(Datas4);
    free(XData4);
  }

  return(0);
plError:
  fprintf(stderr,"ERROR in plotting -- Break.\n");
error:
  return(-1);
}

/*JS*********************************************************************
*   ANALYZE4
*************************************************************************/

static int analyze4(int ord,FPPicture *pics[],int nFiles,int nPics)

/************************************************************************/
{
  double x;
  char *string;
  int i;

  if(MaxRuns >= 0 && ord >= MaxRuns) return(1);

  /*  Check format and break if different data reached  */
  for(i = 0 ; i < nPics ; i++)
    if(pics[i]->FP_Content.FPD_NDim != 2) return(1);

  if(Datas4) {
    /*  Increase memory  */
    if(Index4 >= MaxData4) {
      if((XData4 = (double *)realloc(XData4,(MaxData4 + 10) * sizeof(*XData4)))
	 == NULL) goto error;
      for(i = 0 ; i < nPics ; i++)
	if((Datas4[i] = (double *)realloc(Datas4[i],(MaxData4 + 10) *
					  sizeof(*Datas4[i]))) == NULL)
	  goto error;
      MaxData4 += 10;
    }
  }

  if(Verbose)
    printf(" %d: \"%s\"\n",1 + ord,pics[0]->FP_Title ? pics[0]->FP_Title :
	   "<empty>");

  /*  Extract time from title  */
  if(pics[0]->FP_Title == NULL ||
     (string = strstr(pics[0]->FP_Title,"t =")) == NULL ||
     sscanf(string,"t = %lg",&x) != 1) {
    fprintf(stderr,"ERROR: Cannot extract time from \"%s\", nr %d\n",
	    pics[0]->FP_Title ? pics[0]->FP_Title : "",ord);
    x = (double) Index4;
  }

  if(XData4) {
    XData4[Index4] = x;

    /*  Check if this time has appeared before  */
    for(i = Index4 ; i > 0 && XData4[i - 1] > XData4[Index4] ; ) i--;

    if(i > 0 && XData4[i - 1] == XData4[Index4]) {
      fprintf(stderr,"WARNING: Skipping second appearance of time %g at %d.\n",
	      XData4[Index4],ord);
      goto ende;
    }
  }

  for(i = 0 ; i < nPics ; i++) {
    x = rhoExpect((double*)pics[i]->FP_Content.FPD_Data,
		pics[i]->FP_Content.FPD_Dims[0],
		pics[i]->FP_Content.FPD_Dims[1],
		((double*)pics[i]->FP_Content.FPD_Boundaries)[0],
		((double*)pics[i]->FP_Content.FPD_Boundaries)[1],
		((double*)pics[i]->FP_Content.FPD_Boundaries)[2],
		((double*)pics[i]->FP_Content.FPD_Boundaries)[3],
		NULL);

    if(Datas4)
      Datas4[i][Index4] = x;
    else
      printf("%g",x);
  }
  if(Datas4) Index4++;
  else printf("\n");

ende:
  return(0);
error:
  return(-1);
}

/* ------------------------------------------------------------------ */
struct AnaMode Descript5 = {
  "zEROPOINT",
  "Find zero points of one dimensional data.",
  "\
Usage: analyze zeropoint <infile>\n\
  Find the zero points of the one-dimensional data and write the results\n\
  to stdout. Only the first picture in a sequence of <npics> is processed.\n\
  A cubic spline interpolation is used. Default for <npics> is %d.\n\
  Use '-q' option to get vanilla output.\n\
  Options:\n\
    -e<eps>  Values <= eps are treated as zero (defaults to 0.0).\n",
  NPICS5, do5
};

/* ***  For routine analyze5   *** */
double Epsilon5 = 0.0;

/*JS*********************************************************************
*   DO5
*************************************************************************/

static int do5(int argc,char *argv[])

/************************************************************************/
{
  int *readModes,i;

  if(argc != 1) return(-2);

  if((readModes = (int *)malloc(NPics * sizeof(*readModes))) == NULL)
    goto error;

  /*  How to read pictures  */
  readModes[0] = FPLOT_MODE_READ|FPLOT_MODE_CASTTODOUBLE|
    FPLOT_MODE_GENAXEDATA;
  for(i = 1 ; i < NPics ; i++) readModes[i] = 0;

  if(fplotAnalyze(argv,1,readModes,NPics,analyze5)) {
    fprintf(stderr,"ERROR in routine fplotAnalyze!\n");
    goto error;
  }
  free(readModes);

  return(0);
error:
  return(-1);
}

/*JS*********************************************************************
*   ANALYZE5
*************************************************************************/

static int analyze5(int ord,FPPicture *pics[],int nFiles,int nPics)

/************************************************************************/
{
  FPData *picData;

  if(Verbose)
    printf("%s:\n",pics[0]->FP_Title ? pics[0]->FP_Title : "<empty>");

  for(picData = &pics[0]->FP_Content ; picData ; picData = picData->FPD_Next) {
    double *x,*y,*y2;
    int i,n;

    if(picData->FPD_NDim != 1) goto error;

    n = picData->FPD_Dims[0];

    x = (double *)picData->FPD_AxeValues[0];
    y = (double *)picData->FPD_Data;

    if((y2 = spline(x,y,n,HUGE_VAL,HUGE_VAL,NULL)) == NULL) goto error;

    for(i = 0 ; i < (n-1) ; i++) {
      if(fabs(y[i]) <= Epsilon5) {
	printf(Verbose ? "  y[ %g ] = %g\n" : "%g ",x[i],y[i]);
      } else if((y[i] < 0.0 && y[i + 1] > 0.0) ||
		(y[i] > 0.0 && y[i + 1] < 0.0)) {
	double left,right,fLeft,fRight;

	left   = x[i];
	right  = x[i+1];
	fLeft  = splint(left,x,y,y2,n,i);
	fRight = splint(right,x,y,y2,n,i);

	while(fabs(left - right) > (EPSILON * fabs(left + right))) {
	  double f,fm;

	  f = 0.5 * (left + right);
	  fm = splint(f,x,y,y2,n,i);

	  if((fLeft < 0.0) ? fm < 0.0 : fm > 0.0) {
	    left  = f;
	    fLeft = fm;
	  } else {
	    right  = f;
	    fRight = fm;
	  }

	  if(fm == 0.0) break;
	}

	printf(Verbose ? "  y[ %g ] = 0.0\n" : "%g ",0.5 * (left + right));
      }
    }
    printf("\n");

    free(y2);
  }

  return(0);
error:
  return(-1);
}

/* ------------------------------------------------------------------ */
struct AnaMode Descript6 = {
  "cALCULATE",
  "Calculate values of one dimensional data at arbitrary points.",
  "\
Usage: analyze calculate <infile> <arg> ...\n\
  Read one-dimensional data from file <infile> and calculate the values of\n\
  positions <arg> .. of the data and write the results to stdout. Only the\n\
  first picture in a sequence of <npics> is processed. A cubic spline\n\
  interpolation is used. Default for <npics> is %d. Use '-q' option to get\n\
  vanilla output.\n",
  NPICS6, do6
};

/* ***  For routine analyze6   *** */
static double *Positions6;
static int     NPositions6;

/*JS*********************************************************************
*   DO6
*************************************************************************/

static int do6(int argc,char *argv[])

/************************************************************************/
{
  int *readModes,i;

  if(argc < 2) {
    fprintf(stderr,"ERROR: Not enough arguments!\n");
    goto error;
  }

  if((readModes = (int *)malloc(NPics * sizeof(*readModes))) == NULL)
    goto error;

  /*  How to read pictures  */
  readModes[0] = FPLOT_MODE_READ|FPLOT_MODE_CASTTODOUBLE|
    FPLOT_MODE_GENAXEDATA;
  for(i = 1 ; i < NPics ; i++) readModes[i] = 0;

  if((Positions6 = (double *)malloc((argc - 1) * sizeof(*Positions6))) == NULL)
    goto error;

  NPositions6 = argc - 1;
  for(i = 1 ; i < argc ; i++)  Positions6[i - 1] = strtod(argv[i],NULL);

  if(fplotAnalyze(argv,1,readModes,NPics,analyze6)) {
    fprintf(stderr,"ERROR in routine fplotAnalyze!\n");
    goto error;
  }

  free(Positions6);
  free(readModes);

  return(0);
error:
  return(-1);
}

/*JS*********************************************************************
*   ANALYZE6
*************************************************************************/

static int analyze6(int ord,FPPicture *pics[],int nFiles,int nPics)

/************************************************************************/
{
  FPData *picData;

  if(Verbose)
    printf("%s:\n",pics[0]->FP_Title ? pics[0]->FP_Title : "<empty>");

  for(picData = &pics[0]->FP_Content ; picData ; picData = picData->FPD_Next) {
    double *x,*y,*y2;
    int i,n;

    if(picData->FPD_NDim != 1) goto error;

    n = picData->FPD_Dims[0];

    x = (double *)picData->FPD_AxeValues[0];
    y = (double *)picData->FPD_Data;

    if((y2 = spline(x,y,n,HUGE_VAL,HUGE_VAL,NULL)) == NULL) goto error;

    for(i = 0 ; i < NPositions6 ; i++) {
      double val;

      val = splint(Positions6[i],x,y,y2,n,-1);

      if(Verbose) printf("  y[%g] = %g\n",Positions6[i],val);
      else printf("%g ",val);
    }

    printf("\n");

    free(y2);
  }

  return(0);
error:
  return(-1);
}

/* ------------------------------------------------------------------ */
struct AnaMode Descript7 = {
  "aVERAGE",
  "Average an expression over multidimensional data.",
  "\
Usage: analyze average <expr> <infile> [<outfile>]\n\
  Calculates the expectation value of the expression <expr> when summing\n\
  over the data from file <infile>. <npics> is interpreted as the dimension\n\
  of the data that is read (Defaults to %d). If <outfile> is given, plot the\n\
  values versus the number of the picture to file <outfile>, otherwise print\n\
  only the results to stdout. The axe values in the expression are denoted\n\
  \'x1\' .. and the function values \'f1\' ...\n\
  Options:\n\
    -c   Do not use volume element, instead divide by the number of\n\
         values which have been added up.\n",
  NPICS7, do7
};

/* ***  For routine analyze7  *** */
static const char *String7;
static Expression *Expr7;
static double *Data7;
static int Index7,MaxData7;

/*JS*********************************************************************
*   DO7
*************************************************************************/

static int do7(int argc,char *argv[])

/************************************************************************/
{
  int i;
  char **vars,*ptr,*string;
  char *outfile;
  FPFile *fp;
  int rMode[1] = { FPLOT_MODE_READ|FPLOT_MODE_NOGETMINMAX };

  if(argc == 2) outfile = NULL;
  else if(argc == 3) outfile = argv[2];
  else return(-2);

  if((vars = (char **)malloc((NPics + 2) * sizeof(*vars) +
			     (3 + (int)log10((double)NPics)) * NPics)) == NULL)
    goto error;

  /*  Preset variables  */
  ptr = (char *)&vars[NPics + 2];
  vars[0] = "f";
  for(i = 1 ; i <= NPics ; i++) {
    vars[i] = ptr;
    ptr += 1 + sprintf(ptr,"x%d",i);
  }
  vars[NPics + 1] = NULL;

  String7 = argv[0];

  if((Expr7 = CompExpression(String7,(const char **)vars,NULL,
			     (const char **)&string))
     == NULL)
    goto error;

  if(*string) {
    fprintf(stderr,"ERROR: Garbage in expression -- Break.\n>>%s\n",string);
    goto error;
  }

  if(outfile == NULL) Index7 = -1;

  if(fplotAnalyze(&argv[1],1,rMode,1,analyze7)) {
    fprintf(stderr,"ERROR in routine fplotAnalyze!\n");
    goto error;
  }

  if(Data7 && outfile) {
    if((fp = fplotOpen(outfile,(Mode & MODE_FPAPPEND) ? FPLOT_APPEND :
		       FPLOT_WRITE)) == NULL ||
       fplotStart((Mode & MODE_NOEND) ? NULL : fp) ||
       fplotText(fp,"\
Expectation value of expression \"%s\" (dim = %d) in file \"%s\"",
		 argv[0],Index7,argv[1]) ||
       fplotVector(fp,0,NULL,Data7,Index7,1,1.0,(double)Index7,1.0,1.0) ||
       fplotEnd((Mode & MODE_NOEND) ? NULL : fp) ||
       fplotClose(fp)) {
      fprintf(stderr,"ERROR in plotting -- Break.\n");
      goto error;
    }
  }

  FreeExpression(Expr7);

  return(0);
error:
  return(-1);
}

/*JS*********************************************************************
*   ANALYZE7
*************************************************************************/

static int analyze7(int ord,FPPicture *pics[],int nFiles,int nPics)

/************************************************************************/
{
  FPData *picData;

  if(Verbose)
    printf("%s:\n",pics[0]->FP_Title ? pics[0]->FP_Title : "<empty>");

  for(picData = &pics[0]->FP_Content ; picData ; picData = picData->FPD_Next) {
    double *data,*vals,sum,vol;
    int *index;
    int i,dim,arrayIndex1;

    dim = picData->FPD_NDim;

    /*  Stop if encountering different dimension  */
    if(dim == 0 || dim != NPics) return(1);

    if((vals = (double *)malloc((dim + 1) * sizeof(*vals))) == NULL ||
       (index = (int *)malloc(dim * sizeof(*index))) == NULL)
      goto error;

    data = (double *)picData->FPD_Data;

    /*  Preset variables  */
    for(i = 0 ; i < dim ; i++) {
      index[i] = 0;
      vals[1 + i] = ((double *)picData->FPD_Boundaries)[2 * i];
    }

    arrayIndex1 = 0;
    sum = 0.0;
    do {
      /*  Generate values and evaluate  */
      vals[0] = data[arrayIndex1];
      arrayIndex1++;

#ifdef DEBUG
      printf("VALS: ");
      for(i = 0 ; i <= dim ; i++) printf("%g ",vals[i]);
      printf("\n");
#endif

      /*  Evaluate  */
      sum += EvalExpression(Expr7,vals,NULL,NULL);

      /*  Increment  */
      for(i = 0 ; i < dim ; i++) {
	index[i]++;

	if(index[i] < picData->FPD_Dims[i]) {
	  vals[1 + i] = ((double *)picData->FPD_Boundaries)[2 * i] +
	    (index[i] * (((double *)picData->FPD_Boundaries)[2 * i + 1] -
			 ((double *)picData->FPD_Boundaries)[2 * i])) /
			   (picData->FPD_Dims[i] - 1);
	  break;
	}

	index[i] = 0;
	vals[1 + i] = ((double *)picData->FPD_Boundaries)[2 * i];
      }
    } while(i < dim);

    /*  Get volume element  */
    vol = 1.0;
    if(Mode & MODE7_VOLNELEM) {
      for(i = 0 ; i < dim ; i++)
	vol *= (double)picData->FPD_Dims[i];
      vol = 1.0 / vol;
    } else {
      for(i = 0 ; i < dim ; i++)
	vol *= (((double *)picData->FPD_Boundaries)[2 * i + 1] -
		((double *)picData->FPD_Boundaries)[2 * i]) /
	  (picData->FPD_Dims[i] - 1);
    }

    sum *= vol;

    if(Index7 >= 0) {
      if(Index7 >= MaxData7) {
	if((Data7 = (double *)realloc(Data7,(MaxData7 + 20) * sizeof(*Data7)))
	   == NULL) goto error;
	MaxData7 += 20;
      }
      Data7[Index7++] = sum;
    } else if(Verbose)
      printf(" <%s> = %g\n",String7,sum);
    else
      printf(" %g",sum);

    free(vals);
    free(index);
  }

  if(!Verbose && Index7 < 0) printf("\n");

  return(0);
error:
  return(-1);
}
