/*JS*********************************************************************
*
*    Program : ANALYZE
*    Language: ANSI-C
*    Author  : Joerg Schoen
*    Purpose : Analyze data in various ways.
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id$";
#endif

/*********     INCLUDES                                         *********/
#include "analyze.h"

/*********     DEFINES                                          *********/
/*  START-DEFINITIONS */
/* ***  For routine analyze1  *** */
#define NPICS1   2

#define MODE1_MONOTONICY    MODE_USER

/* ***  For routine transform2  *** */
#define NPICS2   1
#define MAXARCS  801

#define MODE2_SYMMETRIZE    MODE_USER
#define MODE2_LININTERPOL   (MODE_USER<<1)

/* ***  For routine transform3  *** */
#define NPICS3   2

#define MODE3_RENORMALIZE   MODE_USER
/*  END-DEFINITIONS  */

/*********     PROTOTYPES                                       *********/
Prototype struct AnaMode Descript1;
static int               do1(int argc,char *argv[]);
static int               analyze1(int ord,FPPicture *pics[],int nFiles,
				  int nPics);
Prototype double         rhoExpect(double *a,int nx,int ny,
				   double xMin,double xMax,
				   double yMin,double yMax,
				   double *phiExpect);

Prototype double         LinFactor1;

/* ------------------------------------------------------------------ */
Prototype struct AnaMode Descript2;
static int               do2(int argc,char *argv[]);
static int               transform2(int ord,FPPicture *pics[],int nPics);

Prototype int            MaxArcs2;

/* ------------------------------------------------------------------ */
Prototype struct AnaMode Descript3;
static int               do3(int argc,char *argv[]);
static int               transform3(int ord,FPPicture *pics[],int nPics);

/*********     GLOBAL/STATIC VARIABLES                          *********/
/* ------------------------------------------------------------------ */
struct AnaMode Descript1 = {
  "oVERLAP",
  "Overlap of dumped wavefunctions.",
  "\
Usage: analyze overlap <infile1> <infile2> [<outfile>]\n\
  Read <npics> consecutive pictures from <infile1> and <infile2>, whereas the\n\
  first picture in both files is assumed to be a dump of a wave functions,\n\
  the second to be a two dimensional plot of a wave function. Calculate\n\
  the norm, arc and the expectation value for rho and phi of the\n\
  overlap of these wave functions. The results are written to <outfile>.\n\
  Default for <npics> is %d.\n\
  Options:\n\
    -M        Preserve monotonicy of arcs in mode 1.\n\
    -l<fac>   Add up correction to arcs in mode 1: +fac*time.\n",
  NPICS1, do1
};

/* ***  For routine analyze1  *** */
static double *Data1a,*Data1b,*Data1c,*Data1d,*XData1;
static int Index1,MaxData1;

double LinFactor1;

/*JS*********************************************************************
*   DO1
*************************************************************************/

static int do1(int argc,char *argv[])

/************************************************************************/
{
  int *readModes,i;
  char *outfile;

  if(argc == 2) outfile = OUTPUTFILE;
  else if(argc == 3) outfile = argv[2];
  else return(-2);

  if((readModes = (int *)malloc(NPics * sizeof(*readModes))) == NULL)
    goto error;

  /*  How to read pictures  */
  readModes[0] = FPLOT_MODE_READ|FPLOT_MODE_CASTTODOUBLE|
    FPLOT_MODE_NOGETMINMAX;
  readModes[1] = FPLOT_MODE_READ|FPLOT_MODE_CASTTODOUBLE;
  for(i = 2 ; i < NPics ; i++) readModes[i] = 0;

  if(fplotAnalyze(argv,2,readModes,NPics,analyze1)) {
    fprintf(stderr,"ERROR in routine fplotAnalyze!\n");
    goto error;
  }
  free(readModes);

  if(Data1a) {
    FPFile *fp;

    if(LinFactor1 != 0.0) {
      double fac = LinFactor1;

      /*  Add linear factor to data  */
      for(i = 0 ; i < Index1 ; i++) Data1b[i] += fac * XData1[i];
    }

    if(Mode & MODE1_MONOTONICY) { /*  Preserve monotonicity  */
      double offset;

      offset = 0.0;
      for(i = 1 ; i < Index1 ; i++) {
	double data;

	data = Data1b[i] + offset;

	if(fabs(data - Data1b[i - 1]) > PI) {
	  offset += (data < Data1b[i-1]) ? (2.0*PI) : -(2.0*PI);
	  data = Data1b[i] + offset;
	}
	Data1b[i] = data;
      }
    }

    if((fp = fplotOpen(outfile,(Mode & MODE_FPAPPEND) ? FPLOT_APPEND :
		       FPLOT_WRITE)) == NULL ||
       fplotStart((Mode & MODE_NOEND) ? NULL : fp) ||
       fplotText(fp,"Norms of overlap \"%s\" with \"%s\"",argv[0],
		 argv[1]) ||
       fplotVector(fp,0,XData1,Data1a,Index1,1,XData1[0],XData1[Index1 - 1],
		   1.0,1.0) ||
       fplotEnd((Mode & MODE_NOEND) ? NULL : fp) ||
       fplotStart((Mode & MODE_NOEND) ? NULL : fp) ||
       fplotText(fp,LinFactor1 == 0.0 ? "Arcs of \"%s\" with \"%s\"" :
		 "Arcs of \"%s\" with \"%s\" (corrected with l=%g)",
		 argv[0],argv[1],LinFactor1) ||
       fplotVector(fp,0,XData1,Data1b,Index1,1,XData1[0],XData1[Index1 - 1],
		   1.0,1.0) ||
       fplotEnd((Mode & MODE_NOEND) ? NULL : fp) ||
       fplotStart((Mode & MODE_NOEND) ? NULL : fp) ||
       fplotText(fp,"<rho> of \"%s\" ",argv[0]) ||
       fplotVector(fp,0,XData1,Data1c,Index1,1,XData1[0],XData1[Index1 - 1],
		   1.0,1.0) ||
       fplotEnd((Mode & MODE_NOEND) ? NULL : fp) ||
       fplotStart((Mode & MODE_NOEND) ? NULL : fp) ||
       fplotText(fp,"<phi> of \"%s\" ",argv[0]) ||
       fplotVector(fp,0,XData1,Data1d,Index1,1,XData1[0],XData1[Index1 - 1],
		   1.0,1.0) ||
       fplotEnd((Mode & MODE_NOEND) ? NULL : fp) ||
       fplotClose(fp)) {
      printf("ERROR in plotting -- Break.\n");
      goto error;
    }
  }

  return(0);
error:
  return(-1);
}

/*JS*********************************************************************
*   ANALYZE1
*************************************************************************/

static int analyze1(int ord,FPPicture *pics[],int nFiles,int nPics)

/************************************************************************/
{
  dcmplx *p1,*p2,s;
  double n1,n2,r1,r2,r3,r4;
  char *string;
  int i;
  long dim;

  if(MaxRuns >= 0 && ord >= MaxRuns) return(1);

  if(pics[1]->FP_Content.FPD_NDim != 2) goto formatError;

  /*  Get dimension and check consistency  */
  for(i = 0 ; i < nFiles ; i++) {
    if(pics[i * nPics]->FP_Content.FPD_NDim != 1 ||
       pics[i * nPics]->FP_Content.FPD_Type !=
       (FPLOT_TYPE_COMPLEXDATA|FPLOT_TYPE_DUMPDATA)) goto formatError;

    if(i == 0) {
      dim = pics[i * nPics]->FP_Content.FPD_Dims[0] / 2;
    } else if(dim != (pics[i * nPics]->FP_Content.FPD_Dims[0] / 2)) {
      printf("ERROR: Inconsistent dimension of data (%ld != %ld)!\n",
	     dim,pics[i * nPics]->FP_Content.FPD_Dims[0]);
      goto error;
    }
  }

  /*  Get pointer to data  */
  p1 = (dcmplx *)pics[0]->FP_Content.FPD_Data;
  p2 = (dcmplx *)pics[nPics]->FP_Content.FPD_Data;

  if(Verbose)
    printf("Number %d, Dimension %ld:\n\
  \"%s\" <-> \"%s\"\n",1 + ord,dim,
	   pics[0]->FP_Title ? pics[0]->FP_Title : "<empty>",
	   pics[nPics]->FP_Title ? pics[nPics]->FP_Title : "<empty>");

  /*  Calc norms and scalar product  */
  n1 = sqrt(RE(scalprod(p1,p1,dim)));
  n2 = sqrt(RE(scalprod(p2,p2,dim)));
  s  = scalprod(p1,p2,dim);
  RE(s) /= (n1 * n2);
  IM(s) /= (n1 * n2);

  r1 = sqrt(ABS2C(s));
  r2 = atan2(IM(s),RE(s));

#ifdef OLD
  /*  Restrict to positive values  */
  if(r2 < 0.0) r2 += 2.0 * PI;
#endif

  /*printf(">>>>%s<<<<\n",pics[1]->FP_Title);*/
  r3 = rhoExpect((double*)pics[1]->FP_Content.FPD_Data,
		 pics[1]->FP_Content.FPD_Dims[0],
		 pics[1]->FP_Content.FPD_Dims[1],
		 ((double*)pics[1]->FP_Content.FPD_Boundaries)[0],
		 ((double*)pics[1]->FP_Content.FPD_Boundaries)[1],
		 ((double*)pics[1]->FP_Content.FPD_Boundaries)[2],
		 ((double*)pics[1]->FP_Content.FPD_Boundaries)[3],
		 &r4);

  if(Verbose)
    printf("  Normalized scalar product is %g + i %g = %g * e^(i %g)\n\n",
	   RE(s),IM(s),r1,r2);

  if(Index1 >= MaxData1) {
    if((Data1a = (double *)realloc(Data1a,(MaxData1 + 10) * sizeof(*Data1a)))
       == NULL ||
       (Data1b = (double *)realloc(Data1b,(MaxData1 + 10) * sizeof(*Data1b)))
       == NULL ||
       (Data1c = (double *)realloc(Data1c,(MaxData1 + 10) * sizeof(*Data1c)))
       == NULL ||
       (Data1d = (double *)realloc(Data1d,(MaxData1 + 10) * sizeof(*Data1d)))
       == NULL ||
       (XData1 = (double *)realloc(XData1,(MaxData1 + 10) * sizeof(*XData1)))
       == NULL)
      goto error;
    MaxData1 += 10;
  }

  /*  Extract time from title  */
  if(pics[0]->FP_Title == NULL ||
     (string = strstr(pics[0]->FP_Title,"t =")) == NULL ||
     sscanf(string,"t = %lg",&XData1[Index1]) != 1) {
    fprintf(stderr,"ERROR: Cannot extract time from \"%s\", nr %d\n",
	    pics[0]->FP_Title ? pics[0]->FP_Title : "",ord);
    XData1[Index1] = (double) Index1;
  }

  /*  Check if this time has appeared before  */
  for(i = Index1 ; i > 0 && XData1[i - 1] > XData1[Index1] ; ) i--;

  if(i > 0 && XData1[i - 1] == XData1[Index1]) {
    fprintf(stderr,"WARNING: Skipping second appearance of time %g at %d.\n",
	    XData1[Index1],ord);
  } else {
    Data1a[Index1] = r1;
    Data1b[Index1] = r2;
    Data1c[Index1] = r3;
    Data1d[Index1] = r4;
    Index1++;
  }

  return(0);
formatError:
  fprintf(stderr,"ERROR: Wrong data for me!\n");
error:
  return(-1);
}

/*JS*********************************************************************
*   RHOEXPECT
*************************************************************************/

double rhoExpect(double *a,int nx,int ny,double xMin,double xMax,
		 double yMin,double yMax,double *phiExpect)

/************************************************************************/
{
  int i,j;
  double x,y,dx,dy,result,norm,result2;

  dx = (xMax - xMin) / (nx - 1);
  dy = (yMax - yMin) / (ny - 1);

  result = result2 = norm = 0.0;

  for(i = 0 ; i < nx ; i++) {
    x = xMin + i * dx;

    for(j = 0 ; j < ny ; j++) {
      double rho;

      y = yMin + j * dy;

      rho = sqrt(x * x + y * y);

      norm   += a[i + nx * j];
      result += a[i + nx * j] * rho;

      if(phiExpect && rho > 0.0) {
	double phi = atan2(x,y);

	/*if(phi < 0.0) phi += (2.0 * PI);*/
	result2 +=  a[i + nx * j] * phi;
      }
    }
  }

  if(phiExpect) *phiExpect = (norm != 0.0) ? result2 / norm : 0.0;

  return(norm != 0.0 ? result / norm : 0.0);
}

/* ------------------------------------------------------------------ */
struct AnaMode Descript2 = {
  "iNTEGRATERHO",
  "Integrate coordinate rho of two dimensional wave functions.",
  "\
Usage: analyze integraterho <infile> [<outfile>]\n\
  Read <npics> consecutive pictures from <infile> and in the first one which\n\
  is assumed to be two-dimensional integrate over variable rho to get the\n\
  wave function as a function of phi solely. Write the results to <outfile>.\n\
  Default for <npics> is %d.\n\
  Options:\n\
    -a<narcs> Set number of intermediate arcs.\n\
    -s Symmetrize result by taking phi = -fabs(phi).\n\
    -S Use simple linear interpolation instead of splines for integration.\n",
  NPICS2, do2
};

/* ***  For routine transform2  *** */
int MaxArcs2 = MAXARCS;

/*JS*********************************************************************
*   DO2
*************************************************************************/

static int do2(int argc,char *argv[])

/************************************************************************/
{
  int readMode;
  char *outfile;

  if(argc == 1) outfile = OUTPUTFILE;
  else if(argc == 2) outfile = argv[1];
  else return(-2);

  readMode = FPLOT_MODE_READ|FPLOT_MODE_CASTTODOUBLE|FPLOT_MODE_NOGETMINMAX;
  if(Mode & MODE_NOEND) readMode |= FPLOT_TRANS_NOEND;
  if(Mode & MODE_FPAPPEND) readMode |= FPLOT_TRANS_APPEND;

  if(fplotTransform(argv[0],outfile,readMode,NPics,transform2)) {
    fprintf(stderr,"ERROR in routine fplotTransform!\n");
    return(-1);
  }

  return(0);
}

/*JS*********************************************************************
*   TRANSFORM2
*************************************************************************/

static int transform2(int ord,FPPicture *pics[],int nPics)

/************************************************************************/
{
  FPPicture *pic;
  int i,nx,ny;
  double x1a[2],x2a[2],dx,dy,rMax,dr,dphi;
  double *arcs,*data,*y2;

  if((MaxRuns >= 0 && ord >= MaxRuns) || pics[0]->FP_Content.FPD_NDim != 2)
    return(1);

  /*  Transform all but the first picture  */
  for(i = 1 ; i < nPics ; i++) {
    fplotFreepic(pics[i]);
    pics[i] = NULL;
  }

  pic = pics[0];

  nx = pic->FP_Content.FPD_Dims[0];
  ny = pic->FP_Content.FPD_Dims[1];

  x1a[0] = ((double*)pic->FP_Content.FPD_Boundaries)[0];
  x1a[1] = ((double*)pic->FP_Content.FPD_Boundaries)[1];
  x2a[0] = ((double*)pic->FP_Content.FPD_Boundaries)[2];
  x2a[1] = ((double*)pic->FP_Content.FPD_Boundaries)[3];

  dx = (x1a[1] - x1a[0]) / (nx - 1);
  dy = (x2a[1] - x2a[0]) / (ny - 1);

  rMax = MAX(fabs(x1a[0]),fabs(x1a[1]));
  rMax = MAX(rMax,fabs(x2a[0]));
  rMax = MAX(rMax,fabs(x2a[1]));
  dr = 0.5 * MIN(dx,dy);  /*dr = 0.5 * rMax / MAX(nx,ny);*/
  dphi = (2.0 * PI) / (double)(MaxArcs2 - 1);

  if((arcs = (double*) malloc(MaxArcs2 * sizeof(*arcs))) == NULL) goto error;

  data = (double *)pic->FP_Content.FPD_Data;

  /*  Get coefficients from two dimensional spline  */
  if(!(Mode & MODE2_LININTERPOL) &&
     (y2 = spline2d(x1a,x2a,data,-nx,-ny,NULL)) == NULL) goto error;

  for(i = 0 ; i < MaxArcs2 ; i++) {
    double r,phi,s,c;

    phi = -PI + i * dphi;
    s = sin(phi);
    c = cos(phi);

    /*  Symmetrize data using only negative phi's  */
    if(Mode & MODE2_SYMMETRIZE) phi = -fabs(phi);

    for(arcs[i] = 0.0, r = dr ; r <= rMax ; r += dr) {
      if(Mode & MODE2_LININTERPOL) {
	int ix,iy;
	double x,y,d00,d01,d10,d11;

	/*  Get coordinates of start point and fraction  */
	x = (r * s - x1a[0]) / dx;
	y = (r * c - x2a[0]) / dy;
	ix = (int)x;
	iy = (int)y;
	x -= ix;
	y -= iy;

	/*  Interpolate linearly between four neighbouring points  */
	d00 = d01 = d10 = d11 = 0.0; /*  Set to zero outside data  */
	if(0 <= ix     && ix     < nx && 0 <= iy     && iy     < ny)
	  d00 = data[ix     + nx * iy];
	if(0 <= (ix+1) && (ix+1) < nx && 0 <= iy     && iy     < ny)
	  d01 = data[(ix+1) + nx * iy];
	if(0 <= ix     && ix     < nx && 0 <= (iy+1) && (iy+1) < ny)
	  d10 = data[ix     + nx * (iy+1)];
	if(0 <= (ix+1) && (ix+1) < nx && 0 <= (iy+1) && (iy+1) < ny)
	  d11 = data[(ix+1) + nx * (iy+1)];

	/*  Include volume element in computation  */
	arcs[i] += r * dr *
	  ((1.0 - y) * ((1.0 - x) * d00 + x * d01) +
	   y         * ((1.0 - x) * d10 + x * d11));
      } else {
	double x,y;

	/*  Get coordinates  */
	x = r * s;
	y = r * c;

	/*  Include volume element in computation  */
	arcs[i] += r * dr * splint2d(x,y,x1a,x2a,data,-nx,-ny,y2);
      }
    }
  }

  if(!(Mode & MODE2_LININTERPOL)) free(y2);

  free(pic->FP_Content.FPD_Data);
  pic->FP_Content.FPD_Data = arcs;

  /*  Convert to 2D picture  */
  pic->FP_Content.FPD_NDim = 1;

  pic->FP_Content.FPD_Dims[0] = MaxArcs2;

  ((double*)pic->FP_Content.FPD_Boundaries)[0] = -PI;
  ((double*)pic->FP_Content.FPD_Boundaries)[1] = PI;
  pic->FP_Content.FPD_FMin = 1.0;
  pic->FP_Content.FPD_FMax = 1.0;

  if(changeTitle(pic,"rho average of \"%s\"")) goto error;

  if(Verbose)
    printf("%2d: %s\n",1 + ord,pic->FP_Title ? pic->FP_Title : "<empty>");

  return(0);
error:
  return(-1);
}

/* ------------------------------------------------------------------ */
struct AnaMode Descript3 = {
  "suBSTRACT",
  "Substract plotted wave function from following ones.",
  "\
Usage: analyze substract <infile> [<outfile>]\n\
  Take first <npics> two-dimensional pictures from <infile> and subtract\n\
  them from all the following ones. Write the results to <outfile>. Default\n\
  for <npics> is %d.\n\
  Options:\n\
    -r Rescale subtracted packets before subtracting.\n",
  NPICS3, do3
};

/*JS*********************************************************************
*   DO3
*************************************************************************/

static int do3(int argc,char *argv[])

/************************************************************************/
{
  int readMode;
  char *outfile;

  if(argc == 1) outfile = OUTPUTFILE;
  else if(argc == 2) outfile = argv[1];
  else return(-2);

  readMode = FPLOT_MODE_READ|FPLOT_MODE_CASTTODOUBLE|FPLOT_MODE_NOGETMINMAX;
  if(Mode & MODE_NOEND) readMode |= FPLOT_TRANS_NOEND;
  if(Mode & MODE_FPAPPEND) readMode |= FPLOT_TRANS_APPEND;

  if(fplotTransform(argv[0],outfile,readMode,NPics,transform3)) {
    fprintf(stderr,"ERROR in routine fplotTransform!\n");
    return(-1);
  }

  return(0);
}

/*JS*********************************************************************
*   TRANSFORM3
*************************************************************************/

static int transform3(int ord,FPPicture *pics[],int nPics)

/************************************************************************/
{
  int n,i,j;
static double **PicSave;
static int NxSave,NySave;

  if((MaxRuns >= 0 && ord >= MaxRuns) || pics[0]->FP_Content.FPD_NDim != 2)
    return(1);

  if(ord == 0) {
    NxSave = pics[0]->FP_Content.FPD_Dims[0];
    NySave = pics[0]->FP_Content.FPD_Dims[1];

    if(Verbose)
      printf("  Saving first %d pictures of dimension %dx%d\n",
	     nPics,NxSave,NySave);

    for(n = 0 ; n < nPics ; n++)
      if(pics[n]->FP_Content.FPD_NDim != 2 ||
	 pics[n]->FP_Content.FPD_Dims[0] != NxSave ||
	 pics[n]->FP_Content.FPD_Dims[1] != NySave) {
	printf("ERROR: Illegal format -- Break.\n");
	goto error;
      }

    if((PicSave = (double **)malloc(nPics * sizeof(*PicSave))) == NULL ||
       (PicSave[0] = (double *)malloc(nPics * NxSave * NySave *
				      sizeof(**PicSave))) == NULL)
      goto error;

    for(n = 1 ; n < nPics ; n++) PicSave[n] = &PicSave[n - 1][NxSave * NySave];

    for(n = 0 ; n < nPics ; n++) {
      double norm;
      double *data = (double *) pics[n]->FP_Content.FPD_Data;

      norm = 0.0;
      for(j = 0 ; j < NySave ; j++)
	for(i = 0 ; i < NxSave ; i++)
	  norm += PicSave[n][i + NxSave * j] = data[i + NxSave * j];
      norm *= (((double *)pics[n]->FP_Content.FPD_Boundaries)[1] -
	       ((double *)pics[n]->FP_Content.FPD_Boundaries)[0]) *
		 (((double *)pics[n]->FP_Content.FPD_Boundaries)[3] -
		  ((double *)pics[n]->FP_Content.FPD_Boundaries)[2]) /
		    ((NxSave - 1) * (NySave - 1));

      if((Mode & MODE3_RENORMALIZE) && norm != 0.0) {
#ifdef DEBUG
	printf("NORM %g\n",norm);
#endif
	norm = 1.0 / norm;
	for(j = 0 ; j < NySave ; j++)
	  for(i = 0 ; i < NxSave ; i++)
	    PicSave[n][i + NxSave * j] *= norm;
      }
    }
  }

  /*  Substract data  */
  if(Verbose)
    printf("%2d: Subtracting.\n",1 + ord);

  for(n = 0 ; n < nPics ; n++) {
    double *data = (double *) pics[n]->FP_Content.FPD_Data;

    if(Mode & MODE3_RENORMALIZE) {
      double norm;

      norm = 0.0;
      for(j = 0 ; j < NySave ; j++)
	for(i = 0 ; i < NxSave ; i++)
	  norm += data[i + NxSave * j];
      norm *= (((double *)pics[n]->FP_Content.FPD_Boundaries)[1] -
	       ((double *)pics[n]->FP_Content.FPD_Boundaries)[0]) *
		 (((double *)pics[n]->FP_Content.FPD_Boundaries)[3] -
		  ((double *)pics[n]->FP_Content.FPD_Boundaries)[2]) /
		    ((NxSave - 1) * (NySave - 1));

      for(j = 0 ; j < NySave ; j++)
	for(i = 0 ; i < NxSave ; i++)
	  data[i + NxSave * j] = fabs(data[i + NxSave * j] -
				      norm * PicSave[n][i + NxSave * j]);
    } else {
      for(j = 0 ; j < NySave ; j++)
	for(i = 0 ; i < NxSave ; i++)
	  data[i + NxSave * j] = fabs(data[i + NxSave * j] -
				      PicSave[n][i + NxSave * j]);
    }
  }

  return(0);
error:
  return(-1);
}
