/*JS*********************************************************************
*
*    Program : ANALYZE
*    Language: ANSI-C
*    Author  : Joerg Schoen
*    Purpose : Analyze data in various ways.
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id$";
#endif

/*********     INCLUDES                                         *********/
#include "analyze.h"

/*********     DEFINES                                          *********/
#ifndef OUTPUTFILE
/*  START-DEFINITIONS */
/*  Structure to store one mode  */
struct AnaMode {
  const char *ModeName,*ShortHelp,*LongHelp;
  int NPics;
  int (*ModeFunc)(int argc,char *argv[]);
};

/*  Default name of output file  */
#define OUTPUTFILE   "ana.out"

#define MODE_FPAPPEND       (1<<0)
#define MODE_NOEND          (1<<1)

#define MODE_NOOUTPUT       (1<<2)

#define MODE_USER           (1<<8)
/*  END-DEFINITIONS  */
#endif

#define OPTSTRING  "hqn:m:sSa:rMl:e:ANpgfbc"

/*********     PROTOTYPES                                       *********/
static int               help(char *mode);
Prototype int            changeTitle(FPPicture *pic,char *mess);

Prototype int            Verbose,Mode,NPics,MaxRuns;

/* ------------------------------------------------------------------ */
static struct AnaMode    Descript0;
static int               do0(int argc,char *argv[]);
static int               transform0(int ord,FPPicture *pics[],int nPics);

/*********     GLOBAL/STATIC VARIABLES                          *********/
int Verbose = 1;
int Mode;

int NPics = -1;
int MaxRuns = -1;

static struct AnaMode *Descriptions[] = {
  &Descript0, &Descript1, &Descript2, &Descript3, &Descript4,
  &Descript5, &Descript6, &Descript7, &Descript8, &Descript9,
  &Descript10, &Descript11, &Descript12, &Descript13
};

/*JS*********************************************************************
*   HELP
*************************************************************************/

static int help(char *mode)

/************************************************************************/
{
  int i;

  if(mode == NULL) {
    printf("\
Usage: analyze [<opt>] <mode> <args> ...                 Joerg Schoen 1994-97\n\
  The actual analysis depends on <mode>, which may be one of (upper case\n\
  characters are optional):\n");

    for(i = 0 ; i < (sizeof(Descriptions) / sizeof(Descriptions[0])) ; i++)
      printf("    %s: %s\n",Descriptions[i]->ModeName,
	     Descriptions[i]->ShortHelp);

    printf("\
  To get more information about a specific mode, enter \"analyze -h <mode>\".\n\
  The default output file for a mode is \"%s\".\n\
  Overall options:\n\
    -h        Displays this help information.\n\
    -q        Be quiet during operation. This is the default when output\n\
              goes not to a terminal.\n\
    -A        Append to output files.\n\
    -N        Suppress end marker in output file.\n\
    -n<npics> Set number of consecutive pictures to read. Default is\n\
              different for each mode.\n\
    -m<max>   Set number of maximum transformations to do.\n",OUTPUTFILE);

    return(0);
  }

  for(i = 0 ; i < (sizeof(Descriptions) / sizeof(Descriptions[0])) ; i++)
    if(striacmp(mode,Descriptions[i]->ModeName) == 0) {
      printf(Descriptions[i]->LongHelp,Descriptions[i]->NPics);
      return(0);
    }

  fprintf(stderr,"ERROR: No mode \"%s\"!\n",mode);

  return(-1);
}

/*JS*********************************************************************
*   MAIN
*************************************************************************/

int main(int argc,char *argv[])

/************************************************************************/
{
  char *string;
  int i,flag,helpFlag;

  /*  Get search path for input files from environment variable  */
  if((FPlotOpenPath = getenv(JSENVPATH)) == NULL) errno = 0;

#ifndef CONFIG_NO_POSIX
  /*  If not writing to a terminal, suppress output  */
  if(!isatty(fileno(stdout))) Verbose = 0;
#endif

  helpFlag = FALSE;

  /*  First scan general options, then command line  */
  for(flag = 1 ; flag >= 0 ; flag--) {
    char *env;

    if(flag == 1) {
      if((env = getenv(JSENVOPT)) == NULL) continue;
    } else
      env = NULL;

    while((i = jsGetOpt(env ? NULL : &argc,env ? (void *) env :
			(void *)argv,OPTSTRING,&string)) != EOF) {
      switch(i) {
      default:
      case 'h':
	helpFlag = TRUE;
	break;
      case 'q':
	Verbose = 0;
	break;
      case 'A':
	Mode |= MODE_FPAPPEND;
	break;
      case 'N':
	Mode |= MODE_NOEND;
	break;
      case 'n':
	NPics = atoi(string);
	break;
      case 'm':
	MaxRuns = atoi(string);
	break;
      /* -------------------------------------- */
      case 'M':
	Mode |= MODE1_MONOTONICY;
	break;
      case 'l':
	LinFactor1 = strtod(string,NULL);
	break;
      /* -------------------------------------- */
      case 'a':
	MaxArcs2 = atoi(string);
	break;
      case 's':
	Mode |= MODE2_SYMMETRIZE;
	break;
      case 'S':
	Mode |= MODE2_LININTERPOL;
	break;
      /* -------------------------------------- */
      case 'r':
	Mode |= MODE3_RENORMALIZE;
	break;
      /* -------------------------------------- */
      case 'e':
	Epsilon5 = strtod(string,NULL);
	break;
      /* -------------------------------------- */
      case 'c':
	Mode |= MODE7_VOLNELEM;
	break;
      /* -------------------------------------- */
      case 'f':
	Mode |= MODE8_FFTMETHOD;
	break;
      /* -------------------------------------- */
      case 'g':
	Mode |= MODE10_GENAXEDATA;
	break;
      /* -------------------------------------- */
      case 'p':
	Mode |= MODE11_POLYNOMINT;
	break;
      /* -------------------------------------- */
      case 'b':
	Mode |= MODE12_BACKTRAFO;
	break;
      }
    }
  }

  /*  Print general or specific help  */
  if(argc < 3 || helpFlag) {
    help(argc < 2 ? NULL : argv[1]);
    return(5);
  }

  /*  Find mode  */
  for(i = 0 ; i < (sizeof(Descriptions) / sizeof(Descriptions[0])) ; i++)
    if(striacmp(argv[1],Descriptions[i]->ModeName) == 0) {
      int ret;

      if(NPics == -1) NPics = Descriptions[i]->NPics;

      ret = (*(Descriptions[i]->ModeFunc))(argc - 2,&argv[2]);
      if(ret == -2) goto noargs;
      if(ret) goto error;
      break;
    }

  if(i == (sizeof(Descriptions) / sizeof(Descriptions[0]))) {
    fprintf(stderr,"ERROR analyze: Unknown mode \"%s\" -- Break.\n",argv[1]);
    return(10);
  }

  return(0);
noargs:
  fprintf(stderr,"ERROR: Not enough arguments for mode \"%s\" -- Break.\n",
	  string);
error:
  jsperror("analyze");
  return(30);
}

/*JS*********************************************************************
*   CHANGETITLE
*************************************************************************/

int changeTitle(FPPicture *pic,char *mess)

/************************************************************************/
{
  char *string,*string2,*title;

  if((title = pic->FP_Title) == NULL) title = "<empty>";

  if((string = (char *)malloc(strlen(mess) + strlen(title))) == NULL)
    return(-1);

  /*  Preserve only first line of title  */
  if((string2 = strchr(title,'\n'))) *string2 = '\0';

  sprintf(string,mess,title);

  if(pic->FP_Title) free(pic->FP_Title);
  pic->FP_Title = string;

  return(0);
}

/* ------------------------------------------------------------------ */
static struct AnaMode Descript0 = {
  "gENERATE",
  "Generate selected features for data.",
  "\
Usage: analyze generate <infile> <outfile> <feature> ...\n\
  Read data from <infile> and write it to <outfile>, generating selected\n\
  features in the data:\n\
    fLOAT:   Cast all values to single precision.\n\
    dOUBLE:  Cast all values to double precision.\n\
    aXEDATA: Generate value for axes.\n\
    sORT:    Sort all axe values in increasing order.\n\
    bOUNDS:  Generate minimum/ maximum values for axes and data.\n\
    tRUNC <d> <t>: Truncate all values, retaining only <d> binary digits and\n\
             set numbers whose absolute value is below <t> to zero.\n",
  1, do0
};

/* ***  For routine transform0  *** */
static int Digits0;
static double Thresh0;

/*JS*********************************************************************
*   DO0
*************************************************************************/

static int do0(int argc,char *argv[])

/************************************************************************/
{
  char *outfile;
  int readMode,i,j;
static struct {
  char *Name;
  int   IFlag;
  int   Flag;
} GenNames[] = {
  { "fLOAT",   FPLOT_MODE_CASTTODOUBLE, FPLOT_MODE_CASTTOFLOAT  },
  { "dOUBLE",  FPLOT_MODE_CASTTOFLOAT,  FPLOT_MODE_CASTTODOUBLE },
  { "aXEDATA", 0,                       FPLOT_MODE_GENAXEDATA   },
  { "sORT",    0,                       FPLOT_MODE_SORTDATA     },
  { "bOUNDS",  FPLOT_MODE_NOGETMINMAX,  0                       },
  { NULL, 0 }
};
  int (*transFunc)(int ord,FPPicture *pics[],int nPics);

  if(argc < 1) return(-2);

  outfile = (argc < 2) ? OUTPUTFILE : argv[1];
  transFunc = NULL;

  /*  What to do?  */
  readMode = FPLOT_MODE_READ | FPLOT_MODE_NOGETMINMAX;
  for(i = 2 ; i < argc ; i++) {
    if(striacmp(argv[i],"tRUNC") == 0) {
      char *string;

      if((i += 2) >= argc) goto syntError;
      Digits0 = strtol(argv[i-1],&string,10);
      if(string == argv[i-1] || *string) goto syntError;
      Thresh0 = strtod(argv[i],&string);
      if(string == argv[i] || *string) goto syntError;
      transFunc = transform0;
      continue;
    }

    for(j = 0 ; GenNames[j].Name ; j++)
      if(striacmp(argv[i],GenNames[j].Name) == 0) {
	readMode = (readMode & ~(GenNames[j].IFlag)) | GenNames[j].Flag;
	break;
      }

    if(GenNames[j].Name == NULL) {
      fprintf(stderr,"ERROR: Illegal feature \"%s\" -- Break.\n",argv[i]);
      goto error;
    }
  }

  if(fplotTransform(argv[0],outfile,readMode,NPics,transFunc)) {
    fprintf(stderr,"ERROR in routine fplotTransform!\n");
    goto error;
  }

  return(0);
syntError:
  fprintf(stderr,"ERROR: Wrong syntax in feature \"trunc\"-- Break.\n");
error:
  return(-1);
}

/*JS*********************************************************************
*   TRANSFORM0
*************************************************************************/

static int transform0(int ord,FPPicture *pics[],int nPics)

/************************************************************************/
{
  int j;
  FPData *picData;
  double thresh;
  int digits;

  thresh = Thresh0;
  digits = Digits0;

  for(j = 0 ; j < nPics ; j++) {
    for(picData = &pics[j]->FP_Content ; picData ;
	picData = picData->FPD_Next) {
      int dim,i;

      /*  Get full dimension  */
      for(dim = 1, i = 0 ; i < picData->FPD_NDim ; i++)
	dim *= picData->FPD_Dims[i];

      /*  Transform data, axe values and boundaries  */
      if(picData->FPD_Flags & FPLOT_FLAGS_SINGLE) {
	float *data;

	/*  Data values */
	data = (float *)picData->FPD_Data;
	for(i = 0 ; i < dim ; i++)
	  data[i] = truncNumber(data[i],thresh,digits);

	/*  Boundary values  */
	data = (float *)picData->FPD_Boundaries;
	for(i = 0 ; i < (2*picData->FPD_NDim) ; i++)
	  data[i] = truncNumber(data[i],thresh,digits);

	/*  Axe values  */
	if(picData->FPD_AxeValues) {
	  int n,len;

	  for(n = 0 ; n < picData->FPD_NDim ; n++) {
	    if((data = (float *)picData->FPD_AxeValues[n]) == NULL) continue;
	    len = picData->FPD_Dims[n];

	    for(i = 0 ; i < len ; i++)
	      data[i] = truncNumber(data[i],thresh,digits);
	  }
	}
      } else {
	double *data;

	/*  Data values */
	data = (double *)picData->FPD_Data;
	for(i = 0 ; i < dim ; i++)
	  data[i] = truncNumber(data[i],thresh,digits);

	/*  Boundary values  */
	data = (double *)picData->FPD_Boundaries;
	for(i = 0 ; i < (2*picData->FPD_NDim) ; i++)
	  data[i] = truncNumber(data[i],thresh,digits);

	/*  Axe values  */
	if(picData->FPD_AxeValues) {
	  int n,len;

	  for(n = 0 ; n < picData->FPD_NDim ; n++) {
	    if((data = (double *)picData->FPD_AxeValues[n]) == NULL) continue;
	    len = picData->FPD_Dims[n];

	    for(i = 0 ; i < len ; i++)
	      data[i] = truncNumber(data[i],thresh,digits);
	  }
	}
      }

      /*  Data boundaries  */
      picData->FPD_FMin = truncNumber(picData->FPD_FMin,thresh,digits);
      picData->FPD_FMax = truncNumber(picData->FPD_FMax,thresh,digits);
    }
  }

  return(0);
}
