/*JS*********************************************************************
*
*    Program : A2FPLOT
*    Language: ANSI-C
*    Author  : Joerg Schoen
*    Purpose : Read ASCII data and write in fplot style.
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id: a2fplot.c,v 1.1 1997/03/06 09:50:35 joerg Stab joerg $";
#endif

/*********     INCLUDES                                         *********/
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>

#include <errno.h>

#include <jssubs.h>

#include <fplot.h>

/*********     DEFINES                                          *********/
#define OPTSTRING   "ho:a:t:x:npT:"

#define COMMENTSTART   '#'

#define DEFOUTNAME   "a2fplot.out"

#define MODE_APPEND     (1<<0)
#define MODE_NOEND      (1<<1)
#define MODE_PARALLEL   (1<<2)

/*********     PROTOTYPES                                       *********/
static double           *readData(FILE *fp,long *pLen);

/*JS*********************************************************************
*   MAIN
*************************************************************************/

int main(int argc,char *argv[])

/************************************************************************/
{
  FPFile *fp;
  char *string,*outName,*text,*axes;
  double *data,**axeData,*bounds;
  unsigned long *dims;
  long dim;
  int i,mode,nDim,type;

  outName = DEFOUTNAME;
  text = axes = NULL;
  mode = 0;
  type = 0;
  while((i = jsGetOpt(&argc,argv,OPTSTRING,&string)) != EOF) {
    switch(i) {
    default:
      goto error2;
    case 'h':
      goto help;
    case 'a':
      mode |= MODE_APPEND;
    case 'o':
      outName = string;
      break;
    case 't':
      text = string;
      break;
    case 'x':
      axes = string;
      break;
    case 'n':
      mode |= MODE_NOEND;
      break;
    case 'p':
      mode |= MODE_PARALLEL;
      break;
    case 'T':
      type = atoi(string);
      break;
    }
  }

  /*  Get rank of data  */
  nDim = MAX(argc - 1,1);

  /*  Set up work arrays  */
  if((dims = (unsigned long *)malloc(nDim * sizeof(*dims))) == NULL ||
     (bounds = (double *)malloc(2 * nDim * sizeof(*bounds))) == NULL ||
     (axeData = (double **)malloc(nDim * sizeof(*axeData))) == NULL)
    goto error;
  for(i = 0 ; i < nDim ; i++) {
    axeData[i] = NULL;
    /*  Default interval is [0,1]  */
    bounds[2*i]   = 0.0;
    bounds[2*i+1] = 1.0;
  }

  /*  Read data  */
  if((data = readData(stdin,&dim)) == NULL) {
    if(errno) goto error;
    goto error2;
  }

  if((mode & MODE_PARALLEL) && nDim == 1) {
    double *nData;
    int len;

    /*  Mark that data contains axe values at the end  */
    bounds[1] = 0.0;

    /*  Resort one dimensional data  */
    if((nData = (double *)malloc(dim * sizeof(*nData))) == NULL)
      goto error;

    len = dim / 2;
    for(i = 0 ; i < len ; i++) {
      nData[i]       = data[2 * i + 1]; /*  y values  */
      nData[len + i] = data[2 * i];     /*  x values  */
    }
    free(data);
    data = nData;
  }

  /*  Now set up dimension of data  */
  if(argc > 1) {
    long dim2;
    int index;

    dim2 = 1;
    index = -1;
    for(i = 0 ; i < nDim ; i++) {
      int index1;

      index1 = -1;
      if(argv[1 + i][0] == '*' || argv[1 + i][0] == '.') {
	/*  This dimension should be calculated from number of data  */
	index1 = i;
	string = argv[1 + i] + 1;
      } else {
	/*  Dimension directly given  */
	dim2 *= dims[i] = strtoul(argv[1 + i],&string,10);
	if(argv[1 + i] == string) goto error;
      }

      /*  Anything appended?  */
      if(*string == '+') {
	/*  Mark that this dimension will get axe values
	 *   from trailing values in data.
	 */
	bounds[2*i+1] = 0.0;
      } else if(*string == 'f') { /*  Read axe data from extra file  */
	FILE *aFp;
	long len;

	/*  Boundaries will be determined later  */
	bounds[2*i+1] = 0.0;

	/*  File name where axe values are found is appended  */
	if((aFp = fopen(string + 1,"r")) == NULL ||
	   (axeData[i] = readData(aFp,&len)) == NULL) {
	  if(errno) goto error;
	  goto error2;
	}

	/*  Requested determination from numbers read?  */
	if(index1 == i) {
	  /*  Take number of data values from file  */
	  dim2 *= dims[i] = len;
	  index1 = -1;
	} else if(len < dims[i]) {
	  fprintf(stderr,"\
a2fplot: Not enough values for axe in file \"%s\" -- Break.\n",string + 1);
	  goto error;
	} else if(len > dims[i]) {
	  fprintf(stderr,"\
WARNING: %ld superfluous values for axe in file \"%s\" -- Ignored.\n",
		  len - dims[i],string + 1);
	}

	if(fclose(aFp)) goto error;
      }

      if(index1 >= 0) {
	if(index >= 0) {
	  fprintf(stderr,"\
a2fplot: Only one dimension can be estimated -- Break.\n");
	  goto error;
	}
	index = index1;
      }
    }

    /*  Any dimension to estimate?  */
    if(index >= 0) {
      dims[index] = dim / (bounds[2*index+1] == 0.0 ? (dim2 + 1) : dim2);
      dim2 *= dims[index];
    }

    /*  Now handle extra values that belong to some axes  */
    for(i = 0 ; i < nDim ; i++)
      if(bounds[2*i+1] == 0.0 && axeData[i] == NULL) {
	if((dim - dim2) < dims[i]) {
	  fprintf(stderr,"\
WARNING: Not enough values for axes. Remaining axes not set up.\n");
	  break;
	}
	axeData[i] = &data[dim2];
	dim2 += dims[i];
      }

    /*  Now check if enough data are present  */
    if(dim2 < dim) {
      fprintf(stderr,"WARNING: %ld superfluous data values -- Ignored.\n",
	      dim - dim2);
    } else if(dim2 > dim) {
      fprintf(stderr,"\
a2fplot: To few values for required dimension -- Break.\n");
      goto error2;
    }
  } else {
    /*  Simplest case: one dimensional data  */
    if(mode & MODE_PARALLEL) {
      dims[0] = dim / 2;
      axeData[0] = &data[dims[0]];
      if(dim & 1)
	fprintf(stderr,"WARNING: Superfluous data value -- Ignored.\n");
    } else {
      dims[0] = dim;
    }
  }

  /* **  Now do the job  ** */
  if((fp = fplotOpen2(outName,(mode & MODE_APPEND) ? FPLOT_APPEND :
		      FPLOT_WRITE)) == NULL)
    goto error;

  if(fplotStart((mode & MODE_NOEND) ? NULL : fp) ||
     (text && fplotText(fp,text,dim)) ||
     fplotTensor(fp,type,nDim,(const double **)axeData,data,dims,NULL,bounds,
		 1.0,1.0) ||
     (axes && fplotAxesS(fp,axes)) ||
     fplotEnd((mode & MODE_NOEND) ? NULL : fp))
    goto error;

  if(fplotClose(fp)) goto error;

  free(data);
  free(axeData);
  free(bounds);
  free(dims);

  return(0);
error:
  jsperror("a2fplot");
error2: /*  Error message already printed  */
  return(30);
help:
  fprintf(stderr,"\
Usage: a2fplot [<opts>] [<dim> ...]                        Joerg Schoen 1996\n\
  Read data values from standard input and write them in 'fplot' style to\n\
  \"%s\". White space and comment lines (introduced via '%c') are\n\
  ignored. If no dimensions are given, write as one dimensional data. If a\n\
  dimension is '*' or '.', it is automatically calculated from the number of\n\
  data read. If a '+' sign is appended to a dimension, the axe values are\n\
  taken from the remaining data that have been read, if a 'f' and then a\n\
  filename is appended, axe data are read from this file. Otherwise no axe\n\
  data is produced for this dimension.\n\
  Options:\n\
    -h Show this help information.\n\
    -o <file>  Change name of output file. '-' means standard output.\n\
    -a <file>  Same as '-o', but appends.\n\
    -T <type>  Set type of data to plot.\n\
    -t <text>  Write additional text to output file.\n\
    -x <text>  Write axe description to output file. Text for different axes\n\
               must be separated with ';'.\n\
    -n         Suppress end marker in output file.\n\
    -p         Treat one dimensional data as pairs <xvalue> <yvalue>.\n\
",DEFOUTNAME,COMMENTSTART);
  return(5);
}

/*JS*********************************************************************
*   READDATA Read double values from a file until EOF. Values may be
*    separated by space or ','. Comment lines, starting with '#', are
*    ignored.
*************************************************************************/

static double *readData(FILE *fp,long *pLen)

/************************************************************************/
{
  double *ptr;
  int ptrI,ptrM;

  /*  Preset pointers  */
  ptr = NULL;
  ptrI = ptrM = 0;

  for(;;) {
    double x;
    int c;

    /*  Skip any whitespace  */
    do
      c = getc(fp);
    while((isspace(c) || c == ',') && c != EOF);

    if(c == COMMENTSTART) {
      /*  Read til EOL  */
      do
	c = getc(fp);
      while(c != '\n' && c != EOF);
      if(c != EOF) continue;
    }

    /*  Prevent pushing back EOF  */
    if(c == EOF) break;
    ungetc(c,fp);

    if(fscanf(fp,"%lg",&x) != 1) break;

    if(ptrI >= ptrM) {
      ptrM += MAX(ptrM,1000);
      if((ptr = (double *)realloc(ptr,ptrM * sizeof(*ptr))) == NULL)
	goto error;
    }
    ptr[ptrI++] = x;
  }

  if(!feof(fp))
    fprintf(stderr,"WARNING: Stopped reading after %d values.\n",ptrI);

  /*  Handle special case where no data values present  */
  if(ptrI == 0) {
    fprintf(stderr,"a2fplot: No values in file.\n");
    return(NULL);
  }

  /*  Save used memory  */
  ptr = (double *)realloc(ptr,ptrI * sizeof(*ptr));

  *pLen = ptrI;

  return(ptr);
error:
  return(NULL);
}
