/*JS*********************************************************************
*
*    Program : FLIB2C
*    Language: ANSI-C
*    Author  : Joerg Schoen
*    Purpose : Creates Header files for linkage to fortran libraries.
*
*************************************************************************/

#ifndef lint
static const char rcsid[] = "$Id$";
#endif

/*JS*********************************************************************/
/*********     INCLUDES 					*********/
/************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

/* For function 'mkdir'  */
#include <sys/types.h>
#include <sys/stat.h>

/*JS*********************************************************************/
/*********     DEFINES						*********/
/************************************************************************/
#define VERSION "1.4 "__DATE__

#define MAXLINE 1000 /*  Maximum length of line   */

/*   If C to FORTRAN link routines are needed,	 */
/*    then C link routines get this praefix	 */
#define LINKPRAEFIX "__"

/*  Pattern for symbol that surrounds header file.  */
#define DEFINE_SYMBOL    "__%s_h__"

/*   Routine to drop intermediate spaces    */
#define DROPSPACES()    while( isspace(*(unsigned char *)LinePos)) LinePos++

/*  For printing the line and the position the error occured.	*/
#define PRERRORLINE(Line,Pos) \
  printf(">%.*s\n>%*s%s\n",Pos,Line,Pos,"",Line + (Pos))

/*  C-names can contain a '_' too    */
#define MYISALPHA(c)    (isalpha(c) || (c) == '_')
#define MYISALNUM(c)    (isalnum(c) || (c) == '_')

/*   Structure to store the declaration of one variable    */
struct Variable {
  struct Variable *Next;
  int		   Type; /*  See defines below	  */
  union {
    char	     Name[1];
    struct Variable *Tag; /*  If root node, points to child nodes   */
  } C;
};

/*   The lower bits contain number of '*' after basic type    */
#define TYPE_VOID     0x100
#define TYPE_SHORT    0x200
#define TYPE_INT      0x300
#define TYPE_LONG     0x400
#define TYPE_FLOAT    0x500
#define TYPE_DOUBLE   0x600
#define TYPE_CHAR     0x700
#define TYPE_DCMPLX   0x800
#define TYPE_CMPLX    0x900

#define NUM_TYPES     9

#define MASK_TYPE     0xff00
#define MASK_LENGTH   0x00ff
#define TYPE_SHIFT    8

#define TYPE_FUNCTION 0x10000

/*	Table for the implicit types	  */
int ImplicitTypes['z' - 'a' + 1];

/*JS*********************************************************************/
/*********     PROTOTYPES					*********/
/************************************************************************/
extern int		fillbuf(void);
extern void		preprocessor(void);
extern struct Variable *getvariable(int nr);
extern void		printprototype(FILE *outfp,struct Variable *def,
				       int mode);
extern void		printdefine(FILE *outfp,struct Variable *def,
				    int mode);

extern const char      *strcompare(const char *input,const char *string);
extern short		stripcomments(char *,char *,short);

/*JS*********************************************************************/
/*********     GLOBAL VARIABLES 				*********/
/************************************************************************/
char *TypeNames[] = {
  "void", "short", "int", "long", "float", "double", "char", "dcmplx",
  "cmplx", NULL
};

char *TempNameChars[] = {
  "?",     "h",     "i",   "l",    "f",     "d",      "s",    "D",
  "C",     NULL
};

#ifdef DEBUG
/*  For debugging   */
int   Debug;
#endif

#if defined(_AIX) || defined(__hpux)
/*   IBM RS6000 and HP-UX: No postfix for FORTRAN names  */
char *funcpostfix = "";
#else
/*   Other machines: '_' after every name    */
char *funcpostfix = "_";
#endif

/*    Stuff for input file  */
FILE *FileP;
int   LineNr,CommentState,FileState;
char *LinePos,LineBuffer[MAXLINE];

char  NameBuffer[MAXLINE];
char  FileNameBuffer[L_tmpnam + MAXLINE];
char  CommandBuffer[L_tmpnam + MAXLINE];

/*JS*********************************************************************
*   MAIN
*************************************************************************/

int main(int argc,char **argv)

/************************************************************************/
{
  char *string,*string2,*inputfile;
  char *linklibrary;
  int	defaultfloat,mode;
  int	i,j,maxtypes[NUM_TYPES];
  struct Variable *first,*last,*var,*curr,*var2;
  FILE *outfp;

  /*	No appendix to function names	 */
  defaultfloat = TYPE_FLOAT;
  first = last = NULL;
  linklibrary = NULL;
  mode = 1;

  /*	Drop program name    */
  argv++; argc--;

  /*	Parse arguments      */
  for(i = 0 , j = 0; i < argc ; i++) {
    if(*argv[i] == '-') {
      switch(argv[i][1]) {
      default:
	printf("ERROR: Unknown option '-%c' -- Break.\n",argv[i][1]);
      case 'h':
	goto help;
#ifdef DEBUG
      case 'D':
	/*   Get debugging mode   */
	Debug = strtol(argv[i] + 2,NULL,0);
	break;
#endif
      case 'l':
	if(argv[i][2] == '\0') {
	  /*  Name is next argument    */
	  argv[i] = NULL;

	  if(++i >= argc) {
	    printf("ERROR: No directory name after '-l' -- Break.\n");
	    goto help;
	  }

	  linklibrary = argv[i];
	} else
	  linklibrary = argv[i] + 2;
	break;
      case 'p':
	funcpostfix = argv[i] + 2;
	break;
      case 'd':
	defaultfloat = TYPE_DOUBLE;
	break;
      case 'f':
	defaultfloat = TYPE_FLOAT;
	break;
      case 'u':
	mode = 0;
	break;
      }

      /*   Drop this option	*/
      argv[i] = NULL;
    } else
      j++;
  }

  if(j != 2) goto help;

  for(j = 0 ; argc-- > 0 ; argv++) {
    if(*argv) {
      if(j == 0) {
	/*   Save this name for later usage	*/
	inputfile = *argv;

	/*   Open definition file     */
	if( (FileP = fopen(inputfile,"r")) == NULL) {
	  printf("ERROR: Could not open \"%s\" -- Break.\n",
		 inputfile);
	  goto help;
	}
      } else {
	/*   Open output file	      */
	if( (outfp = fopen(*argv,"w")) == NULL) {
	  printf("ERROR: Could not open \"%s\" -- Break.\n",*argv);
	  goto help;
	}
      }
      j++;
    }
  }

  /*   The names of the temporary variables are named	*/
  /*	according to the input file name.		*/
  /*   Create new variable name     */
  if((string = strrchr(inputfile,'/')) == NULL) string = inputfile;
  else string++;

  for(string2 = NameBuffer ; MYISALPHA(*(unsigned char *)string) ; )
    *string2++ = *string++;
  *string2 = '\0';

  if(mode) {
    char c;

    /*	Set up table for implicits correctly:	 */
    /*	 a-h,o-z is float/double and i-n is int  */
    for(c = 'a' ; c <= 'h' ; c++)
      ImplicitTypes[c - 'a'] = defaultfloat;

    for(c = 'o' ; c <= 'z' ; c++)
      ImplicitTypes[c - 'a'] = defaultfloat;

    for(c = 'i' ; c <= 'n' ; c++)
      ImplicitTypes[c - 'a'] = TYPE_INT;
  } else {
    char c;

    /*  Set up all character to none  */
    for(c = 'a' ; c <= 'z' ; c++)
      ImplicitTypes[c - 'a'] = 0;
  }

  /* ***********      Write Header information	 ******** */
  fprintf(outfp,"#ifndef "DEFINE_SYMBOL"\n\
/* ******************************************************************** */\n\
/* *****  This file is generated by FLIB2C  Joerg Schoen 1993-95  ***** */\n\
/* *****   from %-20s				   ***** */\n\
/* *****   %s	***** */\n\
/* ******************************************************************** */\n\n\
#define " DEFINE_SYMBOL " 1\n\
\n\
#ifdef __cplusplus  /*  C++ compatibility  */\n\
extern \"C\" {\n\
#endif\n\n",NameBuffer,inputfile,linklibrary ?
	  "You need the appropriate C library to use this file!" :
	  "This file is stand alone.                           ",NameBuffer);

  /* ****************************************************************** */
  /* Now read input file. Set up default variables with implicit rules	*/
  /*  and count maximal needed temporary static variables		*/
  for(i = 0 ; i < NUM_TYPES ; i++) maxtypes[i] = 0;

  for( fillbuf() ; FileState != EOF ; ) {
    int counttypes[NUM_TYPES];

#ifdef DEBUG
    if(Debug & 2)
      printf("%d <%s>\n",LineNr,LinePos);
#endif

    /*	 Preprocessor command?	*/
    if(*LinePos == '#') {
      preprocessor();
      fillbuf();
      continue;
    } else if(*LinePos == '!') {
      /*    Write line as it is to outputfile	 */
      fprintf(outfp,"%s",LinePos + 1);
      fillbuf();
      continue;
    }

    /*	 Ignore "extern" at start of line     */
    if((string = (char *)strcompare(LinePos,"extern")) &&
       isspace(*(unsigned char *)string)) {
      LinePos = string;
      DROPSPACES();
    }

    /*	 Now get function definition	 */
    if((var = getvariable(0)) == NULL) {
      printf("Break.\n");
      goto ende;
    }
    DROPSPACES();

    if(*LinePos != '(') {
      printf("ERROR Line %d: No function definition -- Line ignored.\n",
	     LineNr);
      PRERRORLINE(LineBuffer,LinePos - LineBuffer);
      (void) fillbuf();
      continue;
    }

    /*	 Get a main node for this function definition	  */
    if((curr = (struct Variable *)malloc(sizeof(*curr))) == NULL)
      goto nomem;

    /*	 First node is routine name	 */
    curr->C.Tag = var;

    /*	Check if routine declared twice   */
    for(var2 = first ; var2 ; var2 = var2->Next)
      if( !strcmp(var2->C.Tag->C.Name,var->C.Name) ) {
	printf("ERROR Line %d: Function \"%s\" declared twice -- Break.\n",
	       LineNr,var->C.Name);
	PRERRORLINE(LineBuffer,LinePos - LineBuffer);
	goto ende;
      }

    /*	Now read the parameters    */
    i = 0; /*  preset counter	*/
    do {
      /*   Drop '(' resp ','    */
      LinePos++;
      DROPSPACES();

      /*   Empty variable list?   */
      if((string = (char *)strcompare(LinePos,"void"))) {
	/*  Check if that's all  */
	while(isspace(*(unsigned char *)string)) string++;

	if(*string == ')') {
	  LinePos = string;
	  break;
	}
      } else if(*LinePos == ')') break;

      if( (var->Next = getvariable(i + 1)) == NULL)
	goto ende;

      /*  Go to next variable	 */
      var = var->Next;
      var->Next = NULL; /*  mark end correctly	  */
      i++; /*  increase counter   */

      /*   Cannot handle 'char' variables, only 'char *'   */
      if((var->Type & (MASK_TYPE + MASK_LENGTH)) == TYPE_CHAR) {
	printf("ERROR Line %d: Cannot handle 'char' variables correctly\
 -- Break.\n",LineNr);
	PRERRORLINE(LineBuffer,LinePos - LineBuffer);
	goto ende;
      }

      /*   Check, if variable declared twice	*/
      for(var2 = curr->C.Tag->Next ; var2->Next ; var2 = var2->Next)
	if( !strcmp(var2->C.Name,var->C.Name) ) {
	  printf("ERROR Line %d: Variable \"%s\" declared twice in \
function definition\n -- Break.\n",LineNr,var->C.Name);
	  PRERRORLINE(LineBuffer,LinePos - LineBuffer);
	  goto ende;
	}

      DROPSPACES();
    } while(*LinePos == ',');

    /*	  Mark end of list with NULL	*/
    var->Next = NULL;

    /*	  Save number of Variables in Type field   */
    curr->Type = i;

    /*	 Check correct end of variable definition    */
    if(*LinePos++ != ')') {
      printf("ERROR Line %d: No closing bracket in function definition!\
 -- Break.\n",LineNr);
      PRERRORLINE(LineBuffer,LinePos - LineBuffer);
      goto ende;
    }

    /*	 Drop ';' at end of line if existant   */
    DROPSPACES();
    if(*LinePos == ';') {
      LinePos++;
      DROPSPACES();
    }

    /****************************************************************/
    /*	 Set up correct Type of function if not explicitly given.   */
    /****************************************************************/
    var = curr->C.Tag;
    /*	  Definitions with no type are assumed to be 'void'  */
    if(var->Type == 0) var->Type = TYPE_VOID;

    for(i = 0 ; i < NUM_TYPES ; i++) counttypes[i] = 0;
    for(var = var->Next ; var ; var = var->Next) {
      /*   Check if no type is explicitly set	 */
      if( (var->Type & MASK_TYPE) == 0) {
	/*   Decide from first character of variable or function name  */
	if((i = ImplicitTypes[tolower(*(unsigned char *)var->C.Name) - 'a'])
	   == 0) {
	  printf("ERROR Line %d: No implicit rule for variable \"%s\"\
 -- Break.\n",LineNr,var->C.Name);
	  PRERRORLINE(LineBuffer,LinePos - LineBuffer);
	  goto ende;
	}
	/*   Set up this type, conserve number of '*'   */
	var->Type += i;
      }

#ifdef DEBUG
      if(Debug & 8) {
	printf("<%x>%s",var->Type,var->C.Name);
	if(var->Next) putchar(',');
      }
#endif

      /*   Count variables   */
      if(!(var->Type & TYPE_FUNCTION) &&
	 ((var->Type & MASK_LENGTH) == 0 ||
	  (var->Type & (MASK_TYPE + MASK_LENGTH)) == (TYPE_CHAR + 1)))
	counttypes[((var->Type & MASK_TYPE) >> TYPE_SHIFT) - 1]++;
    }

    /*	 Check if no temporary assignments needed   */
    /*	  and set number of parameters negative     */
    {
      int flag;

      for(flag = i = 0 ; i < NUM_TYPES ; i++) flag |= counttypes[i];

      if(!flag) curr->Type = -curr->Type;
    }

#ifdef DEBUG
    if(Debug & 8)
      printf(");\n");
#endif

    /*	  Compare to maximal values	  */
    for(i = 0 ; i < NUM_TYPES ; i++)
      if(counttypes[i] > maxtypes[i]) maxtypes[i] = counttypes[i];

    /*	 Mark if function needs strlen function    */
    if(counttypes[(TYPE_CHAR >> TYPE_SHIFT) - 1])
      curr->Type = -curr->Type;

    /*	  Hang function definition in global list    */
    if(last) {
      /*   Hang at end of list	 */
      last->Next = curr;
      last = curr;
    } else {
      /*   First element of list   */
      first = last = curr;
    }
    /*	 Mark end right   */
    curr->Next = NULL;

#ifdef DEBUG
    /*	 Print out current function definition	  */
    if(Debug & 4) {
      var = curr->C.Tag;

      printf("FUNCTION: <%#x>%s(",var->Type,var->C.Name);

      for(var = var->Next ; var ; var = var->Next) {
	printf("<%#x>%s",var->Type,var->C.Name);

	if(var->Next) putchar(',');
      }
      printf(");\n");
    }
#endif

    /*	 Check if end of line	 */
    if(!*LinePos) {
      (void)fillbuf();
    }
  }
  /* ******************************************************************** */

  /*   Mark end of global list with NULL    */
  if(last)
    last->Next = NULL;
  else {
    printf("ERROR: File contains no function declarations -- Break.\n");
    goto ende;
  }

  /* ********************************************************** */
  /* ******    Now create header file	      ***************** */
  /* ********************************************************** */
  /*	Print out definitions for temporary static variables	*/
  if(maxtypes[(TYPE_CHAR >> TYPE_SHIFT) - 1] && !linklibrary)
    fprintf(outfp,"\n\
/* ********	 We need the strlen function   ******** */\n\
#include <string.h>\n");

#ifdef OLDOLD
  /*  This statement is wrong: Checks only if 'dcmplx' used, will fail
   *   if 'dcmplx *' is used. In that case we need also cmplx type
   */
  if((maxtypes[(TYPE_CMPLX >> TYPE_SHIFT) - 1] ||
      maxtypes[(TYPE_DCMPLX >> TYPE_SHIFT) - 1]) && !linklibrary)
    fprintf(outfp,"\n\
/* ********	   We need complex types       ******** */\n\
typedef struct {\n\
  float re,im;\n\
} cmplx;\n\
typedef struct {\n\
  double re,im;\n\
} dcmplx;\n");
#endif

  if(!linklibrary) {
    int k;

    fprintf(outfp,"\n\
/* **********  Temporary variables for fortran linkage	 ********** */");

    for(k = 0 ; k < NUM_TYPES ; k++)
      if(maxtypes[k]) {
#if 1
	if(k == ((TYPE_CHAR >> TYPE_SHIFT) - 1))
	  fprintf(outfp,"\nstatic %s *__%s%s[%d];\n",TypeNames[k],
		  NameBuffer,TempNameChars[k],maxtypes[k]);
	else
	  fprintf(outfp,"\nstatic %s __%s%s[%d];\n",TypeNames[k],
		  NameBuffer,TempNameChars[k],maxtypes[k]);
#else
	for(i = 1 ; i <= maxtypes[k] ; i++) {
	  if(i % 5 == 1) fprintf(outfp,"\nstatic %s  ",TypeNames[k]);

	  /*  In case of char use character pointers  */
	  if(k == ((TYPE_CHAR >> TYPE_SHIFT) - 1)) fprintf(outfp,"*");

	  fprintf(outfp,"__%s%s%d%c",NameBuffer,TempNameChars[k],i,
		  i % 5 == 0 || i == maxtypes[k] ? ';' : ',');
	}

	putc('\n',outfp);
#endif
      }
  }

  /* ******************************************* */
  /*   Write out Prototypes for the function	 */
  fprintf(outfp,"\n\
/* *******************	   PROTOTYPES	   ************************ */\n");

  for(curr = first ; curr ; curr = curr->Next) {
    printprototype(outfp,curr,linklibrary ? 1 : 0);
  }

  if(!linklibrary) {
    /* ******************************************* */
    /*	 Print defines for fortran linkage	   */
    fprintf(outfp,"\n\
/* ********  Now defines for linkage to fortran library    ************ */\n");

    for(curr = first ; curr ; curr = curr->Next) {
      printdefine(outfp,curr,0);
    }
  } else {
    fprintf(outfp,"\n\n\
/* ********  Now defines to intermediate C library	   ************ */\n");

    for(curr = first ; curr ; curr = curr->Next) {
      /*  Only if routine has any parameters	*/
      if(curr->Type)
	fprintf(outfp,"#define %-10s "LINKPRAEFIX"%s\n",curr->C.Tag->C.Name,
		curr->C.Tag->C.Name);
      else if(*funcpostfix)
	fprintf(outfp,"#define %-10s %s%s\n",curr->C.Tag->C.Name,
		curr->C.Tag->C.Name,funcpostfix);
    }
  }

  fprintf(outfp,"\n\
#ifdef __cplusplus\n\
}\n\
#endif\n\n\
#endif /*   End of machine generated file    */\n");

  /*	Now create whole directory with link routines	 */
  if(linklibrary) {
    /*	 Get temporary directory    */
    sprintf(FileNameBuffer,"%s_tmp_dir",linklibrary);
    j = strlen(FileNameBuffer);

    /*	 Create directory    */
    printf("  Creating directory %s...\n",FileNameBuffer);
    (void) mkdir(FileNameBuffer,0755);

    /*	 Append '/'    */
    FileNameBuffer[j++] = '/';
    FileNameBuffer[j] = '\0';

    /*	 Create temporary Makefile    */
    strcpy(FileNameBuffer + j,"Makefile");
#ifdef DEBUG
    if(Debug)
      printf("Creating temporary Makefile.\n");
#endif
    if( (outfp = fopen(FileNameBuffer,"w")) == NULL) {
      printf("ERROR: Couldn't open \"%s\" -- Break.\n",FileNameBuffer);
      goto ende;
    }

    fprintf(outfp,"\
#\n\
#   This file and all sources created by FLIB2C   Joerg Schoen 1993-95.\n\
#    Simply enter \"make\" to get a linking library.\n\
#\n\
CFLAGS = -O\n\
LIBRARY = %s\n\
SRCS = ",linklibrary);

    for(curr = first , i = 0 ; curr ; curr = curr->Next) {
      /*  Only if routine has any arguments   */
      if(curr->Type) {
	fprintf(outfp,"%s.c ",curr->C.Tag->C.Name);

	if( (i++ % 3) == 2 && curr->Next)
	  fprintf(outfp,"\\\n\t");
      }
    }

    fprintf(outfp,"\n\n\
OBJS = $(SRCS:.c=.o)\n\n\
$(LIBRARY) : $(OBJS)\n\
\tar r $(LIBRARY) $(OBJS)\n\
\trm $(OBJS)\n"
#ifndef DEBUG
"\trm $(SRCS)\n"
#endif
"\n.c.o:   ; $(CC) $(CFLAGS) -c -o $@ $<\n");

    fclose(outfp);

    /*	 Now create all the files     */
    for(curr = first ; curr ; curr = curr->Next) {
      /*   Only if routine has any arguments	*/
      if(!curr->Type)
	continue;

      /* ******   Create files	  ****** */
      /*   First build up filename	*/
      sprintf(FileNameBuffer + j,"%s.c",curr->C.Tag->C.Name);
#ifdef DEBUG
      if(Debug)
	printf("Creating \"%s\"\n",FileNameBuffer);
#endif
      if( (outfp = fopen(FileNameBuffer,"w")) == NULL) {
	printf("ERROR: Couldn't open \"%s\" -- Break.\n",
		FileNameBuffer);
	goto ende;
      }

      fprintf(outfp,"\
/* ****************************************************************** */\n\
/* *****   This file created by FLIB2C  Joerg Schoen 1993-95.   ***** */\n\
/* *****    Linkage routine to FORTRAN. 			***** */\n\
/* ****************************************************************** */\n\n");

      if(curr->Type < 0)
	fprintf(outfp,"\
/* ********	 We need the strlen function   ******** */\n\
#include <string.h>\n\n");
      printprototype(outfp,curr,0);
      putc('\n',outfp);
      printdefine(outfp,curr,1);

      fclose(outfp);
    }

    /*	 Now tell user what to do  */
    FileNameBuffer[j] = '\0';
    printf("Done. Simply enter \"cd %s ; make\" to create the link library.\n",
	   FileNameBuffer);
  }

ende:
  fclose(FileP);

  return(0);

nomem:
  printf("ERROR: No memory available!\n");

  if(FileP) fclose(FileP);

  return(20);

help:
  if(FileP) fclose(FileP);

  printf("\
FLIB2C V" VERSION "                                    Joerg Schoen 1993-95\n\
Usage: flib2c [<options>] <deffile> <outfile>\n\
  Creates a C-Header file for linkage to fortran libraries from\n\
  the file <deffile> and writes to file <outfile>. Options are:\n\
    -h Shows this information.\n\
    -p<string> Append this string to every called fortran routine.\n\
	       Default: \"%s\".\n\
    -l<libname> Create linking library libname (instead of only header file).\n\
    -d Default floating point type is 'double' instead of 'float'.\n\
    -f Default floating point type is 'float'.\n\
    -u Set all implicits to none.\n",funcpostfix);
  return(10);
}

/*JS*********************************************************************
*   FILLBUF Fills up line buffer until EOF. In that case 1 is returned.
*************************************************************************/

int fillbuf(void)

/************************************************************************/
{
  /*	Read as long as comments or empty lines    */
  do {
    if( fgets(LineBuffer,MAXLINE,FileP) == NULL) {
      FileState = EOF;
      return(1);
    }

    LinePos = LineBuffer;
    LineNr++;

    /*	 Lines starting with a '!' are not changed     */
    if(*LinePos == '!')
      break;

    CommentState = stripcomments(LineBuffer,LineBuffer,CommentState);

    /*	 Drop spaces at beginning of line    */
    DROPSPACES();
  } while(*LinePos == '\0');

  return(0);
}

/*JS*********************************************************************
*   PREPROCESSOR
*************************************************************************/

void preprocessor(void)

/************************************************************************/
{
  char *string,c,cend;
  int i;

  if((string = (char *)strcompare(LinePos + 1,"implicit"))) {
    LinePos = string;
    DROPSPACES();

    if(strcompare(LinePos,"none")) {
      /*    Clear all implicit definitions   */
      for(c = 'a' ; c <= 'z' ; c++)
	ImplicitTypes[c - 'a'] = 0;
    } else {
      /*   Read rules  */
      do {
	/*   First try to find a Type	 */
	for(i = 0 ; TypeNames[i] ; i++)
	  if((string = (char *)strcompare(LinePos,TypeNames[i]))) break;

	if(string == NULL || isalnum(*(unsigned char *)string)) {
	  printf("ERROR Line %d: Unknown variable type in\
 \"#define implicit\"\n -- Remainder of Line ignored\n",LineNr);
	  PRERRORLINE(LineBuffer,LinePos - LineBuffer);
	  goto ende;
	}

	/*  Read type  */
	i = (i + 1) << TYPE_SHIFT;
	LinePos = string;

	/*   Count number of stars   */
	DROPSPACES();
	while(*LinePos == '*') {
	  i++;
	  LinePos++;
	  DROPSPACES();
	}

	if(*LinePos != '(') {
	  printf("ERROR Line %d: Syntax error, no opening bracket --\n\
 Remainder of line ignored.\n",LineNr);
	  PRERRORLINE(LineBuffer,LinePos - LineBuffer);
	  goto ende;
	}

	do {
	  LinePos++;
	  DROPSPACES();

	  /*  Get  first character  */
	  c = tolower(*(unsigned char *)LinePos);

	  if(!isalpha((unsigned char)c) ) {
	    printf("ERROR Line %d: Not a letter in rule\
 -- Remainder of line ignored.\n",LineNr);
	    PRERRORLINE(LineBuffer,LinePos - LineBuffer);
	    goto ende;
	  }
	  LinePos++;
	  DROPSPACES();

	  if(*LinePos == '-') {
	    LinePos++;
	    DROPSPACES();
	    cend = tolower(*(unsigned char *)LinePos);

	    if(!isalpha((unsigned char)cend) ) {
	      printf("ERROR Line %d: Not a letter in rule\
 -- Remainder of line ignored.\n",LineNr);
	      PRERRORLINE(LineBuffer,LinePos - LineBuffer);
	      goto ende;
	    }

	    if(cend < c) {
	      char temp;

	      /*  Exchange  */
	      temp = c; c = cend; cend = temp;
	    }
	    LinePos++;
	    DROPSPACES();
	  } else {
	    cend = c;
	  }

#ifdef DEBUG
	  if(Debug & 16) {
	    printf("   Setting '%c' to '%c' to %s",c,cend,
		   TypeNames[(i>>TYPE_SHIFT) - 1]);

	    for(j = i & MASK_LENGTH ; j-- > 0 ; )
	      putchar('*');
	    putchar('\n');
	  }
#endif
	  /*   Set these types	 */
	  for(; c <= cend ; c++)
	    ImplicitTypes[c - 'a'] = i;

	} while(*LinePos == ',');

	if(*LinePos != ')') {
	  printf("ERROR Line %d: Syntax error, no closing bracket --\
 Remainder of line ignored.\n",LineNr);
	  PRERRORLINE(LineBuffer,LinePos - LineBuffer);
	  goto ende;
	}
	LinePos++;
	DROPSPACES();
      } while(*LinePos++ == ',');

      if(LinePos[-1]) {
	printf("WARNING: Remainder of line ignored!\n>>>%s\n",
	       LinePos - 1);
      }
    }

#ifdef DEBUG
    if(Debug & 16) {
      printf("------ Table of implicit Types: ------------\n");

      for(c = 'a' , i = 0 ; c <= 'z' ; c++ , i++) {
	j = ImplicitTypes[c - 'a']>>TYPE_SHIFT;

	printf("'%c':%s",c,j ? TypeNames[j - 1] : "???" );

	for(j = ImplicitTypes[c - 'a'] & MASK_LENGTH ; j-- > 0 ; )
	  putchar('*');

	if( (i % 5) == 4) putchar('\n');
	else putchar(' ');
      }
      putchar('\n');
    }
#endif
  } else {
    printf("ERROR Line %d: Unknown Preprocessor command -- Ignored\n",LineNr);
    PRERRORLINE(LineBuffer,LinePos - LineBuffer);
  }

ende:
  return;
}

/*JS*********************************************************************
*   GETVARIABLE Reads in variable definition in the form:
*    "[type] {'*'} [name|'(*'name')']"
*************************************************************************/

struct Variable *getvariable(int nr)

/************************************************************************/
{
  struct Variable *varptr;
  char	 *string,*string2;
  int i,j;

  /*	First drop some spaces	  */
  DROPSPACES();

  if(*LinePos == '\0' && fillbuf()) goto error;

  /*   First try to find a Type    */
  for(i = 0 , string2 = NULL ; TypeNames[i] ; i++)
    if((string2 = (char *)strcompare(LinePos,TypeNames[i]))) break;

  if(string2 == NULL || isalnum(*(unsigned char *)string2) ) {
    /*	 No type or name started with 'int' etc. Set default 0.   */
    i = 0;
  } else {
    LinePos = string2;
    i = (i + 1) << TYPE_SHIFT;
  }

  /*   Note: a definition like "*var" is allowed too.  */
  /*	In that case the type is chosen in the main    */
  /*	program from the first letter of the variable  */
  /*	(FORTRAN convention).                          */

  /*   Count number of following "*" characters   */
  for(j = 0 ; ; j++) {
    DROPSPACES();

    if(*LinePos != '*')
      break;

    LinePos++;
  }

  /*	 User wrote more than 255 stars. Stupid user!!!   */
  if(j >= MASK_LENGTH) {
    printf("ERROR Line %d: To many '*' in type! -- Break\n",LineNr);
    PRERRORLINE(LineBuffer,LinePos - LineBuffer);
    goto error;
  }

  /*   Now set up full type    */
  i += j;

  /*   Count number of characters in variable name   */
  DROPSPACES();

  /*   Variable is function?  */
  if(*LinePos == '(') {
    LinePos++;
    DROPSPACES();

    /*	Allow user to suround name with brackets or not  */
    if(*LinePos == '*') {
      LinePos++;
      DROPSPACES();
    }

    i |= TYPE_FUNCTION;
  }

  /*   Is there any character of the name?	     */
  if(!MYISALPHA(*(unsigned char *)LinePos) ) {
    printf("ERROR Line %d: No variable name given!\n",LineNr);
    PRERRORLINE(LineBuffer,LinePos - LineBuffer);
    goto error;
  }

  for(j = 1 , string = LinePos + 1 ; MYISALNUM(*(unsigned char *)string) ;
      string++)
    j++;

  if((varptr = (struct Variable *)malloc(sizeof(*varptr) + j)) == NULL)
    goto error;

  varptr->Type = i;
  strncpy(varptr->C.Name,LinePos,j);
  varptr->C.Name[j] = '\0';

  LinePos = string;

  /*  Check closing bracket in case of function call  */
  if(i & TYPE_FUNCTION) {
    DROPSPACES();
    if(*LinePos != ')') {
      printf("ERROR Line %d: No closing bracket in function variable!\n",
	     LineNr);
      PRERRORLINE(LineBuffer,LinePos - LineBuffer);
      goto error;
    }
    LinePos++;
  }

#ifdef DEBUG
  if(Debug & 2)
    printf("GETVARIABLE: Type: %#x Name: <%s>\n",i,varptr->C.Name);
#endif

  return(varptr);
 error:
  return(NULL);
}

/*JS*********************************************************************
*   PRINTPROTOTYPE Prints out a C Prototype for the function definition
*    def. If mode == 0 prints a Prototype for the fortran routine,
*    otherwise for a C linking routine.
*************************************************************************/

void printprototype(FILE *outfp,struct Variable *def,int mode)

/************************************************************************/
{
  int stringcount,i,j;
  struct Variable *var;

  fprintf(outfp,"extern ");

  for(stringcount = 0 , var = def->C.Tag , j = 0 ;
      var ; var = var->Next , j++) {
    fprintf(outfp,"%s ",
	    TypeNames[((var->Type & MASK_TYPE) >> TYPE_SHIFT) - 1]);

    /*	  Check if pointer type    */
    i = var->Type & MASK_LENGTH;

    if(!mode && !(var->Type & TYPE_FUNCTION) && i == 0 &&
       var->Type != TYPE_VOID && j != 0) {
      /*   In reality parameter is pointer type   */
      putc('*',outfp);
    } else {
      /*   Print actual number of '*' (including 0)   */
      for( ; i-- > 0 ; )
	putc('*',outfp);
    }

    /*	  Count number of string variables    */
    if( var->Type == (TYPE_CHAR + 1) ) stringcount++;

    if(mode && j == 0) fprintf(outfp,LINKPRAEFIX);

    /*	  Print '(*' for function variable  */
    if(j != 0 && (var->Type & TYPE_FUNCTION)) fprintf(outfp,"(*");

    /*	  Now print variable name   */
    fprintf(outfp,"%s",var->C.Name);

    if(j != 0 && (var->Type & TYPE_FUNCTION)) fprintf(outfp,")()");

    /*	  Delimiter   */
    if(j == 0) {
      /*   Print postfix  */
      if( !mode)
	fprintf(outfp,"%s",funcpostfix);

      /*   And '('    */
      putc('(',outfp);
    } else if(var->Next) {
      putc(',',outfp);

      if(j % 4 == 3)
	fprintf(outfp,"\n\t");
    }
  }

  if(mode == 0) {
    /*	 For every string variable has to be an invisible argument  */
    /*	  at the end of the function call -- the string length	    */
    for( ; stringcount-- > 0 ; ) {
      fprintf(outfp,",int");
    }
  }

  /*   Function with no arguments?   */
  if(j == 1)
    fprintf(outfp,"void"); /*  Argument "void"   */

  /*   Print end of declaration      */
  fprintf(outfp,");\n");

  return;
}

/*JS*********************************************************************
*   PRINTDEFINE
*************************************************************************/

void printdefine(FILE *outfp,struct Variable *def,int mode)

/************************************************************************/
{
  int count[NUM_TYPES],i,j;
  struct Variable *var;
  char *string;

  if(!mode)
    fprintf(outfp,"#define ");

  /*   Print "function(parameter,parameter,...)"    */
  for(var = def->C.Tag , j = 0 ; var ; var = var->Next , j++) {
    if(mode) {
      fprintf(outfp,"%s ",
	      TypeNames[((var->Type & MASK_TYPE) >> TYPE_SHIFT) - 1]);

      /*    Check if pointer type    */
      for(i = var->Type & MASK_LENGTH ; i-- > 0 ; )
	putc('*',outfp);

      if(j == 0)
	fprintf(outfp,LINKPRAEFIX);
    }

    /*	Print variable / function name	  */
    fprintf(outfp,"%s",var->C.Name);

    if(var == def->C.Tag) {
      putc('(',outfp);
    } else if(var->Next) {
      putc(',',outfp);

      /*    At most 9 parameters in one line   */
      if(j % 9 == 8) {
	if(!mode)
	  fprintf(outfp," \\\n\t");
	else
	  putc('\n',outfp);
      }
    }
  }

  /*	Print call     */
  if(j == 1) {
    if(mode) {
      printf("INTERNAL ERROR: No define for \"%s\" function!\n",
	     def->C.Tag->C.Name);
      return;
    }
    fprintf(outfp,") %s%s()\n\n",def->C.Tag->C.Name,funcpostfix);
    return;
  }

  /*   Actual definition start in separate line   */
  if(!mode) {
    fprintf(outfp,") \\\n\t(");

    /*	 Print out assignments to temporary variables	 */
    for(i = 0 ; i < NUM_TYPES ; i++) count[i] = 1;
    for(var = def->C.Tag->Next , j = 0 ; var ; var = var->Next) {
      /*  If variable is no pointer, make    */
      /*   assignment to temporary variable  */
      if(!(var->Type & TYPE_FUNCTION) && ((var->Type & MASK_LENGTH) == 0 ||
					  var->Type == (TYPE_CHAR + 1))) {

	string = TempNameChars[((var->Type & MASK_TYPE) >> TYPE_SHIFT) - 1];
	i = count[(var->Type & MASK_TYPE) >> TYPE_SHIFT]++;

	/*   Print assignment to temporary variable	*/
#if 1
	fprintf(outfp,"__%s%s[%d] = (%s), ",NameBuffer,string,i - 1,
		var->C.Name);
#else
	fprintf(outfp,"__%s%s%d = (%s), ",NameBuffer,string,i,
		var->C.Name);
#endif

	/*   At most 4 assignments in one line    */
	if(j++ % 4 == 3)
	  fprintf(outfp,"\\\n\t");
      }
    }

    /*	 Now end line, function call is in one separate line	*/
    /*	 Check if not done before   */
    if(j % 4)
      fprintf(outfp,"\\\n\t");

    for(i = 0 ; i < NUM_TYPES ; i++) count[i] = 1;
    for(var = def->C.Tag , j = 0 ; var ; var = var->Next , j++) {
      if(!(var->Type & TYPE_FUNCTION) && var != def->C.Tag &&
	 ((var->Type & MASK_LENGTH) == 0 || var->Type == (TYPE_CHAR + 1))) {
	string = TempNameChars[((var->Type & MASK_TYPE) >> TYPE_SHIFT) - 1];
	i = count[(var->Type & MASK_TYPE) >> TYPE_SHIFT]++;

	if(var->Type != (TYPE_CHAR + 1)) putc('&',outfp);

	/*   Print address of current temporary variable    */
#if 1
	fprintf(outfp,"__%s%s[%d]",NameBuffer,string,i - 1);
#else
	fprintf(outfp,"__%s%s%d",NameBuffer,string,i);
#endif
      } else {
	/*   Print variable itself     */
	fprintf(outfp,"%s",var->C.Name);
      }

      if(var == def->C.Tag) {
	/*   Postfix and opening brace	  */
	fprintf(outfp,"%s(",funcpostfix);
      } else if(var->Next) {
	/*   Delimiter to next variable     */
	putc(',',outfp);

	/*   At most 5 parameters per line  */
	if(j % 5 == 4)
	  fprintf(outfp," \\\n\t");
      }
    }

    /*	 Any "char*" variables in function call?    */
    /*	 Then at the end the length of the string   */
    /*	  has to be given.			    */
    for(i = 1 ; i < count[TYPE_CHAR >> TYPE_SHIFT] ; i++ , j++) {
#if 1
      fprintf(outfp,",strlen(__%s%s[%d])",NameBuffer,
	      TempNameChars[(TYPE_CHAR >> TYPE_SHIFT) - 1],i - 1);
#else
      fprintf(outfp,",strlen(__%s%s%d)",NameBuffer,
	      TempNameChars[(TYPE_CHAR >> TYPE_SHIFT) - 1],i);
#endif

      /*   At most 5 parameters per line  */
      if(j % 5 == 4)
	fprintf(outfp," \\\n\t");
    }

    /*	 Print end of function definition     */
    fprintf(outfp,") )\n\n");
  } else {
    fprintf(outfp,")\n\n{\n\t");

    var = def->C.Tag;
    if(var->Type != TYPE_VOID)
      fprintf(outfp,"return(");

    for(j = 0 ; var ; var = var->Next , j++) {
      if(j == 0) {
	fprintf(outfp,"%s%s(",var->C.Name,funcpostfix);
	continue;
      } else {
	if( (var->Type & MASK_LENGTH) == 0 && !(var->Type & TYPE_FUNCTION))
	  putc('&',outfp);

	/*   Print variable itself     */
	fprintf(outfp,"%s",var->C.Name);
      }

      if(var->Next) {
	/*   Delimiter to next variable     */
	putc(',',outfp);

	/*   At most 5 parameters per line  */
	if(j % 8 == 7)
	  fprintf(outfp,"\n\t\t");
      }
    }

    for(var = def->C.Tag->Next ; var ; var = var->Next)
      if(var->Type == (TYPE_CHAR + 1))
	fprintf(outfp,",strlen(%s)",var->C.Name);

    putc(')',outfp);

    if(def->C.Tag->Type != TYPE_VOID)
      fprintf(outfp,")");

    fprintf(outfp,";\n}\n");
  }

  return;
}

/*JS*********************************************************************
*   STRCOMPARE Compares 'string' totally with the beginning of 'input'.
*    Returns NULL when the comparison fails, otherwise returns the first
*    position after the string in 'input'.
*************************************************************************/

const char *strcompare(const char *input,const char *string)

/************************************************************************/
{
  while(*string) {
    if(*string++ !=  *input++) return(NULL);
  }

  return(input);
}

/*JS*********************************************************************
*   STRIPCOMMENTS Drops C like comments from the line. Returns status
*    after line: != 0 means unfinished comment in this line
*************************************************************************/

short stripcomments(char * in,char * out,short status)

/************************************************************************/
{
  register char c;
  register char * lastnonspace = out - 1;

  while((c = *in++)) {
    /* The following condition leaves a '\' at the   */
    /*	end of the line (see below too).             */
    if(c == '\\' && !*in) break;

    /*	Nested comments are possible if in the following   */
    /*	 line "!status" is dropped.                        */
    if( !status && c == '/' && *in == '*') {
      /* Found start of comment  */
      *out++ = ' ';
      /*  The character sequence '/' '*' '/' is NOT a comment!   */
      in++;   status++;
    } else if( status && c == '*' && *in == '/') {
      /*  End of comment replace by spaces   */
      *out++ = c = ' ';
      in++;   status--;
    }

    if(status) {
      *out++ = ' '; /* Write spaces instead of comments */
    } else {
      if(!isspace((unsigned char)c) ) lastnonspace = out;
      *out++ = c; /*  write char  */
    }
  }

  /*  End line after last non- blank  */
  if(c == '\\') {
    /*	 Mark continued line (with '\' at the end).    */
    /*	First space, so '\' is not concatenated with something else  */
    *++lastnonspace = ' ';
    *++lastnonspace = '\\';
  }

  *++lastnonspace = '\0';

  return(status);
}
